package prototools

import (
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/reflect/protodesc"
	"google.golang.org/protobuf/reflect/protoreflect"
	"google.golang.org/protobuf/reflect/protoregistry"
	"google.golang.org/protobuf/types/descriptorpb"
	"google.golang.org/protobuf/types/dynamicpb"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/v2"
)

var (
	ErrSchema        = xerrors.NewSentinel("schema error")
	ErrNotMessage    = xerrors.NewSentinel("target is not a message")
	ErrNotFound      = xerrors.NewSentinel("message not found")
	ErrBadAnnotation = xerrors.NewSentinel("bad annotation")
)

func validateInterfaceMessageDescriptor(resolver *protoregistry.Files, name protoreflect.FullName) error {
	descriptor, err := resolver.FindDescriptorByName(name)
	if err != nil {
		return ErrNotFound.Wrap(xerrors.Errorf("Can not find name %q: %w", name, err))
	}
	md, ok := descriptor.(protoreflect.MessageDescriptor)
	if !ok {
		return ErrNotMessage.Wrap(xerrors.Errorf("invalid type. name: %q, type: %T", name, descriptor))
	}

	for i := 0; i < md.Fields().Len(); i++ {
		field := md.Fields().Get(i)

		fieldOptions := field.Options().(*descriptorpb.FieldOptions)
		if !proto.HasExtension(fieldOptions, taskletv2.E_TaskletField) {
			continue
		}

		var taskletFieldOptions *taskletv2.TaskletFieldOptions
		extension := proto.GetExtension(
			fieldOptions,
			taskletv2.E_TaskletField,
		)
		if v, ok := extension.(*taskletv2.TaskletFieldOptions); !ok {
			return xerrors.Errorf("Can not cast extension to TaskletFieldOptions. Field: %s", field.FullName())
		} else {
			taskletFieldOptions = v
		}
		if taskletFieldOptions.GetSecret() != nil {
			if field.Kind() != protoreflect.MessageKind {
				return ErrBadAnnotation.Wrap(
					xerrors.Errorf(
						"Wrong field kind for secret annotation. Field: %q, Kind: %v",
						field.FullName(),
						field.Kind().String(),
					),
				)
			}
			if field.IsList() || field.IsMap() {
				return ErrBadAnnotation.Wrap(
					xerrors.Errorf(
						"Repeated and map annotations not supported for secrets. Field: %q",
						field.FullName(),
					),
				)

			}
			if field.Message().FullName() != secretRefDescriptor.FullName() {
				return ErrBadAnnotation.Wrap(
					xerrors.Errorf(
						"Wrong field type for secret annotation. Field: %q, ExpectedType: %q, ActualType: %q",
						field.FullName(),
						secretRefDescriptor.FullName(),
						field.Message().FullName(),
					),
				)
			}
		}
	}
	return nil

}

func CheckInputOutputSchema(fds *descriptorpb.FileDescriptorSet, req *taskletv2.IOSimpleSchemaProto) error {

	resolver, err := protodesc.NewFiles(fds)

	if err != nil {
		return ErrSchema.Wrap(err)
	}

	for _, name := range []string{req.InputMessage, req.OutputMessage} {
		castedName := protoreflect.FullName(req.InputMessage)
		if !castedName.IsValid() {
			return ErrSchema.Wrap(xerrors.Errorf("Bad message name. Name: %q", name))
		}
		if err := validateInterfaceMessageDescriptor(resolver, castedName); err != nil {
			return ErrSchema.Wrap(err)
		}
	}
	return nil
}

func ParseMessage(fds *descriptorpb.FileDescriptorSet, inputMessageName string, input *taskletv2.ExecutionInput) (
	*dynamicpb.Message,
	error,
) {
	resolver, err := protodesc.NewFiles(fds)

	if err != nil {
		return nil, xerrors.Errorf("Failed to prepare schemas: %v", err)
	}

	inputMessage, err := MakeMessage(resolver, inputMessageName)
	if err != nil {
		return nil, err
	}

	if err := proto.Unmarshal(input.GetSerializedData(), inputMessage); err != nil {
		return nil, err
	}

	return inputMessage, nil

}

func MakeMessage(resolver protodesc.Resolver, name string) (*dynamicpb.Message, error) {
	descriptor, err := resolver.FindDescriptorByName(protoreflect.FullName(name))
	if err != nil {
		return nil, xerrors.Errorf("Failed to query input message schema: %v", err)
	}
	messageDescriptor, ok := descriptor.(protoreflect.MessageDescriptor)
	if !ok {
		return nil, xerrors.Errorf("Not a message descriptor: %q", name)
	}

	return dynamicpb.NewMessage(messageDescriptor), nil
}

func MakeIOMessages(fds *descriptorpb.FileDescriptorSet, ioSchema *taskletv2.IOSimpleSchemaProto) (
	*dynamicpb.Message,
	*dynamicpb.Message,
	error,
) {
	resolver, err := protodesc.NewFiles(fds)

	if err != nil {
		return nil, nil, xerrors.Errorf("Failed to prepare schemas: %v", err)
	}

	inputMessage, err := MakeMessage(resolver, ioSchema.GetInputMessage())
	if err != nil {
		return nil, nil, err
	}

	outputMessage, err := MakeMessage(resolver, ioSchema.GetOutputMessage())
	if err != nil {
		return nil, nil, err
	}
	return inputMessage, outputMessage, nil
}

var secretRefDescriptor protoreflect.MessageDescriptor

func init() {
	secretRefDescriptor = (&taskletv2.SecretRef{}).ProtoReflect().Descriptor()
}
