package requestctx

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

type ctxKey struct{}

var authKey ctxKey

var ErrNoAuth = xerrors.NewSentinel("missing auth subject")
var ErrNoUserAuth = xerrors.NewSentinel("user auth required")
var ErrNoSandboxAuth = xerrors.NewSentinel("sandbox task auth required")
var ErrNoExecutionAuth = xerrors.NewSentinel("tasklet execution auth required")

type SubjectKind int

const (
	InvalidSubject SubjectKind = iota
	UserSubject
	SandboxTaskSubject
	ExecutionSubject
)

type AuthSubject struct {
	kind        SubjectKind
	login       string
	sandboxTask sandbox.SandboxTaskID
	executionID consts.ExecutionID
}

func NewInvalid() AuthSubject {
	return AuthSubject{kind: InvalidSubject}
}

func NewUser(login string) AuthSubject {
	return AuthSubject{
		kind:  UserSubject,
		login: login,
	}
}

func NewSandboxTask(taskID sandbox.SandboxTaskID) AuthSubject {
	return AuthSubject{
		kind:        SandboxTaskSubject,
		sandboxTask: taskID,
	}
}

func NewExecutionID(executionID consts.ExecutionID) AuthSubject {
	return AuthSubject{
		kind:        ExecutionSubject,
		executionID: executionID,
	}
}

func (u *AuthSubject) GetKind() SubjectKind {
	return u.kind
}

type UserAuth struct {
	login string
}

func (u UserAuth) Login() string {
	return u.login
}

func (u *AuthSubject) RequireUserAuth() (UserAuth, error) {
	if u.kind != UserSubject {
		return UserAuth{}, ErrNoUserAuth
	}
	return UserAuth{login: u.login}, nil
}

type SandboxTaskAuth struct {
	task sandbox.SandboxTaskID
}

func (s SandboxTaskAuth) TaskID() sandbox.SandboxTaskID {
	return s.task
}

func (u *AuthSubject) RequireSandboxAuth() (SandboxTaskAuth, error) {
	if u.kind != SandboxTaskSubject {
		return SandboxTaskAuth{}, ErrNoSandboxAuth
	}
	return SandboxTaskAuth{task: u.sandboxTask}, nil
}

type ExecutionAuth struct {
	execution consts.ExecutionID
}

func (e ExecutionAuth) ExecutionID() consts.ExecutionID {
	return e.execution
}

func (u *AuthSubject) RequireExecutionAuth() (ExecutionAuth, error) {
	if u.kind != ExecutionSubject {
		return ExecutionAuth{}, ErrNoExecutionAuth
	}
	return ExecutionAuth{execution: u.executionID}, nil
}

func (u *AuthSubject) Clone() AuthSubject {
	rv := *u
	return rv
}

func (u *AuthSubject) ToString() (res string) {
	// NB: ToSting for explicit conversion (do not implement Stringer)
	switch u.kind {
	case InvalidSubject:
		res = "invalid"
	case UserSubject:
		res = "user:" + u.login
	case SandboxTaskSubject:
		res = fmt.Sprintf("sb:%d", u.sandboxTask.ToInt())
	case ExecutionSubject:
		res = fmt.Sprintf("execution:%q", u.executionID.String())
	default:
		panic(fmt.Sprintf("invalid kind: %v", u.kind))
	}
	return
}

func WithObject(ctx context.Context, object AuthSubject) context.Context {
	value := ctx.Value(&authKey)
	if value == nil {
		objectCopy := object.Clone()
		return context.WithValue(ctx, &authKey, &objectCopy)
	}
	stored, ok := value.(*AuthSubject)
	if !ok {
		panic(fmt.Sprintf("Invalid context type for authKey: %T", value))
	}

	if *stored != object {
		panic(fmt.Sprintf("double auth in single context: Old: %v, New %v", *stored, object))
	}
	return ctx

}

// WithUserName augments context with User auth
func WithUserName(ctx context.Context, user string) context.Context {
	return WithObject(ctx, AuthSubject{kind: UserSubject, login: user})
}

// WithSandboxTaskName augments context with SandboxTaskID auth
func WithSandboxTaskName(ctx context.Context, task sandbox.SandboxTaskID) context.Context {
	return WithObject(ctx, AuthSubject{kind: SandboxTaskSubject, sandboxTask: task})
}

// WithExecutionID augments context with ExecutionID auth
func WithExecutionID(ctx context.Context, executionID consts.ExecutionID) context.Context {
	return WithObject(ctx, AuthSubject{kind: ExecutionSubject, executionID: executionID})
}

func GetAuthSubject(ctx context.Context) (AuthSubject, error) {
	if auth, ok := ctx.Value(&authKey).(*AuthSubject); ok && auth != nil {
		return auth.Clone(), nil
	}
	return NewInvalid(), ErrNoAuth
}
