package revproxy

import (
	"context"
	"net/http"
	"strings"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/log"
)

var requestIDKey struct{}

func WithRequestID(ctx context.Context, reqID uuid.UUID) context.Context {
	return context.WithValue(ctx, &requestIDKey, reqID)
}

func ContextRequestID(ctx context.Context) (reqID uuid.UUID) {
	val := ctx.Value(&requestIDKey)
	if val != nil {
		reqID = val.(uuid.UUID)
	}
	return
}

func AddLogger(l log.Logger, next http.Handler) http.Handler {
	return http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			// Prepare fields to log
			var scheme string
			if r.TLS != nil {
				scheme = "https"
			} else {
				scheme = "http"
			}
			proto := r.Proto
			method := r.Method
			remoteAddr := r.RemoteAddr
			userAgent := r.UserAgent()
			uri := strings.Join([]string{scheme, "://", r.Host, r.RequestURI}, "")

			l.Debug(
				"request started",
				log.String("scheme", scheme),
				log.String("proto", proto),
				log.String("method", method),
				log.String("remote-addr", remoteAddr),
				log.String("user-agent", userAgent),
				log.String("uri", uri),
			)

			t1 := time.Now()
			next.ServeHTTP(w, r)

			l.Debug(
				"request completed",
				log.String("scheme", scheme),
				log.String("proto", proto),
				log.String("method", method),
				log.String("remote-addr", remoteAddr),
				log.String("user-agent", userAgent),
				log.String("uri", uri),
				log.Duration("elapsed", time.Since(t1)),
			)
		},
	)
}
