package revproxy

import (
	"context"
	"fmt"
	"net/http"
	"net/textproto"
	"strconv"

	"github.com/grpc-ecosystem/grpc-gateway/runtime"
	"google.golang.org/grpc"

	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/core/resource"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/library/go/httputil/swaggerui"
	"a.yandex-team.ru/library/go/yandex/solomon/reporters/puller/httppuller"
)

type BackendConfig struct {
	Host string `yaml:"host" valid:"required"`
	Port int    `yaml:"port" valid:"greater=0"`
}

func (bc *BackendConfig) Endpoint() string {
	return fmt.Sprintf("%s:%d", bc.Host, bc.Port)
}

type ProxyConfig struct {
	Port    int            `yaml:"port" valid:"greater=0"`
	Backend *BackendConfig `yaml:"backend"`
}

type HTTPProxy struct {
	conf   *ProxyConfig
	logger log.Logger
	Srv    *http.Server
	Mux    *runtime.ServeMux
}

func addSwaggerHandle(mux *http.ServeMux) error {
	schema := resource.Get("tasklet_service.swagger.json")
	if schema == nil {
		return xerrors.New("Failed to load swagger schema. IDE launch?")
	}
	opts := make([]swaggerui.Option, 0)
	opts = append(opts, swaggerui.WithJSONScheme(schema))
	swaggerHandler := http.StripPrefix(
		"/_swagger/",
		http.FileServer(
			swaggerui.NewFileSystem(opts...),
		),
	)
	mux.Handle("/_swagger/", swaggerHandler)
	return nil
}

func NewProxy(conf *ProxyConfig, l log.Logger, sr *solomon.Registry) (*HTTPProxy, error) {
	rootMux := http.NewServeMux()
	proxyMux := runtime.NewServeMux(
		runtime.WithMarshalerOption(
			runtime.MIMEWildcard, &runtime.JSONPb{
				OrigName:    true,
				AnyResolver: nil,
			},
		),
		runtime.WithIncomingHeaderMatcher(
			func(key string) (string, bool) {
				key = textproto.CanonicalMIMEHeaderKey(key)
				switch key {
				case consts.RequestIDHeader:
					return key, true
				}
				return runtime.DefaultHeaderMatcher(key)
			},
		),
	)
	if err := addSwaggerHandle(rootMux); err != nil {
		l.Errorf("Swagger handle not registered: %+v", err)
	}

	solomonMiddleware := httpmetrics.New(
		sr.WithPrefix("revproxy").WithPrefix("http"),
		// TODO: httpmetrics.WithEndpointKey()
	)

	rootMux.Handle("/_solomon/", httppuller.NewHandler(sr))
	rootMux.Handle("/", solomonMiddleware(proxyMux))
	srv := &http.Server{
		Addr:    ":" + strconv.Itoa(conf.Port),
		Handler: AddLogger(l, rootMux),
	}

	return &HTTPProxy{
		conf,
		l,
		srv,
		proxyMux,
	}, nil
}

func (hp *HTTPProxy) Serve() error {
	hp.logger.Debugf("Starting to server http proxy with config %+v", hp.conf)
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	opts := []grpc.DialOption{grpc.WithInsecure()}
	endpoint := hp.conf.Backend.Endpoint()
	hp.logger.Infof("Use endpoint %q as backend", endpoint)

	if err := taskletv2.RegisterTaskletServiceHandlerFromEndpoint(ctx, hp.Mux, endpoint, opts); err != nil {
		return err
	}
	return hp.Srv.ListenAndServe()
}

func (hp *HTTPProxy) Stop(ctx context.Context) error {
	return hp.Srv.Shutdown(ctx)
}
