package sbresolver

import (
	"context"
	"runtime/debug"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"

	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/state"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

var TickIDField = "resolver_tick_id"

type Resolver struct {
	conf    *Config
	sbx     *sandbox.Client
	logger  log.Logger
	storage *state.State
	stop    chan struct{}
}

func New(conf *Config, sbx *sandbox.Client, logger log.Logger) (*Resolver, error) {
	return &Resolver{
		conf:    conf,
		sbx:     sbx,
		logger:  logger,
		storage: &state.SandboxState,
		stop:    make(chan struct{}),
	}, nil
}

func (r *Resolver) Bootstrap(ctx context.Context) error {
	if !r.conf.Enabled {
		return nil
	}

	return r.doTick(ctx)
}

func (r *Resolver) Run() error {
	if !r.conf.Enabled {
		<-r.stop
		return nil
	}
MainLoop:
	for {
		tickCtx, cancel := context.WithCancel(context.Background())
		tickChan := r.Tick(tickCtx)
		select {
		case <-tickChan:
			cancel()
			// noop
		case <-r.stop:
			cancel()
			break MainLoop
		}

		select {
		case <-time.After(time.Second * time.Duration(r.conf.ResolvePeriod+1)):
			// noop
		case <-r.stop:
			break MainLoop
		}
	}
	return nil
}

func (r *Resolver) Stop() {
	close(r.stop)
}

func (r *Resolver) resolveResource(
	ctx context.Context, resourceType consts.SandboxResourceType, query *SandboxResourceQuery,
) error {

	var (
		resource *sandbox.ResourceInfo
		err      error
	)
	if query.ResourceID != 0 {
		resource, err = r.sbx.GetResourceInfo(ctx, query.ResourceID)
	} else {
		resource, err = r.sbx.LookupResource(ctx, query.Type, query.Owner, query.Attributes)
	}
	if err != nil {
		return err
	}

	ctxlog.Info(
		ctx,
		r.logger,
		"Resolved resource",
		log.String("resource_type", resourceType.String()),
		log.Int64("resource_id", resource.ID),
	)
	r.storage.SetResource(resourceType, *resource)
	return nil
}

func (r *Resolver) doTick(ctx context.Context) error {
	var err error
	for _, resource := range []struct {
		tp    consts.SandboxResourceType
		query *SandboxResourceQuery
	}{
		{consts.ArcClientResourceType, r.conf.ArcClient},
		{consts.SandboxResourceManagerType, r.conf.SandboxResourceManager},
		{consts.TaskletExecutorResourceType, r.conf.TaskletExecutor},
		{consts.TaskletTaskResourceType, r.conf.TaskletTask},
	} {
		if err = r.resolveResource(ctx, resource.tp, resource.query); err != nil {
			r.logger.Errorf(
				"Error on resolving resource %v with query %+v: %v",
				resource.tp.String(), resource.query, err,
			)
		}
	}
	return err
}

func (r *Resolver) Tick(ctx context.Context) <-chan struct{} {
	ctx = ctxlog.WithFields(
		ctx,
		log.String(TickIDField, uuid.Must(uuid.NewV4()).String()),
	)
	ctxlog.Info(ctx, r.logger, "New tick")
	defer func() {
		ctxlog.Info(ctx, r.logger, "Tick finished")
	}()

	tickCtx, tickCancel := context.WithCancel(ctx)

	go func() {
		defer tickCancel()
		defer func() {
			if p := recover(); p != nil {
				err := xerrors.Errorf("Panic: %v", p)
				stack := debug.Stack()
				fields := []log.Field{
					log.Error(err),
					log.Any("trace", string(stack)),
				}
				ctxlog.Error(tickCtx, r.logger, "Recovered panic", fields...)
			}
		}()
		err := r.doTick(tickCtx)
		if err != nil {
			ctxlog.Error(tickCtx, r.logger, "Tick failed", log.Error(err))
		} else {
			ctxlog.Debug(tickCtx, r.logger, "Tick done")
		}
	}()
	return tickCtx.Done()
}
