package schemaregistry

import (
	"context"
	"errors"
	"time"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	taskletApi "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/requestctx"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
)

type Handler struct {
	log log.Logger
	db  storage.IStorage
}

func New(l log.Logger, s storage.IStorage) (*Handler, error) {
	return &Handler{
		log: l,
		db:  s,
	}, nil
}

func (h *Handler) CreateSchema(
	ctx context.Context,
	request *taskletApi.CreateSchemaRequest,
) (*taskletApi.CreateSchemaResponse, error) {

	login := ""
	if subject, err := requestctx.GetAuthSubject(ctx); err != nil {
		return nil, status.Errorf(codes.Internal, "Missing subject context: %v", err)
	} else {
		userAuth, err := subject.RequireUserAuth()
		if err != nil {
			return nil, status.Errorf(codes.PermissionDenied, err.Error())
		}
		login = userAuth.Login()
	}

	if err := ValidateCreateSchemaRequest(request); err != nil {
		return nil, err.Err()
	}

	hash := ""
	if h, err := calculateFDSHash(request.GetSchema()); err != nil {
		return nil, status.Errorf(codes.InvalidArgument, "Invalid fds: %v", err)
	} else {
		hash = h
	}
	req := common.SchemaRecord{
		Hash:        hash,
		User:        login,
		Timestamp:   time.Now().UnixNano(),
		Fds:         request.GetSchema(),
		Annotations: request.GetAnnotations(),
	}
	record, err := h.db.EnsureSchema(ctx, req)
	if err != nil {
		return nil, err
	}

	response := &taskletApi.CreateSchemaResponse{
		Hash:        record.Hash,
		Annotations: record.Annotations,
		Meta: &taskletApi.SchemaMetadata{
			User:      record.User,
			Timestamp: record.Timestamp,
		},
	}
	return response, nil
}

func (h *Handler) GetSchema(ctx context.Context, request *taskletApi.GetSchemaRequest) (
	*taskletApi.GetSchemaResponse,
	error,
) {
	record, err := h.db.GetSchema(ctx, request.GetHash())
	if err != nil {
		if errors.Is(err, common.ErrObjectNotFound) {
			return nil, status.Errorf(codes.NotFound, "Schema not found")
		} else {
			return nil, err
		}
	}

	response := &taskletApi.GetSchemaResponse{
		Schema:      record.Fds,
		Annotations: record.Annotations,
		Meta: &taskletApi.SchemaMetadata{
			User:      record.User,
			Timestamp: record.Timestamp,
		},
	}
	return response, nil
}
