package schemaregistry

import (
	"context"
	"encoding/base64"
	"testing"

	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/require"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/descriptorpb"
	"google.golang.org/protobuf/types/known/structpb"

	"a.yandex-team.ru/library/go/test/requirepb"
	taskletApi "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/requestctx"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/mock"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
)

func mustStruct(v map[string]interface{}) *structpb.Struct {
	rv, err := structpb.NewStruct(v)
	if err != nil {
		panic(err)
	}
	return rv
}

// protoc -o /dev/stdout some.proto  | base64
//goland:noinspection SpellCheckingInspection
const sampleFDS string = "" +
	"Cp8CChZzdHJpbmdfY29udGFpbmVyLnByb3RvEjd0YXNrbGV0LmV4cGVyaW1lbnRhbC5yZWdpc3Ry" +
	"eV90ZXN0LnByaW1pdGl2ZV9jb250YWluZXJzIjQKD1N0cmluZ0NvbnRhaW5lchIhCgxzdHJpbmdf" +
	"dmFsdWUYASABKAlSC3N0cmluZ1ZhbHVlQo0BCkFydS55YW5kZXgudGFza2xldC5leHBlcmltZW50" +
	"YWwucmVnaXN0cnlfdGVzdC5wcmltaXRpdmVfY29udGFpbmVyc1pIYS55YW5kZXgtdGVhbS5ydS90" +
	"YXNrbGV0L2V4cGVyaW1lbnRhbC9yZWdpc3RyeV90ZXN0L3ByaW1pdGl2ZV9jb250YWluZXJzYgZw" +
	"cm90bzM="

func parseFDS() (*descriptorpb.FileDescriptorSet, string) {
	data, err := base64.StdEncoding.DecodeString(sampleFDS)
	if err != nil {
		panic(err)
	}
	fds := &descriptorpb.FileDescriptorSet{}
	if err := proto.Unmarshal(data, fds); err != nil {
		panic(err)
	}
	hash, err := calculateFDSHash(fds)
	if err != nil {
		panic(err)
	}
	return fds, hash
}

func TestHandler_CreateSchema(t *testing.T) {
	fds, hash := parseFDS()

	const sampleUser = "schema_user"
	tests := []struct {
		name         string
		storageSetup func(*testing.T, *mock.MockIStorage)
		request      *taskletApi.CreateSchemaRequest
		response     *taskletApi.CreateSchemaResponse
		wantErr      bool
	}{
		{
			"simple",
			func(tt *testing.T, db *mock.MockIStorage) {
				db.EXPECT().
					EnsureSchema(gomock.Any(), gomock.Any()).
					DoAndReturn(
						func(ctx context.Context, req common.SchemaRecord) (common.SchemaRecord, error) {
							require.Equal(tt, hash, req.Hash)
							require.Equal(tt, sampleUser, req.User)
							require.Less(tt, int64(0), req.Timestamp)
							requirepb.Equal(tt, fds, req.Fds)
							requirepb.Equal(tt, mustStruct(nil), req.Annotations)
							return common.SchemaRecord{
								Hash:        hash,
								User:        sampleUser,
								Timestamp:   req.Timestamp,
								Fds:         req.Fds,
								Annotations: req.Annotations,
							}, nil
						},
					).
					Times(1)
			},
			&taskletApi.CreateSchemaRequest{
				Schema:      proto.Clone(fds).(*descriptorpb.FileDescriptorSet),
				Annotations: mustStruct(nil),
			},
			&taskletApi.CreateSchemaResponse{
				Hash:        hash,
				Annotations: mustStruct(nil),
				Meta: &taskletApi.SchemaMetadata{
					User:      sampleUser,
					Timestamp: 0,
				},
			},
			false,
		},
		{
			"bad_req",
			func(tt *testing.T, db *mock.MockIStorage) {
				db.EXPECT().
					EnsureSchema(gomock.Any(), gomock.Any()).
					Return(common.SchemaRecord{}, nil).
					Times(0)
			},
			&taskletApi.CreateSchemaRequest{
				Schema:      nil,
				Annotations: mustStruct(nil),
			},
			nil,
			true,
		},
	}
	for _, tt := range tests {
		t.Run(
			tt.name, func(t *testing.T) {
				ctrl := gomock.NewController(t)
				defer ctrl.Finish()
				db := mock.NewMockIStorage(ctrl)
				tt.storageSetup(t, db)
				h := &Handler{
					log: testutils.MakeLogger("/dev/null"),
					db:  db,
				}
				ctx := requestctx.WithObject(context.Background(), requestctx.NewUser(sampleUser))
				got, err := h.CreateSchema(ctx, tt.request)
				if (err != nil) != tt.wantErr {
					if st, ok := status.FromError(err); ok {
						require.Fail(t, protojson.Format(st.Proto()))
					} else {
						t.Errorf("CreateSchema() error = %v, wantErr %v", err, tt.wantErr)
					}
					return
				}
				if got.GetMeta() != nil {
					got.Meta.Timestamp = 0
				}
				requirepb.Equal(t, got, tt.response)
			},
		)
	}
}

func TestHandler_GetSchema(t *testing.T) {
	fds, hash := parseFDS()
	// require.Fail(t, protojson.Format(fds))
	const sampleUser = "schema_user"
	const timestamp = 1543
	tests := []struct {
		name         string
		storageSetup func(*testing.T, *mock.MockIStorage)
		request      *taskletApi.GetSchemaRequest
		response     *taskletApi.GetSchemaResponse
		wantErr      bool
	}{
		{
			"simple",
			func(tt *testing.T, db *mock.MockIStorage) {
				db.EXPECT().
					GetSchema(gomock.Any(), gomock.Any()).
					DoAndReturn(
						func(ctx context.Context, reqHash string) (common.SchemaRecord, error) {
							require.Equal(tt, hash, reqHash)
							return common.SchemaRecord{
								Hash:        hash,
								User:        "someone",
								Timestamp:   timestamp,
								Fds:         proto.Clone(fds).(*descriptorpb.FileDescriptorSet),
								Annotations: mustStruct(nil),
							}, nil
						},
					).
					Times(1)
			},
			&taskletApi.GetSchemaRequest{
				Hash: hash,
			},
			&taskletApi.GetSchemaResponse{
				Schema:      proto.Clone(fds).(*descriptorpb.FileDescriptorSet),
				Annotations: mustStruct(nil),
				Meta: &taskletApi.SchemaMetadata{
					User:      "someone",
					Timestamp: timestamp,
				},
			},
			false,
		},
	}
	for _, tt := range tests {
		t.Run(
			tt.name, func(t *testing.T) {
				ctrl := gomock.NewController(t)
				defer ctrl.Finish()
				db := mock.NewMockIStorage(ctrl)
				tt.storageSetup(t, db)
				h := &Handler{
					log: testutils.MakeLogger("/dev/null"),
					db:  db,
				}
				ctx := requestctx.WithObject(context.Background(), requestctx.NewUser(sampleUser))
				got, err := h.GetSchema(ctx, tt.request)
				if (err != nil) != tt.wantErr {
					if st, ok := status.FromError(err); ok {
						require.Fail(t, protojson.Format(st.Proto()))
					} else {
						t.Errorf("CreateSchema() error = %v, wantErr %v", err, tt.wantErr)
					}
					return
				}
				requirepb.Equal(t, got, tt.response)
			},
		)
	}
}
