package ydbstore

import (
	"context"
	"math"
	"text/template"

	"github.com/ydb-platform/ydb-go-sdk/v3/table"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

func (s *Storage) AddBuild(ctx context.Context, b *taskletv2.Build) error {
	if b.GetMeta().GetId() == "" {
		panic("missing object ID")
	}
	if b.GetMeta().GetTaskletId() == "" {
		panic("missing object ID")
	}

	meta := b.GetMeta()

	serialized, err := protojson.Marshal(b)
	if err != nil {
		ctxlog.Infof(ctx, s.l, "Marshall failed: %v", err)
		return err
	}
	query := s.client.QueryPrefix() + Tables[BuildsTable].Queries[InsertQuery]
	s.logQuery(ctx, "add_build", query)

	errQ := s.client.ExecuteWriteQuery(
		ctx,
		query,
		table.ValueParam("$"+IDColumn, ydbTypes.StringValueFromString(meta.Id)),
		table.ValueParam("$"+TaskletIDColumn, ydbTypes.StringValueFromString(meta.TaskletId)),
		table.ValueParam("$"+RevisionColumn, ydbTypes.Int64Value(meta.Revision)),
		table.ValueParam("$"+PayloadColumn, ydbTypes.JSONDocumentValueFromBytes(serialized)),
	)
	if errQ != nil {
		return errQ
	}
	return nil

}

func (s *Storage) GetBuild(ctx context.Context, objID string) (*taskletv2.Build, error) {
	q := s.client.QueryPrefix() + Tables[BuildsTable].Queries[SelectByIDQuery]
	ctxlog.Info(ctx, s.l, q)
	var payload *[]byte = nil
	err := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, q, table.NewQueryParameters(
					table.ValueParam("$"+IDColumn, ydbTypes.StringValueFromString(objID)),
				),
			)
			ctxlog.Infof(ctx, s.l, "QRES: r: %v, err: %v", res, err)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			itemsProcessed := 0
			for res.NextResultSet(ctx, PayloadColumn) {
				for res.NextRow() {
					err := res.Scan(&payload)
					if err != nil {
						return err
					}
					itemsProcessed += 1
					if itemsProcessed > 1 {
						ctxlog.Fatalf(ctx, s.l, "multiple results. Name: %q", objID)
					}
				}
			}
			if itemsProcessed == 0 {
				return common.ErrObjectNotFound
			}
			return res.Err()
		},
	)
	if err != nil {
		return nil, err
	}
	rv := &taskletv2.Build{}
	if err := protojson.Unmarshal(*payload, rv); err != nil {
		return nil, err
	}
	return rv, nil
}

func (s *Storage) GetLastBuild(ctx context.Context, taskletID string) (*taskletv2.Build, error) {
	q := s.client.QueryPrefix() + Tables[BuildsTable].Queries[SelectByNameQuery]
	s.logQuery(ctx, "last_build", q)
	var payload *[]byte = nil
	err := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, q, table.NewQueryParameters(
					table.ValueParam("$"+TaskletIDColumn, ydbTypes.StringValueFromString(taskletID)),
				),
			)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			itemsProcessed := 0
			for res.NextResultSet(c, PayloadColumn) {
				for res.NextRow() {
					err := res.Scan(&payload)
					if err != nil {
						return err
					}
					itemsProcessed += 1
					if itemsProcessed > 1 {
						ctxlog.Fatalf(c, s.l, "multiple results. Name: %q", taskletID)
					}
				}
			}
			return res.Err()
		},
	)
	if err != nil {
		return nil, err
	}
	if payload == nil {
		return nil, common.ErrObjectNotFound
	}

	rv := &taskletv2.Build{}
	if err := protojson.Unmarshal(*payload, rv); err != nil {
		return nil, err
	}
	return rv, nil
}

//goland:noinspection SqlNoDataSourceInspection
var ListBuildsTemplate = template.Must(
	template.New("lbt").Parse(
		// language=SQL
		`
		DECLARE ${{ .C.TaskletID }} AS String;
		DECLARE ${{ .C.Revision }} AS Int64;  -- Start from this revision
		DECLARE $_limit AS UInt64;

		SELECT
			{{ .C.TaskletID }}, {{ .C.Revision }}, {{ .C.P }}
		FROM {{ .C.BuildsTable }} VIEW {{ .C.TaskletIDRevisionView }}
		WHERE
		    {{ .C.TaskletID }} == ${{ .C.TaskletID }} AND
		    {{ .C.Revision }} < ${{ .C.Revision }}
		ORDER BY {{ .C.Revision }} DESC
		LIMIT $_limit
		;
		`,
	),
)

func (s *Storage) ListBuilds(
	ctx context.Context,
	taskletID string,
	options common.ListBuildsQueryOptions,
) (common.BuildsList, error) {
	query := s.client.QueryPrefix() + render(
		ListBuildsTemplate, struct {
			C TGenericRenderingOptions
		}{GenericTemplateOptions},
	)
	queryOptions := table.NewQueryParameters(
		table.ValueParam("$"+TaskletIDColumn, ydbTypes.StringValueFromString(taskletID)),
		table.ValueParam("$"+RevisionColumn, ydbTypes.Int64Value(options.Token)),
		table.ValueParam("$_limit", ydbTypes.Uint64Value(uint64(options.Limit))),
	)

	ctxlog.Infof(ctx, s.l, "List query: %v", query)
	responseBuilds := make([]*taskletv2.Build, 0)
	nextToken := int64(math.MaxInt64)
	queryErr := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, queryOptions,
			)
			ctxlog.Infof(ctx, s.l, "QRES: r: %v, err: %v", res, err)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			for res.NextResultSet(ctx, PayloadColumn) {
				for res.NextRow() {
					var payload *[]byte
					if err := res.Scan(&payload); err != nil {
						return err
					}
					item := &taskletv2.Build{}
					if err := protojson.Unmarshal(*payload, item); err != nil {
						return err
					}
					responseBuilds = append(responseBuilds, item)

					if item.GetMeta().GetRevision() < nextToken {
						nextToken = item.GetMeta().GetRevision()
					}

				}
			}
			if len(responseBuilds) == 0 {
				nextToken = 1
			}
			return res.Err()
		},
	)

	if queryErr != nil {
		return common.BuildsList{}, queryErr
	}
	return common.BuildsList{
		Builds: responseBuilds,
		Token:  nextToken,
	}, nil
}

//goland:noinspection SqlNoDataSourceInspection
var SelectBuildByIDTemplate = template.Must(
	template.New("sbit").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		SELECT
			{{ .P }}
		FROM
			{{ .BuildsTable }}
		WHERE
			{{ .ID }} == ${{ .ID }}
		;
		`,
	),
)

//goland:noinspection SqlNoDataSourceInspection
var SelectBuildByNameTemplate = template.Must(
	template.New("sbnt").Parse(
		// language=SQL
		`
		DECLARE ${{ .TaskletID }} AS String;

		$_rev = (
			SELECT
				Max({{ .Revision }})
			FROM {{ .BuildsTable }} VIEW {{ .TaskletIDRevisionView }}
			where {{ .TaskletID }} == ${{ .TaskletID }}
		);

		SELECT
			{{ .P }}
		FROM {{ .BuildsTable }} VIEW {{ .TaskletIDRevisionView }}
		WHERE
			{{ .TaskletID }} == ${{ .TaskletID }} AND {{ .Revision }} == $_rev
		;
		`,
	),
)

//goland:noinspection SqlNoDataSourceInspection
var InsertBuildTemplate = template.Must(
	template.New("ibt").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		DECLARE ${{ .TaskletID }} AS String;
		DECLARE ${{ .Revision }} AS Int64;
		DECLARE ${{ .P }} AS JsonDocument;


		$_rev = (
			SELECT
				Max({{ .Revision }})
			FROM {{ .BuildsTable }} VIEW {{ .TaskletIDRevisionView }}
			where {{ .TaskletID }} == ${{ .TaskletID }}
		);

		DISCARD SELECT
			Ensure(
				0,
				COALESCE($_rev , 0) + 1 == ${{ .Revision }},
				"Invalid revision. TaskletID: '" || ${{ .TaskletID }} || "', NewRevision: " || CAST(${{ .Revision }} AS String) || ", OldRevision: " || Unwrap(CAST($_rev AS String))
			)
		;

		INSERT INTO {{ .BuildsTable }} ({{ .ID }}, {{ .TaskletID }}, {{ .Revision }}, {{ .P }})
		VALUES (${{ .ID }}, ${{ .TaskletID }}, ${{ .Revision }}, ${{ .P }});
		`,
	),
)

func initBuilds() {
	b := xydb.TableSchema{
		Name: BuildsTable,
		Columns: []xydb.YdbColumn{
			{
				Name:       IDColumn,
				ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
				PrimaryKey: true,
			},
			{
				Name:      TaskletIDColumn,
				ValueType: ydbTypes.Optional(ydbTypes.TypeString),
			},
			{
				Name:      RevisionColumn,
				ValueType: ydbTypes.Optional(ydbTypes.TypeInt64),
			},
			{
				Name:      PayloadColumn,
				ValueType: ydbTypes.Optional(ydbTypes.TypeJSONDocument),
			},
		},
		SecondaryIndexes: []xydb.SecondaryIndex{TaskletIDRevisionIndex},
		Queries:          make(map[xydb.Query]string),
	}
	thisOpts := GenericTemplateOptions
	b.Queries[SelectByIDQuery] = render(SelectBuildByIDTemplate, thisOpts)
	b.Queries[SelectByNameQuery] = render(SelectBuildByNameTemplate, thisOpts)
	b.Queries[InsertQuery] = render(InsertBuildTemplate, thisOpts)
	Tables[b.Name] = b

}
