package ydbstore

import (
	"context"
	"errors"
	"fmt"
	"math"
	"text/template"
	"time"

	ydbTable "github.com/ydb-platform/ydb-go-sdk/v3/table"
	ydbOptions "github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	ydbNamed "github.com/ydb-platform/ydb-go-sdk/v3/table/result/named"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/valid"
	taskletApi "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/yt/go/compression"
)

var InsertExecutionToJournal = xydb.NewQuery(
	"execution_insert_to_journal", `
	DECLARE ${{ .RequestID }} AS String;
	DECLARE ${{ .Operation }} AS String;
	DECLARE ${{ .P }} AS String;
	DECLARE ${{ .TTL }} AS Timestamp;

	INSERT INTO {{ .ExecutionJournalTable }} ({{ .RequestID }}, {{ .Operation }}, {{ .P }}, {{ .TTL }})
	VALUES (${{ .RequestID }}, ${{ .Operation }}, ${{ .P }}, ${{ .TTL }});
`,
)

var LookupExecutionByRequestID = xydb.NewQuery(
	"execution_lookup_by_request_id", `
	DECLARE ${{ .RequestID }} AS String;

	SELECT {{ .P }}
	FROM {{ .ExecutionJournalTable }}
	WHERE {{ .RequestID }} = ${{ .RequestID }};
`,
)

var InsertExecutionQuery = xydb.NewQuery(
	"execution_insert_query",
	// language=YQL
	`
	DECLARE ${{ .ID }} AS String;
	DECLARE ${{ .TaskletID }} AS String;
	DECLARE ${{ .BuildID }} AS String;
	DECLARE ${{ .P }} AS JsonDocument;
	DECLARE ${{ .Input }} AS Optional<String>;
	DECLARE ${{ .Output }} AS Optional<String>;

	$_rev = SELECT {{ .Value }} FROM {{ .ExecutionsMonotonicTable }} WHERE {{ .ID }} == 1;
	UPDATE {{ .ExecutionsMonotonicTable }} SET {{ .Value }} = {{ .Value }} + 1 WHERE {{ .ID }} == 1;

	INSERT INTO {{ .ExecutionsTable }} ({{ .ID }}, {{ .TaskletID }}, {{ .BuildID }}, {{ .ExecutionMonotonicID }}, {{ .P }})
	VALUES (${{ .ID }}, ${{ .TaskletID }}, ${{ .BuildID }} , UNWRAP($_rev), ${{ .P }});

	INSERT INTO {{ .ExecutionBlobsTable }} ({{ .ID }}, {{ .Input }}, {{ .Output }})
	VALUES (${{ .ID }}, ${{ .Input }}, ${{ .Output }});
	`,
)

type ExecutionOperation string

func (eo ExecutionOperation) String() string {
	return string(eo)
}

var ExecutionCreateOperation ExecutionOperation = "create"

func validateExecution(ex *taskletApi.Execution) error {
	meta := ex.GetMeta()
	if err := valid.UUIDv4(meta.GetId()); err != nil {
		return err
	}

	if err := valid.UUIDv4(meta.GetTaskletId()); err != nil {
		return err
	}

	if err := valid.UUIDv4(meta.GetBuildId()); err != nil {
		return err
	}
	if ex.GetStatus().GetStatus() == taskletApi.EExecutionStatus_E_EXECUTION_STATUS_INVALID {
		return xerrors.NewSentinel("Uninitialized status")
	}

	return nil
}

type executionRowBuffer struct {
	payload *[]byte
	input   *[]byte
	output  *[]byte
}

var executionCompressionCodec = compression.NewCodec(compression.CodecIDSnappy)
var protoMarshalOptions = proto.MarshalOptions{}
var protoUnmarshalOptions = proto.UnmarshalOptions{}

func compressProtoMessage(msg proto.Message) ([]byte, error) {
	if serializedData, err := protoMarshalOptions.Marshal(msg); err != nil {
		return nil, xerrors.Errorf("marshal failed: %w", err)
	} else if compressedAndSerializedData, err := executionCompressionCodec.Compress(serializedData); err != nil {
		return nil, xerrors.Errorf("compression failed: %w", err)
	} else {
		return compressedAndSerializedData, nil
	}
}

func decompressProtoMessage[T proto.Message](buf []byte, msg *T) error {
	if buf == nil {
		msg = nil
		return nil
	}
	var rv T
	rv = rv.ProtoReflect().New().Interface().(T)
	if decompressed, err := executionCompressionCodec.Decompress(buf); err != nil {
		return xerrors.Errorf("decompress failed: %w", err)
	} else if err := protoUnmarshalOptions.Unmarshal(decompressed, rv); err != nil {
		return xerrors.Errorf("unmarshall failed: %w", err)
	}
	*msg = rv
	return nil

}

func (b *executionRowBuffer) toExecution() (*taskletApi.Execution, error) {
	execution := &taskletApi.Execution{}
	if err := protojson.Unmarshal(*b.payload, execution); err != nil {
		return nil, xerrors.Errorf("failed to unmarshall payload: %w", err)
	}

	if b.input != nil {
		if execution.Spec.Input != nil {
			return nil, xerrors.New("unexpected: non nil input")
		}
		if err := decompressProtoMessage(*b.input, &execution.Spec.Input); err != nil {
			return nil, xerrors.Errorf("input handling failed: %w", err)
		}
	}

	if b.output != nil {
		if execution.Status == nil {
			return nil, xerrors.New("unexpected: empty status on non-empty output")
		}
		if execution.Status.ProcessingResult == nil {
			return nil, xerrors.New("unexpected: empty processing result")
		}
		if kind := execution.Status.ProcessingResult.Kind; kind != nil {
			return nil, xerrors.Errorf("unexpected: non empty kind on existing output: %T", kind)
		}

		output := &taskletApi.ExecutionOutput{}
		if err := decompressProtoMessage(*b.output, &output); err != nil {
			return nil, xerrors.Errorf("output handling failed: %w", err)
		}
		execution.Status.ProcessingResult.Kind = &taskletApi.ProcessingResult_Output{
			Output: output,
		}
	}
	return execution, nil
}

func fromExecution(e *taskletApi.Execution) (executionRowBuffer, error) {
	// Extracting input and output
	rv := executionRowBuffer{}
	if e.Spec == nil {
		return executionRowBuffer{}, xerrors.New("unexpected: empty spec")
	}
	if e.Spec.Input != nil {
		buf, err := compressProtoMessage(e.Spec.Input)
		if err != nil {
			return executionRowBuffer{}, xerrors.Errorf("failed to process input: %w", err)
		} else {
			rv.input = &buf
		}
	} else {
		rv.input = nil
	}
	// NB: hide input from json marshaller
	inputBackup := e.Spec.Input
	e.Spec.Input = nil
	defer func() {
		e.Spec.Input = inputBackup
	}()

	if e.Status.GetProcessingResult().GetOutput() != nil {
		if buf, err := compressProtoMessage(e.Status.ProcessingResult.GetOutput()); err != nil {
			return executionRowBuffer{}, xerrors.Errorf("failed to process output: %w", err)
		} else {
			rv.output = &buf
		}
		kindBackup := e.Status.ProcessingResult.Kind
		e.Status.ProcessingResult.Kind = nil
		defer func() {
			e.Status.ProcessingResult.Kind = kindBackup
		}()
	}

	if buf, err := protojson.Marshal(e); err != nil {
		return executionRowBuffer{}, xerrors.Errorf("execution marshal failed: %w", err)
	} else {
		rv.payload = &buf
	}
	return rv, nil

}

func (s *Storage) AddExecution(
	ctx context.Context,
	requestID consts.RequestID,
	e *taskletApi.Execution,
) (*taskletApi.Execution, error) {
	if requestID == consts.NilRequestID {
		return nil, xerrors.New("nil request ID")
	}
	ctxlog.Infof(ctx, s.l, "Adding execution. Id: %q", e.GetMeta().GetId())
	if err := validateExecution(e); err != nil {
		return nil, err
	}

	serializedExecution, err := proto.Marshal(e)
	if err != nil {
		return nil, xerrors.Errorf("document marshalling failed: %w", err)
	}

	buf, err := fromExecution(e)
	if err != nil {
		return nil, err
	}

	var journaledResponseBuffer *[]byte

	to := GenericTemplateOptions
	journalQuery := s.client.QueryPrefix() + InsertExecutionToJournal.Query()
	s.logQuery(ctx, InsertExecutionToJournal.Name, journalQuery)
	journalQueryParams := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+to.RequestID, ydbTypes.StringValueFromString(requestID.String())),
		ydbTable.ValueParam("$"+to.Operation, ydbTypes.StringValueFromString(ExecutionCreateOperation.String())),
		ydbTable.ValueParam("$"+to.P, ydbTypes.StringValue(serializedExecution)),
		ydbTable.ValueParam("$"+to.TTL, ydbTypes.TimestampValueFromTime(time.Now().Add(time.Hour))),
	)

	lookupByRequestIDQuery := s.client.QueryPrefix() + LookupExecutionByRequestID.Query()
	s.logQuery(ctx, LookupExecutionByRequestID.Name, lookupByRequestIDQuery)
	lookupByRequestIDQueryParams := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+to.RequestID, ydbTypes.StringValueFromString(requestID.String())),
	)
	insertQuery := s.client.QueryPrefix() + InsertExecutionQuery.Query()
	s.logQuery(ctx, InsertExecutionQuery.Name, insertQuery)
	insertQueryParams := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+to.ID, ydbTypes.StringValueFromString(e.Meta.Id)),
		ydbTable.ValueParam("$"+to.TaskletID, ydbTypes.StringValueFromString(e.Meta.TaskletId)),
		ydbTable.ValueParam("$"+to.BuildID, ydbTypes.StringValueFromString(e.Meta.BuildId)),
		ydbTable.ValueParam("$"+to.P, ydbTypes.JSONDocumentValueFromBytes(*buf.payload)),
		ydbTable.ValueParam("$"+to.Input, ydbTypes.NullableStringValue(buf.input)),
		ydbTable.ValueParam("$"+to.Output, ydbTypes.NullableStringValue(buf.output)),
	)

	txOperation := func(txCtx context.Context, tx ydbTable.TransactionActor) error {
		ctxlog.Info(txCtx, s.l, "Executing lookup by reqId")
		res, err := tx.Execute(txCtx, lookupByRequestIDQuery, lookupByRequestIDQueryParams)
		if err != nil {
			return err
		}
		defer func() { _ = res.Close() }()

		ctxlog.Infof(txCtx, s.l, "Got lookup by reqId result. SetCount: %v", res.ResultSetCount())
		resultCount := 0
		for res.NextResultSet(txCtx, to.P) {
			ctxlog.Infof(txCtx, s.l, "Processing next result set. RowCount: %v", res.CurrentResultSet().RowCount())
			for res.NextRow() {
				resultCount += 1
				if resultCount > 1 {
					panic(fmt.Sprintf("Multiple results from fallback query. ReqID: %q", requestID.String()))
				}
				_ = res.ScanNamed(
					ydbNamed.Optional(to.P, &journaledResponseBuffer),
				)
			}
		}
		if res.Err() != nil {
			return res.Err()
		}
		if resultCount == 1 {
			ctxlog.Info(txCtx, s.l, "returning journaled response")
			return nil
		}

		ctxlog.Info(txCtx, s.l, "journaling execution creation")
		_, errJournal := tx.Execute(txCtx, journalQuery, journalQueryParams)
		if errJournal != nil {
			return errJournal
		}
		ctxlog.Info(txCtx, s.l, "inserting new execution")
		_, errInsert := tx.Execute(txCtx, insertQuery, insertQueryParams)
		return errInsert
	}

	txErr := s.tableDoTx(
		ctx, txOperation, ydbTable.WithTxSettings(
			ydbTable.TxSettings(
				ydbTable.WithSerializableReadWrite(),
			),
		),
	)
	if txErr != nil {
		return nil, txErr
	}
	if journaledResponseBuffer != nil {
		rv := &taskletApi.Execution{}
		if err := proto.Unmarshal(*journaledResponseBuffer, rv); err != nil {
			return nil, xerrors.Errorf("Journaled response unmarshall failed: %w", err)
		}
		return rv, nil
	}
	return proto.Clone(e).(*taskletApi.Execution), nil
}

var SelectExecutionByIDTemplate = xydb.NewQuery(
	"select_execution_by_id",
	// language=YQL
	`
	DECLARE ${{ .ID }} AS String;
	SELECT
		a.{{ .P }} AS {{ .P }},
		b.{{ .Input }} AS {{ .Input }},
		b.{{ .Output }} AS {{ .Output }}
	FROM
		{{ .ExecutionsTable }} AS a
	JOIN {{ .ExecutionBlobsTable }} AS b
	ON a.{{ .ID }} = b.{{ .ID }}
	WHERE
		a.{{ .ID }} = ${{ .ID }}
	;
	`,
)

var SelectExecutionArchiveByIDTemplate = xydb.NewQuery(
	"select_execution_archive_by_id",
	// language=YQL
	`
	DECLARE ${{ .ID }} AS String;
	SELECT
		a.{{ .P }} AS {{ .P }},
		b.{{ .Input }} AS {{ .Input }},
		b.{{ .Output }} AS {{ .Output }}
	FROM
		{{ .ExecutionsArchiveTable }} AS a
	JOIN {{ .ExecutionBlobsTable }} AS b
	ON a.{{ .ID }} = b.{{ .ID }}
	WHERE
		a.{{ .ID }} = ${{ .ID }}
	;
`,
)

func (s *Storage) doGetExecutionByID(
	ctx context.Context,
	session ydbTable.Session,
	executionID consts.ExecutionID,
	archive bool,
) (
	*taskletApi.Execution,
	error,
) {
	query := s.client.QueryPrefix()
	if !archive {
		query += SelectExecutionByIDTemplate.Query()
		s.logQuery(ctx, SelectExecutionByIDTemplate.Name, query)
	} else {
		query += SelectExecutionArchiveByIDTemplate.Query()
		s.logQuery(ctx, SelectExecutionArchiveByIDTemplate.Name, query)
	}
	_, res, err := session.Execute(
		ctx, s.client.ReadTxControl, query, ydbTable.NewQueryParameters(
			ydbTable.ValueParam("$"+IDColumn, ydbTypes.StringValueFromString(executionID.String())),
		),
	)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = res.Close()
	}()
	buf := executionRowBuffer{}
	if err := xydb.EnsureOneRowCursor(ctx, res); err != nil {
		if xerrors.Is(err, xydb.ErrNoRows) {
			return nil, common.ErrObjectNotFound
		}
		if xerrors.Is(err, xydb.ErrMoreThenOneRow) {
			return nil, common.ErrSchemaError.Wrap(
				xerrors.Errorf(
					"Multiple results. ExecutionID: %q",
					executionID.String(),
				),
			)
		}
		return nil, err
	}
	if err := res.ScanNamed(
		ydbNamed.Optional(GenericTemplateOptions.P, &buf.payload),
		ydbNamed.Optional(GenericTemplateOptions.Input, &buf.input),
		ydbNamed.Optional(GenericTemplateOptions.Output, &buf.output),
	); err != nil {
		return nil, err
	}
	if err := res.Err(); err != nil {
		return nil, err
	}
	return buf.toExecution()

}

func (s *Storage) getExecutionByID(ctx context.Context, session ydbTable.Session, executionID consts.ExecutionID) (
	*taskletApi.Execution,
	error,
) {
	// NB: execution migrates from active executions to archive table, not backward
	// This makes reads serializable with writes for single API get requests
	execution, err := s.doGetExecutionByID(ctx, session, executionID, false)
	if err != nil && errors.Is(err, common.ErrObjectNotFound) {
		return s.doGetExecutionByID(ctx, session, executionID, true)
	}
	return execution, err
}

func (s *Storage) GetExecution(ctx context.Context, id string) (*taskletApi.Execution, error) {
	var result *taskletApi.Execution
	executionID := consts.ExecutionID(id)
	op := func(c context.Context, session ydbTable.Session) error {
		value, err := s.getExecutionByID(c, session, executionID)
		if err == nil {
			result = value
		}
		return err
	}
	return result, s.tableDo(ctx, op)
}

var UpdateExecutionTemplate = xydb.NewQuery(
	"update_execution_template",
	// language=YQL
	`
	DECLARE ${{ .ID }} AS String;
	DECLARE ${{ .P }} AS JsonDocument;
	DECLARE ${{ .Input }} AS Optional<String>;
	DECLARE ${{ .Output }} AS Optional<String>;

	UPDATE {{ .ExecutionsTable }}
	SET {{ .P }} = ${{ .P }}
	WHERE {{ .ID }} == ${{ .ID }};

	UPDATE {{ .ExecutionBlobsTable }}
	SET {{ .Input }} = ${{ .Input }}, {{ .Output }} = ${{ .Output }}
	WHERE {{ .ID }} == ${{ .ID }};
`,
)

func (s *Storage) doUpdateExecutionStatus(
	ctx context.Context,
	session ydbTable.Session,
	executionID consts.ExecutionID,
	updates ...common.ExecutionStatusUpdateFunc,
) (*taskletApi.Execution, error) {

	selectQuery := s.client.QueryPrefix()
	selectQuery += SelectExecutionByIDTemplate.Query()
	selectQueryParameters := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+GenericTemplateOptions.ID, ydbTypes.StringValueFromString(executionID.String())),
	)
	s.logQuery(ctx, SelectExecutionByIDTemplate.Name, selectQuery)

	txc := ydbTable.TxControl(ydbTable.BeginTx(ydbTable.WithSerializableReadWrite()))
	buf := executionRowBuffer{}
	tx, res, err := session.Execute(
		ctx, txc, selectQuery, selectQueryParameters,
	)
	ctxlog.Infof(ctx, s.l, "Query result. txOk: %v, err: %+v", tx != nil, err)
	defer func() {
		if tx != nil {
			_ = tx.Rollback(ctx)
		}
	}()
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = res.Close()
	}()
	if err := xydb.EnsureOneRowCursor(ctx, res); err != nil {
		if xerrors.Is(err, xydb.ErrNoRows) {
			return nil, common.ErrObjectNotFound
		}
		if xerrors.Is(err, xydb.ErrMoreThenOneRow) {
			return nil, common.ErrSchemaError.Wrap(
				xerrors.Errorf(
					"Multiple results. ExecutionID: %q",
					executionID.String(),
				),
			)
		}
		return nil, err
	}
	if err := res.ScanNamed(
		ydbNamed.Optional(GenericTemplateOptions.P, &buf.payload),
		ydbNamed.Optional(GenericTemplateOptions.Input, &buf.input),
		ydbNamed.Optional(GenericTemplateOptions.Output, &buf.output),
	); err != nil {
		return nil, err
	}
	if err := res.Err(); err != nil {
		return nil, err
	}
	result, err := buf.toExecution()
	if err != nil {
		return nil, err
	}
	for _, updateOp := range updates {
		if err := updateOp(result.Status); err != nil {
			return nil, xerrors.NewSentinel("Update status failed").Wrap(err)
		}
	}

	outBuf, err := fromExecution(result)
	if err != nil {
		return nil, err
	}

	updateQuery := s.client.QueryPrefix() + UpdateExecutionTemplate.Query()
	updateQueryParameters := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+GenericTemplateOptions.ID, ydbTypes.StringValueFromString(result.Meta.Id)),
		ydbTable.ValueParam("$"+GenericTemplateOptions.P, ydbTypes.JSONDocumentValueFromBytes(*outBuf.payload)),
		ydbTable.ValueParam("$"+GenericTemplateOptions.Input, ydbTypes.NullableStringValue(outBuf.input)),
		ydbTable.ValueParam("$"+GenericTemplateOptions.Output, ydbTypes.NullableStringValue(outBuf.output)),
	)

	_, err = tx.Execute(
		ctx,
		updateQuery,
		updateQueryParameters,
		ydbOptions.WithCollectStatsModeBasic(),
	)
	if err != nil {
		return nil, err
	}

	if _, err := tx.CommitTx(ctx, ydbOptions.WithCommitCollectStatsModeBasic()); err != nil {
		return nil, err
	}

	return result, nil
}

func (s *Storage) UpdateExecutionStatus(
	ctx context.Context,
	id consts.ExecutionID,
	updates ...common.ExecutionStatusUpdateFunc,
) (*taskletApi.Execution, error) {

	var result *taskletApi.Execution
	queryErr := s.tableDo(
		ctx, func(c context.Context, session ydbTable.Session) error {
			var updateErr error
			result, updateErr = s.doUpdateExecutionStatus(c, session, id, updates...)
			return updateErr
		},
	)
	if queryErr != nil {
		return nil, queryErr
	} else {
		return result, nil
	}
}

var ArchiveExecutionTemplate = xydb.NewQuery(
	"archive_execution",
	// language=YQL
	`
	DECLARE ${{ .ID }} AS String;

	INSERT INTO {{ .ExecutionsArchiveTable }}
	SELECT {{ .ID }}, {{ .TaskletID }}, {{ .BuildID }}, {{ .ExecutionMonotonicID }}, {{ .P }}
	FROM {{ .ExecutionsTable }}
	WHERE {{ .ID }} == ${{ .ID }};

	DELETE FROM {{ .ExecutionsTable }} WHERE {{ .ID }} == ${{ .ID }};
		`,
)

func (s *Storage) ArchiveExecution(ctx context.Context, id consts.ExecutionID) error {
	query := ArchiveExecutionTemplate.Query()
	return s.client.Write(
		ctx,
		query,
		ydbTable.ValueParam("$"+GenericTemplateOptions.ID, ydbTypes.StringValueFromString(id.String())),
	)
}

//goland:noinspection SqlNoDataSourceInspection
var ListExecutionsByTaskletTemplate = template.Must(
	template.New("let").Parse(
		// language=SQL
		`
		DECLARE ${{ .TaskletID }} AS String;
		DECLARE ${{ .ExecutionMonotonicID }} AS Int64;  -- Start from this revision


		SELECT
			a.{{ .TaskletID }} AS {{ .TaskletID }},
		  	a.{{ .ExecutionMonotonicID  }} AS {{ .ExecutionMonotonicID }},
		  	a.{{ .P }} AS {{ .P }},
			b.{{ .Input }} AS {{ .Input }},
			b.{{ .Output }} AS {{ .Output }}
		FROM {{ .ExecutionsTable }} VIEW {{ .TaskletIDExecutionIDView }} AS a
		JOIN {{ .ExecutionBlobsTable }} AS b
		ON a.{{ .ID }} = b.{{ .ID }}
		WHERE
			{{ .TaskletID }} == ${{ .TaskletID }} AND
			{{ .ExecutionMonotonicID }} < ${{ .ExecutionMonotonicID }}

		UNION ALL

		SELECT
			a.{{ .TaskletID }} AS {{ .TaskletID }},
		  	a.{{ .ExecutionMonotonicID  }} AS {{ .ExecutionMonotonicID }},
			a.{{ .P }} AS {{ .P }},
			b.{{ .Input }} AS {{ .Input }},
			b.{{ .Output }} AS {{ .Output }}
		FROM {{ .ExecutionsArchiveTable }} VIEW {{ .TaskletIDExecutionIDView }} AS a
		JOIN {{ .ExecutionBlobsTable }} AS b
		ON a.{{ .ID }} = b.{{ .ID }}
		WHERE
			{{ .TaskletID }} == ${{ .TaskletID }} AND
			{{ .ExecutionMonotonicID }} < ${{ .ExecutionMonotonicID }}
		ORDER BY {{ .TaskletID }}, {{ .ExecutionMonotonicID }} DESC
		LIMIT 10
		;
		`,
	),
)

func (s *Storage) ListExecutionsByTasklet(
	ctx context.Context,
	taskletID string,
	pageToken int64,
) (common.ExecutionsList, error) {
	query := s.client.QueryPrefix() + Tables[ExecutionsTable].Queries[ListExecutionsByTaskletQuery]
	s.logQuery(ctx, "list_executions_by_tasklet", query)
	response := make([]*taskletApi.Execution, 0)

	if pageToken <= 0 {
		pageToken = math.MaxInt64
	}
	queryParameters := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+GenericTemplateOptions.TaskletID, ydbTypes.StringValueFromString(taskletID)),
		ydbTable.ValueParam("$"+GenericTemplateOptions.ExecutionMonotonicID, ydbTypes.Int64Value(pageToken)),
	)
	queryErr := s.tableDo(
		ctx, func(c context.Context, session ydbTable.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, queryParameters,
			)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			var monotonicID *int64
			for res.NextResultSet(c, ExecutionMonotonicIDColumn, PayloadColumn) {
				for res.NextRow() {
					buf := executionRowBuffer{}
					if err := res.ScanNamed(
						ydbNamed.Optional(GenericTemplateOptions.ExecutionMonotonicID, &monotonicID),
						ydbNamed.Optional(GenericTemplateOptions.P, &buf.payload),
						ydbNamed.Optional(GenericTemplateOptions.Input, &buf.input),
						ydbNamed.Optional(GenericTemplateOptions.Output, &buf.output),
					); err != nil {
						return err
					}

					if *monotonicID < pageToken {
						pageToken = *monotonicID
					}
					if item, err := buf.toExecution(); err != nil {
						return err
					} else {
						response = append(response, item)
					}
				}
			}
			return res.Err()
		},
	)

	if queryErr != nil {
		return common.ExecutionsList{}, queryErr
	}
	return common.ExecutionsList{
		Executions: response,
		Token:      pageToken,
	}, nil
}

//goland:noinspection SqlNoDataSourceInspection
var ListExecutionsByBuildTemplate = template.Must(
	template.New("let").Parse(
		// language=SQL
		`
		DECLARE ${{ .BuildID }} AS String;
		DECLARE ${{ .ExecutionMonotonicID }} AS Int64;  -- Start from this revision

		SELECT
			a.{{ .BuildID }} AS {{ .BuildID }},
			a.{{ .ExecutionMonotonicID  }} AS {{ .ExecutionMonotonicID }},
		  	a.{{ .P }}  AS {{ .P }},
			b.{{ .Input }} AS {{ .Input }},
			b.{{ .Output }} AS {{ .Output }}
		FROM {{ .ExecutionsTable }} VIEW {{ .BuildIDExecutionIDView }} AS a
		JOIN {{ .ExecutionBlobsTable }} AS b
		ON a.{{ .ID }} = b.{{ .ID }}
		WHERE
			{{ .BuildID }} == ${{ .BuildID }} AND
			{{ .ExecutionMonotonicID }} < ${{ .ExecutionMonotonicID }}

		UNION ALL

		SELECT
			a.{{ .BuildID }} AS {{ .BuildID }},
			a.{{ .ExecutionMonotonicID  }} AS {{ .ExecutionMonotonicID }},
		  	a.{{ .P }}  AS {{ .P }},
			b.{{ .Input }} AS {{ .Input }},
			b.{{ .Output }} AS {{ .Output }}
		FROM {{ .ExecutionsArchiveTable }} VIEW {{ .BuildIDExecutionIDView }} AS a
		JOIN {{ .ExecutionBlobsTable }} AS b
		ON a.{{ .ID }} = b.{{ .ID }}
		WHERE
			{{ .BuildID }} == ${{ .BuildID }} AND
			{{ .ExecutionMonotonicID }} < ${{ .ExecutionMonotonicID }}

		ORDER BY {{ .BuildID }}, {{ .ExecutionMonotonicID }} DESC
		LIMIT 10
		;
		`,
	),
)

func (s *Storage) ListExecutionsByBuild(ctx context.Context, buildID string, pageToken int64) (
	common.ExecutionsList,
	error,
) {
	query := s.client.QueryPrefix() + Tables[ExecutionsTable].Queries[ListExecutionsByBuildQuery]
	s.logQuery(ctx, "list_executions_by_build", query)
	response := make([]*taskletApi.Execution, 0)

	if pageToken <= 0 {
		pageToken = math.MaxInt64
	}

	queryParameters := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+GenericTemplateOptions.BuildID, ydbTypes.StringValueFromString(buildID)),
		ydbTable.ValueParam("$"+GenericTemplateOptions.ExecutionMonotonicID, ydbTypes.Int64Value(pageToken)),
	)
	queryErr := s.tableDo(
		ctx, func(c context.Context, session ydbTable.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, queryParameters,
			)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			var monotonicID *int64
			for res.NextResultSet(c, ExecutionMonotonicIDColumn, PayloadColumn) {
				for res.NextRow() {
					buf := executionRowBuffer{}
					if err := res.ScanNamed(
						ydbNamed.Optional(GenericTemplateOptions.ExecutionMonotonicID, &monotonicID),
						ydbNamed.Optional(GenericTemplateOptions.P, &buf.payload),
						ydbNamed.Optional(GenericTemplateOptions.Input, &buf.input),
						ydbNamed.Optional(GenericTemplateOptions.Output, &buf.output),
					); err != nil {
						return err
					}
					if *monotonicID < pageToken {
						pageToken = *monotonicID
					}
					if item, err := buf.toExecution(); err != nil {
						return err
					} else {
						response = append(response, item)
					}
				}
			}
			return res.Err()
		},
	)

	if queryErr != nil {
		return common.ExecutionsList{}, queryErr
	}
	return common.ExecutionsList{
		Executions: response,
		Token:      pageToken,
	}, nil

}

//goland:noinspection SqlNoDataSourceInspection
var ListActiveExecutionsTemplate = template.Must(
	template.New("laet").Parse(
		// language=SQL
		`
		SELECT
			a.{{ .ID }} AS {{ .ID }},
		  	a.{{ .P }} AS {{ .P }},
			b.{{ .Input }} AS {{ .Input }},
			b.{{ .Output }} AS {{ .Output }}
		FROM {{ .ExecutionsTable }} AS a
		JOIN {{ .ExecutionBlobsTable }} AS b
		ON a.{{ .ID }} = b.{{ .ID }}
		ORDER BY {{ .ID }} DESC
		LIMIT 1000
		;
		`,
	),
)

func (s *Storage) ListActiveExecutions(ctx context.Context) ([]*taskletApi.Execution, error) {
	query := s.client.QueryPrefix() + render(ListActiveExecutionsTemplate, GenericTemplateOptions)
	s.logQuery(ctx, "list_active_executions", query)
	response := make([]*taskletApi.Execution, 0)

	queryParameters := ydbTable.NewQueryParameters()

	queryErr := s.tableDo(
		ctx, func(c context.Context, session ydbTable.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, queryParameters,
			)
			if err != nil {
				ctxlog.Infof(c, s.l, "Query result. err: %+v", err)
				return err
			}
			defer func() {
				_ = res.Close()
			}()
			ctxlog.Infof(
				ctx,
				s.l,
				"Got response. SetCount: %v, CurrentRowCount: %v",
				res.ResultSetCount(),
				res.CurrentResultSet().RowCount(),
			)
			for res.NextResultSet(ctx, PayloadColumn) {
				for res.NextRow() {
					buf := executionRowBuffer{}
					if err := res.ScanNamed(
						ydbNamed.Optional(GenericTemplateOptions.P, &buf.payload),
						ydbNamed.Optional(GenericTemplateOptions.Input, &buf.input),
						ydbNamed.Optional(GenericTemplateOptions.Output, &buf.output),
					); err != nil {
						return err
					}
					if item, err := buf.toExecution(); err != nil {
						return err
					} else {
						response = append(response, item)
					}
				}
			}
			return res.Err()
		},
	)

	if queryErr != nil {
		return nil, queryErr
	}
	return response, nil
}

var (
	ExecutionMonotonicKey int64 = 1
)

func initExecution() {
	// cross queries
	ArchiveExecutionTemplate.MustRender(GenericTemplateOptions)
	{
		em := xydb.TableSchema{
			Name: ExecutionsMonotonicTable,
			Columns: []xydb.YdbColumn{
				{
					Name:       IDColumn,
					ValueType:  ydbTypes.Optional(ydbTypes.TypeInt64),
					PrimaryKey: true,
				},
				{
					Name:      ValueColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeInt64),
				},
			},
			SecondaryIndexes: make([]xydb.SecondaryIndex, 0),
			Queries:          make(map[xydb.Query]string),
		}
		Tables[em.Name] = em
	}
	{
		requestIDToExecutionTable := xydb.TableSchema{
			Name: ExecutionJournalTable,
			Columns: []xydb.YdbColumn{
				{
					Name:       RequestIDColumn,
					ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
					PrimaryKey: true,
				},
				{
					Name:      OperationColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
				{
					Name:      PayloadColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
				{
					Name:      TTLColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeTimestamp),
				},
			},
			SecondaryIndexes: make([]xydb.SecondaryIndex, 0),
			Queries:          make(map[xydb.Query]string),
			TTLSettings: &ydbOptions.TimeToLiveSettings{
				ColumnName: TTLColumn, ExpireAfterSeconds: uint32((24 * time.Hour).Seconds()),
			},
		}
		Tables[requestIDToExecutionTable.Name] = requestIDToExecutionTable
	}
	{
		e := xydb.TableSchema{
			Name: ExecutionsTable,
			Columns: []xydb.YdbColumn{
				{
					Name:       IDColumn,
					ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
					PrimaryKey: true,
				},
				{
					Name:      TaskletIDColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
				{
					Name:      BuildIDColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
				{
					Name:      ExecutionMonotonicIDColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeInt64),
				},
				{
					Name:      PayloadColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeJSONDocument),
				},
			},
			SecondaryIndexes: []xydb.SecondaryIndex{TaskletIDExecutionIDIndex, BuildIDExecutionIDIndex},
			Queries:          make(map[xydb.Query]string),
		}
		thisOpts := GenericTemplateOptions
		thisOpts.Table = e.Name
		InsertExecutionQuery.MustRender(thisOpts)
		InsertExecutionToJournal.MustRender(thisOpts)
		LookupExecutionByRequestID.MustRender(thisOpts)
		SelectExecutionByIDTemplate.MustRender(thisOpts)
		UpdateExecutionTemplate.MustRender(GenericTemplateOptions)

		e.Queries[ListExecutionsByTaskletQuery] = render(ListExecutionsByTaskletTemplate, thisOpts)
		e.Queries[ListExecutionsByBuildQuery] = render(ListExecutionsByBuildTemplate, thisOpts)
		Tables[e.Name] = e
	}
	{
		ea := xydb.TableSchema{
			Name: ExecutionsArchiveTable,
			Columns: []xydb.YdbColumn{
				{
					Name:       IDColumn,
					ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
					PrimaryKey: true,
				},
				{
					Name:      TaskletIDColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
				{
					Name:      BuildIDColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
				{
					Name:      ExecutionMonotonicIDColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeInt64),
				},
				{
					Name:      PayloadColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeJSONDocument),
				},
			},
			SecondaryIndexes: []xydb.SecondaryIndex{TaskletIDExecutionIDIndex, BuildIDExecutionIDIndex},
			Queries:          make(map[xydb.Query]string),
		}
		thisOpts := GenericTemplateOptions
		thisOpts.Table = ea.Name
		SelectExecutionArchiveByIDTemplate.MustRender(GenericTemplateOptions)
		// FAIL: IO already in place
		ea.Queries[ListExecutionsByTaskletQuery] = render(ListExecutionsByTaskletTemplate, thisOpts)
		ea.Queries[ListExecutionsByBuildQuery] = render(ListExecutionsByBuildTemplate, thisOpts)
		Tables[ea.Name] = ea
	}
	{
		blobs := xydb.TableSchema{
			Name: ExecutionBlobsTable,
			Columns: []xydb.YdbColumn{
				{
					Name:       IDColumn,
					ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
					PrimaryKey: true,
				},
				{
					Name:      InputColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
				{
					Name:      OutputColumn,
					ValueType: ydbTypes.Optional(ydbTypes.TypeString),
				},
			},
			SecondaryIndexes: []xydb.SecondaryIndex{},
			Queries:          make(map[xydb.Query]string),
		}
		Tables[blobs.Name] = blobs
	}
}
