package ydbstore

import (
	"context"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/durationpb"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/test/requirepb"
	taskletApi "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

func Test_executionRowBuffer(t *testing.T) {
	gen := testutils.NewObjectGenerator()
	ns := gen.NewNamespace("foo")
	tl := gen.NewTasklet("bar", ns)
	build := gen.NewBuild(tl)

	doCheck := func(tt *testing.T, ex *taskletApi.Execution) {
		buf, err := fromExecution(ex)
		require.NoError(t, err)
		out, err := buf.toExecution()
		require.NoError(t, err)
		requirepb.Equal(t, ex, out)
	}

	t.Run(
		"Full", func(tt *testing.T) {
			ex := gen.NewExecution(build)

			if ex.Status == nil {
				ex.Status = &taskletApi.ExecutionStatus{}
			}
			ex.Status.ProcessingResult = &taskletApi.ProcessingResult{}
			ex.Status.Stats = &taskletApi.ExecutionStats{ExitCode: 1543}

			{
				dummyInput := &taskletApi.SecretRef{Id: "sec-123"}
				serializedDummyInput, err := proto.Marshal(dummyInput)
				require.NoError(tt, err)
				ex.Spec.Input.SerializedData = serializedDummyInput
			}
			{
				dummyOutput := &taskletApi.SecretRef{Version: "ver-123"}
				serializedDummyOutput, err := proto.Marshal(dummyOutput)
				require.NoError(tt, err)
				ex.Status.ProcessingResult.Kind = &taskletApi.ProcessingResult_Output{
					Output: &taskletApi.ExecutionOutput{
						SerializedOutput: serializedDummyOutput,
					},
				}
			}
			doCheck(tt, ex)
		},
	)

	t.Run(
		"EmptyResult", func(tt *testing.T) {
			ex := gen.NewExecution(build)

			if ex.Status == nil {
				ex.Status = &taskletApi.ExecutionStatus{}
			}

			{
				dummyInput := &taskletApi.SecretRef{Id: "sec-123"}
				serializedDummyInput, err := proto.Marshal(dummyInput)
				require.NoError(tt, err)
				ex.Spec.Input.SerializedData = serializedDummyInput
			}
			doCheck(tt, ex)
		},
	)

	t.Run(
		"EmptyInput", func(tt *testing.T) {
			ex := gen.NewExecution(build)

			if ex.Status == nil {
				ex.Status = &taskletApi.ExecutionStatus{}
			}
			ex.Spec.Input = nil
			doCheck(tt, ex)
		},
	)
	t.Run(
		"WithError", func(tt *testing.T) {
			ex := gen.NewExecution(build)

			if ex.Status == nil {
				ex.Status = &taskletApi.ExecutionStatus{}
			}
			{
				dummyInput := &taskletApi.SecretRef{Id: "sec-123"}
				serializedDummyInput, err := proto.Marshal(dummyInput)
				require.NoError(tt, err)
				ex.Spec.Input.SerializedData = serializedDummyInput
			}
			ex.Status.ProcessingResult = &taskletApi.ProcessingResult{}
			ex.Status.ProcessingResult.Kind = &taskletApi.ProcessingResult_ServerError{
				ServerError: &taskletApi.ServerError{
					Code:        17,
					Description: "bla-bla",
				},
			}
			doCheck(tt, ex)
		},
	)

}

type ExecutionsSuite struct {
	suite.Suite
	suiteClient *xydb.Client
	testStorage *Storage
	root        string
	tmpdir      string
}

func (es *ExecutionsSuite) SetupSuite() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.suiteClient = xydb.MustGetYdbClient(ctx, nil, es.T().Name())
	es.root = es.suiteClient.Folder
	es.tmpdir = testutils.TwistTmpDir(es.T())
}

func (es *ExecutionsSuite) SetupTest() {
	logger := testutils.TwistMakeLogger(es.tmpdir, strings.ReplaceAll(es.T().Name(), "/", "_")+".log")
	es.suiteClient.Folder = es.root + "/" + es.T().Name()
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.NoError(PurgeDatabase(ctx, es.suiteClient))
	es.NoError(CreateTables(ctx, es.suiteClient))
	es.testStorage = NewStorage(es.suiteClient, logger)
	es.testStorage.LogQueries = true
}

func (es *ExecutionsSuite) TearDownSuite() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.NoError(es.suiteClient.Close(ctx))
}

func (es *ExecutionsSuite) bootstrapContext(ctx context.Context) (
	*testutils.ObjectGenerator,
	*taskletApi.Namespace,
	*taskletApi.Tasklet,
	*taskletApi.Build,
) {
	r := es.Require()
	og := testutils.NewObjectGenerator()
	ns := og.NewNamespace("test_ns")
	r.NoError(es.testStorage.AddNamespace(ctx, ns))

	tl := og.NewTasklet("test_tl", ns)
	r.NoError(es.testStorage.AddTasklet(ctx, tl))

	b := og.NewBuild(tl)
	if lastBuild, err := es.testStorage.GetLastBuild(ctx, tl.Meta.Id); err != nil {
		if !xerrors.Is(err, common.ErrObjectNotFound) { // NB: First build
			r.NoError(err)
		}
		b.Meta.Revision = 1
	} else {
		b.Meta.Revision = lastBuild.GetMeta().GetRevision() + 1
	}

	r.NoError(es.testStorage.AddBuild(ctx, b))
	return og, ns, tl, b
}

func (es *ExecutionsSuite) TestAddGet() {
	r := es.Require()
	ctx := context.Background()
	og, _, _, b := es.bootstrapContext(ctx)
	ex := og.NewExecution(b)
	ex.Status = &taskletApi.ExecutionStatus{Status: taskletApi.EExecutionStatus_E_EXECUTION_STATUS_EXECUTING}
	requestID := consts.NewRequestID()
	exCreateOut, err := es.testStorage.AddExecution(ctx, requestID, ex)
	r.NoError(err)
	requirepb.Equal(es.T(), ex, exCreateOut)
	exGetOut, err := es.testStorage.GetExecution(ctx, ex.Meta.Id)
	r.NoError(err)
	requirepb.Equal(es.T(), ex, exGetOut)
}

func (es *ExecutionsSuite) TestCreateJournal() {
	r := es.Require()
	ctx := context.Background()
	og, _, _, b := es.bootstrapContext(ctx)
	ex := og.NewExecution(b)
	ex.Status = &taskletApi.ExecutionStatus{Status: taskletApi.EExecutionStatus_E_EXECUTION_STATUS_EXECUTING}
	requestID := consts.NewRequestID()
	exOnCreate, err := es.testStorage.AddExecution(ctx, requestID, ex)
	r.NoError(err)
	requirepb.Equal(es.T(), ex, exOnCreate)
	{
		exOut, err := es.testStorage.AddExecution(ctx, requestID, ex)
		r.NoError(err)
		requirepb.Equal(es.T(), exOnCreate, exOut)
	}
	// NB: mutate stored execution
	{
		op := func(s *taskletApi.ExecutionStatus) error {
			s.Stats = &taskletApi.ExecutionStats{ExitCode: 1543, Duration: durationpb.New(time.Second * 3)}
			return nil
		}
		r.NoError(op(ex.Status))
		_, err := es.testStorage.UpdateExecutionStatus(ctx, consts.ExecutionID(ex.Meta.Id), op)
		r.NoError(err)
	}
	{
		ex2 := og.NewExecution(b)
		ex2.Status = &taskletApi.ExecutionStatus{Status: taskletApi.EExecutionStatus_E_EXECUTION_STATUS_EXECUTING}
		exOut, err := es.testStorage.AddExecution(ctx, requestID, ex2)
		r.NoError(err)
		requirepb.Equal(es.T(), exOnCreate, exOut)
	}
	{
		op := func(s *taskletApi.ExecutionStatus) error {
			s.Status = taskletApi.EExecutionStatus_E_EXECUTION_STATUS_FINISHED
			return nil
		}
		_, err := es.testStorage.UpdateExecutionStatus(ctx, consts.ExecutionID(ex.Meta.Id), op)
		r.NoError(err)
		r.NoError(op(ex.Status))
		r.NoError(es.testStorage.ArchiveExecution(ctx, consts.ExecutionID(ex.Meta.Id)))
		exOut, err := es.testStorage.AddExecution(ctx, requestID, ex)
		r.NoError(err)
		requirepb.Equal(es.T(), exOnCreate, exOut)
	}

	// Get
	{
		exOut, err := es.testStorage.GetExecution(ctx, ex.Meta.Id)
		r.NoError(err)
		requirepb.Equal(es.T(), ex, exOut)
	}

}

func (es *ExecutionsSuite) TestList() {
	r := es.Require()
	ctx := context.Background()
	og, _, tl, b := es.bootstrapContext(ctx)

	var executions []*taskletApi.Execution
	const executionCount = 17
	for i := 0; i < executionCount; i++ {
		ex := og.NewExecution(b)
		ex.Status = &taskletApi.ExecutionStatus{
			Status: taskletApi.EExecutionStatus_E_EXECUTION_STATUS_EXECUTING,
		}

		ex, err := es.testStorage.AddExecution(ctx, consts.NewRequestID(), ex)
		r.NoError(err)
		executions = append(executions, ex)
	}

	executionsByTasklet, err := es.testStorage.ListExecutionsByTasklet(ctx, tl.Meta.Id, 0)
	r.NoError(err)
	for executionsByTasklet.Token > 1 {
		next, err := es.testStorage.ListExecutionsByTasklet(ctx, tl.Meta.Id, executionsByTasklet.Token)
		r.NoError(err)
		executionsByTasklet.Executions = append(executionsByTasklet.Executions, next.Executions...)
		executionsByTasklet.Token = next.Token
	}
	executionsByBuild, err := es.testStorage.ListExecutionsByBuild(ctx, b.Meta.Id, 0)
	r.NoError(err)
	for executionsByBuild.Token > 1 {
		next, err := es.testStorage.ListExecutionsByTasklet(ctx, tl.Meta.Id, executionsByBuild.Token)
		r.NoError(err)
		executionsByBuild.Executions = append(executionsByBuild.Executions, next.Executions...)
		executionsByBuild.Token = next.Token
	}

	for _, list := range []common.ExecutionsList{executionsByTasklet, executionsByBuild} {
		r.Equal(executionCount, len(list.Executions))
		for i := 0; i < executionCount; i++ {
			// NB: query returns in monotonic counter descending order
			requirepb.Equal(es.T(), executions[len(executions)-i-1], list.Executions[i])
		}
	}
}

func (es *ExecutionsSuite) TestUpdate() {
	r := es.Require()
	ctx := context.Background()
	og, _, _, b := es.bootstrapContext(ctx)

	ex := og.NewExecution(b)
	ex.Status = &taskletApi.ExecutionStatus{
		Status: taskletApi.EExecutionStatus_E_EXECUTION_STATUS_EXECUTING,
		Stats:  &taskletApi.ExecutionStats{},
	}

	ex, err := es.testStorage.AddExecution(ctx, consts.NewRequestID(), ex)
	r.NoError(err)
	executionID := consts.ExecutionID(ex.Meta.Id)

	updates := []common.ExecutionStatusUpdateFunc{
		func(st *taskletApi.ExecutionStatus) error {
			st.Stats.ExitCode = 17
			return nil
		},
	}
	exOut, err := es.testStorage.UpdateExecutionStatus(ctx, executionID, updates...)
	r.NoError(err)
	for _, op := range updates {
		r.NoError(op(ex.Status))
	}
	requirepb.Equal(es.T(), ex, exOut)
}

func TestExecutions(t *testing.T) {
	s := &ExecutionsSuite{}
	suite.Run(t, s)
}
