package ydbstore

import (
	"context"
	"text/template"

	"github.com/ydb-platform/ydb-go-sdk/v3/table"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/types"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

//goland:noinspection SqlNoDataSourceInspection
var InsertLabelTemplate = template.Must(
	template.New("ilt").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .TaskletID }} AS String;
		DECLARE ${{ .P }} AS JsonDocument;

		-- ensure no such label
		DISCARD SELECT
			Ensure(
				0,
				${{ .Name }} == "INVALID_c1210258-275d-11ec-96a1-4bbeec6515f9",
				"Object exists. ID: '" || Unwrap({{ .ID }}) || "'"
			)
		FROM {{ .Table }} VIEW {{ .TaskletIDNameView }}
		WHERE
			{{ .TaskletID }} == ${{ .TaskletID }} AND
			{{ .Name }} == ${{ .Name }}
		;

		INSERT INTO {{ .Table }} ({{ .ID }}, {{ .Name }}, {{ .TaskletID }}, {{ .P }})
		VALUES (${{ .ID }}, ${{ .Name }}, ${{ .TaskletID }}, ${{ .P }});
		`,
	),
)

func (s *Storage) AddLabel(ctx context.Context, l *taskletv2.Label) error {
	serialized, err := protojson.Marshal(l)
	if err != nil {
		ctxlog.Infof(ctx, s.l, "Marshall failed: %v", err)
		return err
	}

	query := s.client.QueryPrefix() + Tables[LabelsTable].Queries[InsertQuery]
	ctxlog.Infof(ctx, s.l, "Query: %v", query)
	meta := l.GetMeta()

	errQ := s.client.ExecuteWriteQuery(
		ctx,
		query,
		table.ValueParam("$"+IDColumn, types.StringValueFromString(meta.Id)),
		table.ValueParam("$"+NameColumn, types.StringValueFromString(meta.Name)),
		table.ValueParam("$"+TaskletIDColumn, types.StringValueFromString(meta.TaskletId)),
		table.ValueParam("$"+PayloadColumn, types.JSONDocumentValueFromBytes(serialized)),
	)
	if errQ != nil {
		return errQ
	}
	return nil
}

//goland:noinspection SqlNoDataSourceInspection
var SelectLabelByNameTemplate = template.Must(
	template.New("slt").Parse(
		// language=SQL
		`
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .TaskletID }} AS String;
		SELECT
			{{ .P }}
		FROM
			{{ .Table }} VIEW {{ .TaskletIDNameView }}
		WHERE
			{{ .TaskletID }} == ${{ .TaskletID }} AND
			{{ .Name }} == ${{ .Name }}
		;
		`,
	),
)

func (s *Storage) GetLabelByName(ctx context.Context, taskletID string, name string) (*taskletv2.Label, error) {
	query := s.client.QueryPrefix() + Tables[LabelsTable].Queries[SelectByNameQuery]
	params := table.NewQueryParameters(
		table.ValueParam("$"+TaskletIDColumn, types.StringValueFromString(taskletID)),
		table.ValueParam("$"+NameColumn, types.StringValueFromString(name)),
	)
	ctxlog.Infof(ctx, s.l, "Query: %v", query)
	response := &taskletv2.Label{}
	queryErr := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, params,
			)
			ctxlog.Infof(ctx, s.l, "QRES: r: %v, err: %v", res, err)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			itemsProcessed := 0
			for res.NextResultSet(ctx, PayloadColumn) {
				for res.NextRow() {
					if itemsProcessed > 0 {
						ctxlog.Fatalf(
							ctx,
							s.l,
							"Multiple results for select label query. Name: %q, TaskletId: %q",
							name,
							taskletID,
						)
					}
					var payload *[]byte
					if err := res.Scan(&payload); err != nil {
						return err
					}
					if err := protojson.Unmarshal(*payload, response); err != nil {
						return err
					}
					itemsProcessed += 1
				}
			}
			if itemsProcessed == 0 {
				return common.ErrObjectNotFound
			}
			return res.Err()
		},
	)

	if queryErr != nil {
		return nil, queryErr
	}
	return response, nil
}

//goland:noinspection SqlNoDataSourceInspection
var ListLabelsTemplate = template.Must(
	template.New("llt").Parse(
		// language=SQL
		`
		DECLARE ${{ .TaskletID }} AS String;

		SELECT
			{{ .P }}
		FROM {{ .Table }} VIEW {{ .TaskletIDNameView }}
		WHERE
			{{ .TaskletID }} == ${{ .TaskletID }}
		;
		`,
	),
)

func (s *Storage) ListLabels(ctx context.Context, taskletID string) ([]*taskletv2.Label, error) {
	query := s.client.QueryPrefix() + Tables[LabelsTable].Queries[ListQuery]
	ctxlog.Infof(ctx, s.l, "Query: %v", query)
	response := make([]*taskletv2.Label, 0)

	queryErr := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, table.NewQueryParameters(
					table.ValueParam("$"+TaskletIDColumn, types.StringValueFromString(taskletID)),
				),
			)
			ctxlog.Infof(ctx, s.l, "QRES: r: %v, err: %v", res, err)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			for res.NextResultSet(ctx, PayloadColumn) {
				for res.NextRow() {
					var payload *[]byte
					if err := res.Scan(&payload); err != nil {
						return err
					}
					item := &taskletv2.Label{}
					if err := protojson.Unmarshal(*payload, item); err != nil {
						return err
					}
					response = append(response, item)

				}
			}
			return res.Err()
		},
	)

	if queryErr != nil {
		return nil, queryErr
	}
	return response, nil
}

//goland:noinspection SqlNoDataSourceInspection
var UpdateLabelTemplate = template.Must(
	template.New("ult").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .TaskletID }} AS String;
		DECLARE ${{ .P }} AS JsonDocument;

		$getRevision = ($pp) -> {
			RETURN Unwrap(
				JSON_VALUE(
					$pp,
					"$.spec.revision"
					RETURNING Uint32
					DEFAULT 0 ON EMPTY
					ERROR ON ERROR
				)
			);
		};

		DISCARD SELECT
			Ensure(
				0,
				$getRevision(${{ .P }})  == $getRevision({{ .P }}) + 1,
				"Invalid revision. Object ID: '" || Unwrap({{ .ID }}) || "', Revision: " || CAST(Unwrap($getRevision({{ .P }})) AS String)
			)
		FROM {{ .Table }}
		WHERE
			{{ .ID }} == ${{ .ID }}
		;

		UPSERT INTO {{ .Table }} ({{ .ID }}, {{ .Name }}, {{ .TaskletID }}, {{ .P }})
		VALUES (${{ .ID }}, ${{ .Name }}, ${{ .TaskletID }}, ${{ .P }});
		`,
	),
)

func (s *Storage) UpdateLabel(ctx context.Context, l *taskletv2.Label) error {
	serialized, err := protojson.Marshal(l)
	if err != nil {
		ctxlog.Infof(ctx, s.l, "Marshall failed: %v", err)
		return err
	}

	query := s.client.QueryPrefix() + Tables[LabelsTable].Queries[UpdateQuery]
	ctxlog.Infof(ctx, s.l, "Query: %v", query)
	meta := l.GetMeta()

	errQ := s.client.ExecuteWriteQuery(
		ctx,
		query,
		table.ValueParam("$"+IDColumn, types.StringValueFromString(meta.Id)),
		table.ValueParam("$"+NameColumn, types.StringValueFromString(meta.Name)),
		table.ValueParam("$"+TaskletIDColumn, types.StringValueFromString(meta.TaskletId)),
		table.ValueParam("$"+PayloadColumn, types.JSONDocumentValueFromBytes(serialized)),
	)
	if errQ != nil {
		return errQ
	}
	return nil
}

func initLabels() {
	l := xydb.TableSchema{
		Name: LabelsTable,
		Columns: []xydb.YdbColumn{
			{
				Name:       IDColumn,
				ValueType:  types.Optional(types.TypeString),
				PrimaryKey: true,
			},
			{
				Name:      NameColumn,
				ValueType: types.Optional(types.TypeString),
			},
			{
				Name:      TaskletIDColumn,
				ValueType: types.Optional(types.TypeString),
			},
			{
				Name:      PayloadColumn,
				ValueType: types.Optional(types.TypeJSONDocument),
			},
		},
		SecondaryIndexes: []xydb.SecondaryIndex{TaskletIDNameIndex},
		Queries:          make(map[xydb.Query]string),
	}
	thisOpts := GenericTemplateOptions
	thisOpts.Table = l.Name
	l.Queries[SelectByNameQuery] = render(SelectLabelByNameTemplate, thisOpts)
	l.Queries[InsertQuery] = render(InsertLabelTemplate, thisOpts)
	l.Queries[UpdateQuery] = render(UpdateLabelTemplate, thisOpts)
	l.Queries[ListQuery] = render(ListLabelsTemplate, thisOpts)
	Tables[l.Name] = l
}
