package ydbstore

import (
	"context"
	"text/template"

	"github.com/ydb-platform/ydb-go-sdk/v3"
	"github.com/ydb-platform/ydb-go-sdk/v3/table"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	ydbNamed "github.com/ydb-platform/ydb-go-sdk/v3/table/result/named"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/types"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

//goland:noinspection SqlNoDataSourceInspection
var InsertNamespaceTemplate = template.Must(
	template.New("int").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .P }} AS JsonDocument;

		INSERT INTO {{ .NamespaceTable }} ({{ .ID }}, {{ .Name }}, {{ .P }})
		VALUES (${{ .ID }}, ${{ .Name }}, ${{ .P }})
		;
		`,
	),
)

var UpdateNamespaceTemplate = xydb.NewQuery(
	"update_namespace",
	// language=YQL
	`
	DECLARE ${{ .ID }} AS String;
	DECLARE ${{ .P }} AS JsonDocument;

	UPDATE {{ .NamespaceTable }}
	SET {{ .P }} = ${{ .P }}
	WHERE {{ .ID }} == ${{ .ID }}
	;
`,
)

//goland:noinspection SqlNoDataSourceInspection
var EnsureNoNamespaceTemplate = template.Must(
	template.New("ennt").Parse(
		// language=SQL
		`
		DECLARE ${{ .Name }} AS String;

		DISCARD SELECT
			Ensure(
				0,
				${{ .Name }} == "INVALID_NAMESPACE_c1210258-275d-11ec-96a1-4bbeec6515f9",
				"Object exists. ID: '" || Unwrap({{ .ID }}, "NULL") || "'"
			)
		FROM {{ .NamespaceTable }} VIEW {{ .NameView }}
		WHERE
			{{ .Name }} == ${{ .Name }}
		;
		`,
	),
)

func (s *Storage) AddNamespace(ctx context.Context, ns *taskletv2.Namespace) error {

	preconditionQ := s.client.QueryPrefix() + Tables[NamespaceTable].Queries[EnsureMissing]
	s.logQuery(ctx, "addNamespacePrecondition", preconditionQ)

	updateQ := s.client.QueryPrefix() + Tables[NamespaceTable].Queries[InsertQuery]
	s.logQuery(ctx, "addNamespace", updateQ)

	serialized, err := protojson.Marshal(ns)
	if err != nil {
		ctxlog.Infof(ctx, s.l, "Marshall failed: %v", err)
		return err
	}
	queryErr := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			txControl := table.TxControl(table.BeginTx(table.WithSerializableReadWrite()))
			tx, _, err := session.Execute(
				c, txControl, preconditionQ, table.NewQueryParameters(
					table.ValueParam("$"+NameColumn, types.StringValueFromString(ns.Meta.Name)),
				),
			)
			ctxlog.Infof(c, s.l, "Execute1 result. txOk: %v, err: %+v", tx != nil, err)
			defer func() {
				if tx != nil {
					_ = tx.Rollback(c)
				}
			}()

			if err != nil {
				if oe := ydb.OperationError(err); oe != nil {
					ctxlog.Warnf(c, s.l, "Error descriptor: code: %v, name: %q", oe.Code(), oe.Name())
					return common.ErrObjectExists
				}
				return err
			}
			_, err = tx.Execute(
				c, updateQ, table.NewQueryParameters(
					table.ValueParam("$"+IDColumn, types.StringValueFromString(ns.Meta.Id)),
					table.ValueParam("$"+NameColumn, types.StringValueFromString(ns.Meta.Name)),
					table.ValueParam("$"+PayloadColumn, types.JSONDocumentValueFromBytes(serialized)),
				),
			)
			if err != nil {
				return err
			}
			_, err = tx.CommitTx(c, options.WithCommitCollectStatsModeBasic())
			return err
		},
	)
	return queryErr

}

var SelectNamespaceByNameQuery = xydb.NewQuery(
	"select_namespace_by_name",
	// language=YQL
	`
	DECLARE ${{ .Name }} AS String;
	SELECT
		{{ .P }}
	FROM
		{{ .NamespaceTable }} VIEW {{ .NameView }}
	WHERE
		{{ .Name }} == ${{ .Name }}
	;
	`,
)

func (s *Storage) GetNamespaceByName(ctx context.Context, name string) (*taskletv2.Namespace, error) {
	var payload []byte

	err := s.client.ReadOneRow(
		ctx, SelectNamespaceByNameQuery.Query(), xydb.ReadValues{
			ydbNamed.OptionalWithDefault(PayloadColumn, &payload),
		},
		table.ValueParam("$"+NameColumn, types.StringValueFromString(name)),
	)
	if err != nil {
		if xerrors.Is(err, xydb.ErrNoRows) {
			return nil, common.ErrObjectNotFound
		}
		return nil, xerrors.Errorf("db query failed: %w", err)
	}

	rv := &taskletv2.Namespace{}
	if err := protojson.Unmarshal(payload, rv); err != nil {
		return nil, xerrors.Errorf("payload unmarshall failed: %w", err)
	}
	return rv, nil
}

var SelectNamespaceByIDTemplate = xydb.NewQuery(
	"select_namespace_by_id",
	// language=YQL
	`
			DECLARE ${{ .ID }} AS String;
			SELECT
				{{ .P }}
			FROM
				{{ .NamespaceTable }}
			WHERE
				{{ .ID }} == ${{ .ID }}
			;
			`,
)

func (s *Storage) GetNamespaceByID(ctx context.Context, id string) (*taskletv2.Namespace, error) {
	var payload []byte
	err := s.client.ReadOneRow(
		ctx, SelectNamespaceByIDTemplate.Query(),
		xydb.ReadValues{
			ydbNamed.OptionalWithDefault(PayloadColumn, &payload),
		},
		table.ValueParam("$"+IDColumn, types.StringValueFromString(id)),
	)

	if err != nil {
		if xerrors.Is(err, xydb.ErrNoRows) {
			return nil, common.ErrObjectNotFound
		}
		return nil, xerrors.Errorf("db query failed: %w", err)
	}

	rv := &taskletv2.Namespace{}
	if err := protojson.Unmarshal(payload, rv); err != nil {
		return nil, xerrors.Errorf("payload unmarshall failed: %w", err)
	}
	return rv, nil
}

//goland:noinspection SqlNoDataSourceInspection
var ListNamespaceQueryTemplate = template.Must(
	template.New("lnr").Parse(
		// language=SQL
		`
			DECLARE $owner AS String;
			SELECT
				{{ .C.P }}
			FROM {{ .C.NamespaceTable }}
			WHERE
				{{ if .Q.Owner }}
				JSON_VALUE(
					{{ .C.P }},
					"$.meta.accountId"
					RETURNING String
				) == $owner AND
				{{ end }}
				TRUE
			;
			`,
	),
)

func (s *Storage) getNamespaceTx(ctx context.Context, tx *xydb.TransactionWrapper, id string) (
	*taskletv2.Namespace,
	error,
) {
	var payload []byte
	err := tx.ReadOneRow(
		ctx, SelectNamespaceByIDTemplate.Query(),
		xydb.ReadValues{
			ydbNamed.OptionalWithDefault(PayloadColumn, &payload),
		},
		table.ValueParam("$"+IDColumn, types.StringValueFromString(id)),
	)

	if err != nil {
		if xerrors.Is(err, xydb.ErrNoRows) {
			return nil, common.ErrObjectNotFound
		}
		return nil, xerrors.Errorf("db query failed: %w", err)
	}

	rv := &taskletv2.Namespace{}
	if err := protojson.Unmarshal(payload, rv); err != nil {
		return nil, xerrors.Errorf("payload unmarshall failed: %w", err)
	}
	return rv, nil
}

func (s *Storage) writeNamespaceTx(ctx context.Context, tx *xydb.TransactionWrapper, ns *taskletv2.Namespace) error {
	serialized, err := protojson.Marshal(ns)
	if err != nil {
		return xerrors.Errorf("Marshall failed: %w", err)
	}
	err = tx.Write(
		ctx, UpdateNamespaceTemplate.Query(),
		table.ValueParam("$"+IDColumn, types.StringValueFromString(ns.Meta.Id)),
		table.ValueParam("$"+PayloadColumn, types.JSONDocumentValueFromBytes(serialized)),
	)
	return err
}

func (s *Storage) UpdateNamespace(
	ctx context.Context,
	id string,
	updates ...common.NamespaceUpdateOperation,
) (*taskletv2.Namespace, error) {
	var rv *taskletv2.Namespace
	txErr := s.client.DoTx(
		ctx, func(c context.Context, tx *xydb.TransactionWrapper) error {
			ns, err := s.getNamespaceTx(c, tx, id)
			if err != nil {
				return err
			}
			oldID := ns.Meta.Id
			oldName := ns.Meta.Name

			for _, update := range updates {
				updateErr := update(ns)
				if updateErr != nil {
					return xerrors.Errorf("update document failed: %w", updateErr)
				}
			}
			if ns.Meta.Id != oldID || ns.Meta.Name != oldName {
				panic("immutable fields change")
			}
			rv = ns
			return s.writeNamespaceTx(ctx, tx, ns)
		},
		table.WithTxSettings(
			table.TxSettings(
				table.WithSerializableReadWrite(),
			),
		),
	)
	if txErr != nil {
		rv = nil
	}
	return rv, txErr
}

func (s *Storage) ListNamespaces(ctx context.Context, owner string) ([]*taskletv2.Namespace, error) {
	response := make([]*taskletv2.Namespace, 0)

	query := s.client.QueryPrefix() + render(
		ListNamespaceQueryTemplate, struct {
			C TGenericRenderingOptions
			Q common.ListNamespacesQueryOptions
		}{GenericTemplateOptions, common.ListNamespacesQueryOptions{Owner: owner}},
	)
	ctxlog.Infof(ctx, s.l, "List query: %v", query)
	err := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, table.NewQueryParameters(
					table.ValueParam("$owner", types.StringValueFromString(owner)),
				),
			)
			ctxlog.Infof(c, s.l, "QRES: r: %v, err: %v", res, err)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			for res.NextResultSet(c, PayloadColumn) {
				for res.NextRow() {
					var payload *[]byte = nil
					err := res.Scan(&payload)
					if err != nil {
						return err
					}
					ns := &taskletv2.Namespace{}
					if err := protojson.Unmarshal(*payload, ns); err != nil {
						return err
					}
					response = append(response, ns)
				}
			}
			return res.Err()
		},
	)
	if err != nil {
		return nil, err
	}
	return response, nil
}

func initNamespace() {
	ns := xydb.TableSchema{
		Name: NamespaceTable,
		Columns: []xydb.YdbColumn{
			{
				Name:       IDColumn,
				ValueType:  types.Optional(types.TypeString),
				PrimaryKey: true,
			},
			{
				Name:      NameColumn,
				ValueType: types.Optional(types.TypeString),
			},
			{
				Name:      PayloadColumn,
				ValueType: types.Optional(types.TypeJSONDocument),
			},
		},
		SecondaryIndexes: []xydb.SecondaryIndex{NameIndex},
		Queries:          make(map[xydb.Query]string),
	}
	thisOpts := GenericTemplateOptions
	SelectNamespaceByNameQuery.MustRender(GenericTemplateOptions)
	SelectNamespaceByIDTemplate.MustRender(GenericTemplateOptions)
	UpdateNamespaceTemplate.MustRender(GenericTemplateOptions)
	ns.Queries[EnsureMissing] = render(EnsureNoNamespaceTemplate, thisOpts)
	ns.Queries[InsertQuery] = render(InsertNamespaceTemplate, thisOpts)

	Tables[ns.Name] = ns

}
