package ydbstore

import (
	"context"
	"strings"
	"testing"
	"time"

	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"github.com/gofrs/uuid"
	"github.com/stretchr/testify/suite"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/test/requirepb"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

type NamespaceSuite struct {
	suite.Suite
	suiteClient *xydb.Client
	testStorage *Storage
	root        string
	tmpdir      string
}

func (es *NamespaceSuite) SetupSuite() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.tmpdir = testutils.TwistTmpDir(es.T())
	logger := testutils.TwistMakeLogger(es.tmpdir, strings.ReplaceAll(es.T().Name(), "/", "_")+".log")
	es.suiteClient = xydb.MustGetYdbClient(ctx, logger, es.T().Name())
	es.root = es.suiteClient.Folder
}

func (es *NamespaceSuite) SetupTest() {
	logger := testutils.TwistMakeLogger(es.tmpdir, strings.ReplaceAll(es.T().Name(), "/", "_")+".log")
	es.suiteClient.Folder = es.root + "/" + es.T().Name()
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.NoError(PurgeDatabase(ctx, es.suiteClient))
	es.NoError(CreateTables(ctx, es.suiteClient))
	es.testStorage = NewStorage(es.suiteClient, logger)
	es.suiteClient.SetLogQueries(true)
}

func (es *NamespaceSuite) TearDownSuite() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.NoError(es.suiteClient.Close(ctx))
}

func (es *NamespaceSuite) TestAddGet() {
	r := es.Require()
	ctx := context.Background()

	og := testutils.NewObjectGenerator()
	ns := og.NewNamespace("test_ns")
	r.NoError(es.testStorage.AddNamespace(ctx, ns))
	{
		// NB: get by name
		nsByName, err := es.testStorage.GetNamespaceByName(ctx, ns.Meta.Name)
		r.NoError(err)
		requirepb.Equal(es.T(), ns, nsByName)
	}
	{
		// NB: get by ID
		nsByName, err := es.testStorage.GetNamespaceByID(ctx, ns.Meta.Id)
		r.NoError(err)
		requirepb.Equal(es.T(), ns, nsByName)
	}
	{
		// NB: missing by name
		nsByName, err := es.testStorage.GetNamespaceByName(ctx, "invalid_name")
		r.ErrorIs(err, common.ErrObjectNotFound)
		r.Nil(nsByName)
	}
	{
		// NB: missing by id
		nsByName, err := es.testStorage.GetNamespaceByID(ctx, uuid.Must(uuid.NewV4()).String())
		r.ErrorIs(err, common.ErrObjectNotFound)
		r.Nil(nsByName)
	}
}

func (es *NamespaceSuite) TestUpdate() {
	r := es.Require()
	ctx := context.Background()

	og := testutils.NewObjectGenerator()
	correctNs := og.NewNamespace("test_ns")
	r.NoError(es.testStorage.AddNamespace(ctx, correctNs))

	{
		// NB: update ok
		op := func(ns *taskletv2.Namespace) error {
			ns.Meta.Permissions = &taskletv2.Permissions{
				Subjects: append(
					ns.Meta.Permissions.GetSubjects(),
					&taskletv2.PermissionsSubject{
						Source: taskletv2.PermissionsSubject_E_SOURCE_USER,
						Name:   "gandalf",
						Roles:  []string{string(acmodel.TaskletWrite)},
					},
				),
			}
			return nil
		}
		r.NoError(op(correctNs))
		updatedNs, err := es.testStorage.UpdateNamespace(ctx, correctNs.Meta.Id, op)
		r.NoError(err)
		requirepb.Equal(es.T(), correctNs, updatedNs)

		storedNs, err := es.testStorage.GetNamespaceByID(ctx, correctNs.Meta.Id)
		r.NoError(err)
		requirepb.Equal(es.T(), correctNs, storedNs)
	}

	{
		// NB: update error
		op := func(ns *taskletv2.Namespace) error {
			// spoil document
			ns.Meta.CreatedAt = timestamppb.Now()
			return xerrors.New("update failed")
		}
		updated, err := es.testStorage.UpdateNamespace(ctx, correctNs.Meta.Id, op)
		r.Error(err)
		r.Nil(updated)
		r.ErrorContains(err, "update failed")

		// check not committed
		storedNs, err := es.testStorage.GetNamespaceByID(ctx, correctNs.Meta.Id)
		r.NoError(err)
		requirepb.Equal(es.T(), correctNs, storedNs)
	}

	{
		// NB: update on non existing NS
		op := func(ns *taskletv2.Namespace) error {
			// spoil document
			ns.Meta.CreatedAt = timestamppb.Now()
			return xerrors.New("update failed")
		}
		updated, err := es.testStorage.UpdateNamespace(ctx, uuid.Must(uuid.NewV4()).String(), op)
		r.ErrorIs(err, common.ErrObjectNotFound)
		r.Nil(updated)
	}

}

func TestNamespaces(t *testing.T) {
	s := &NamespaceSuite{}
	suite.Run(t, s)
}
