package ydbstore

import (
	"bytes"
	"context"
	"fmt"
	"text/template"

	"github.com/ydb-platform/ydb-go-genproto/protos/Ydb"
	"github.com/ydb-platform/ydb-go-sdk/v3"

	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/tasklet/experimental/internal/ydbmigrate"
)

const SchemaVersion ydbmigrate.SchemaVersion = 3

var (
	IDColumn                   = "id"
	NameColumn                 = "name"
	NamespaceColumn            = "namespace"
	NamespaceIDColumn          = "namespace_id"
	TaskletColumn              = "tasklet"
	TaskletIDColumn            = "tasklet_id"
	BuildIDColumn              = "build_id"
	ExecutionMonotonicIDColumn = "monotonic_id"
	ValueColumn                = "value"
	RevisionColumn             = "revision"
	PayloadColumn              = "p"
	InputColumn                = "input"
	OutputColumn               = "output"
	RequestIDColumn            = "request_id"
	OperationColumn            = "operation"
	TTLColumn                  = "ttl"
)

var (
	SelectByNameQuery            xydb.Query = "select_by_name"
	SelectByIDQuery              xydb.Query = "select_by_id"
	EnsureMissing                xydb.Query = "ensure_missing"
	InsertQuery                  xydb.Query = "insert"
	UpdateQuery                  xydb.Query = "update"
	ListQuery                    xydb.Query = "list"
	ListExecutionsByTaskletQuery xydb.Query = "list_executions_by_tasklet"
	ListExecutionsByBuildQuery   xydb.Query = "list_executions_by_build"
)

var (
	NameIndex = xydb.SecondaryIndex{
		Name:    "name_view",
		Columns: []string{NameColumn},
	}

	NamespaceNameIndex = xydb.SecondaryIndex{
		Name:    "namespace_name_view",
		Columns: []string{NamespaceColumn, NameColumn},
	}

	TaskletIDRevisionIndex = xydb.SecondaryIndex{
		Name:    "tasklet_id_revision_view",
		Columns: []string{TaskletIDColumn, RevisionColumn},
	}
	TaskletIDNameIndex = xydb.SecondaryIndex{
		Name:    "tasklet_id_name_view",
		Columns: []string{TaskletIDColumn, NameColumn},
	}
	TaskletIDExecutionIDIndex = xydb.SecondaryIndex{
		Name:    "tasklet_id_monotonic_id_view",
		Columns: []string{TaskletIDColumn, ExecutionMonotonicIDColumn},
	}
	BuildIDExecutionIDIndex = xydb.SecondaryIndex{
		Name:    "build_id_monotonic_id_view",
		Columns: []string{BuildIDColumn, ExecutionMonotonicIDColumn},
	}
)

var (
	NamespaceTable           = "namespaces"
	TaskletTable             = "tasklets"
	BuildsTable              = "builds"
	LabelsTable              = "labels"
	ExecutionsMonotonicTable = "executions_monotonic"
	ExecutionsTable          = "executions"
	ExecutionsArchiveTable   = "executions_archive"
	ExecutionBlobsTable      = "execution_blobs"
	SchemaRegistryTable      = "schema_registry"
	ExecutionJournalTable    = "execution_journal"
)

var Tables map[string]xydb.TableSchema

func PurgeDatabase(
	ctx context.Context,
	cli *xydb.Client,
) error {
	return cli.PurgeTablesBySchema(ctx, Tables)
}

func CreateTables(ctx context.Context, cli *xydb.Client) error {

	if err := cli.CreateTablesBySchema(ctx, Tables); err != nil {
		return err
	}

	// Special init for executions
	query := cli.QueryPrefix() + fmt.Sprintf(
		`
			INSERT INTO %v (%v, %v) VALUES (%v, 1);
			`, ExecutionsMonotonicTable, IDColumn, ValueColumn, ExecutionMonotonicKey,
	)

	err := cli.ExecuteWriteQuery(ctx, query)
	if ydb.IsOperationError(err, Ydb.StatusIds_PRECONDITION_FAILED) {
		return nil
	} else {
		return err
	}

}

func render(t *template.Template, data interface{}) string {
	var buf bytes.Buffer
	err := t.Execute(&buf, data)
	if err != nil {
		panic(err)
	}
	return buf.String()
}

type TGenericRenderingOptions struct {
	Table string
	// Columns
	ID                   string
	Name                 string
	Namespace            string
	TaskletID            string
	NamespaceID          string
	Revision             string
	BuildID              string
	ExecutionMonotonicID string
	Value                string
	P                    string
	Input                string
	Output               string
	RequestID            string
	Operation            string
	TTL                  string

	// Indexes
	NameView                 string
	NamespaceNameView        string
	TaskletIDRevisionView    string
	TaskletIDNameView        string
	TaskletIDExecutionIDView string
	BuildIDExecutionIDView   string

	// Tables
	NamespaceTable           string
	BuildsTable              string
	ExecutionsMonotonicTable string
	ExecutionsTable          string
	ExecutionsArchiveTable   string
	ExecutionBlobsTable      string
	ExecutionJournalTable    string
}

var GenericTemplateOptions = TGenericRenderingOptions{
	Table:                "invalid_table_name",
	ID:                   IDColumn,
	Name:                 NameColumn,
	Namespace:            NamespaceColumn,
	TaskletID:            TaskletIDColumn,
	NamespaceID:          NamespaceIDColumn,
	Revision:             RevisionColumn,
	BuildID:              BuildIDColumn,
	ExecutionMonotonicID: ExecutionMonotonicIDColumn,
	Value:                ValueColumn,
	P:                    PayloadColumn,
	Input:                InputColumn,
	Output:               OutputColumn,
	RequestID:            RequestIDColumn,
	Operation:            OperationColumn,
	TTL:                  TTLColumn,

	NameView:                 NameIndex.Name,
	NamespaceNameView:        NamespaceNameIndex.Name,
	TaskletIDRevisionView:    TaskletIDRevisionIndex.Name,
	TaskletIDNameView:        TaskletIDNameIndex.Name,
	TaskletIDExecutionIDView: TaskletIDExecutionIDIndex.Name,
	BuildIDExecutionIDView:   BuildIDExecutionIDIndex.Name,

	NamespaceTable:           NamespaceTable,
	BuildsTable:              BuildsTable,
	ExecutionsMonotonicTable: ExecutionsMonotonicTable,
	ExecutionsTable:          ExecutionsTable,
	ExecutionsArchiveTable:   ExecutionsArchiveTable,
	ExecutionBlobsTable:      ExecutionBlobsTable,
	ExecutionJournalTable:    ExecutionJournalTable,
}

func init() {
	Tables = make(map[string]xydb.TableSchema)
	initNamespace()
	initTasklets()
	initBuilds()
	initLabels()
	initExecution()
	initSchemaRegistry()
}
