package ydbstore

import (
	"context"
	"text/template"

	ydbTable "github.com/ydb-platform/ydb-go-sdk/v3/table"
	ydbNamed "github.com/ydb-platform/ydb-go-sdk/v3/table/result/named"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/descriptorpb"
	"google.golang.org/protobuf/types/known/structpb"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/yt/go/compression"
)

type schemaRowBuffer struct {
	hash        *string
	author      *string
	descriptor  *[]byte
	annotations *[]byte
	timestamp   *int64
}

var schemaCompressionCodec = compression.NewCodec(compression.CodecIDSnappy)

func (s *schemaRowBuffer) Parse() (common.SchemaRecord, error) {
	rv := common.SchemaRecord{}
	if s.hash == nil {
		return rv, xerrors.New("Empty \"hash\"")
	}
	if s.author == nil {
		return rv, xerrors.New("Empty \"author\"")
	}

	if s.timestamp == nil {
		return rv, xerrors.New("Empty \"timestamp\"")
	}

	if s.descriptor == nil {
		return rv, xerrors.New("Empty \"descriptor\"")
	}

	fds := &descriptorpb.FileDescriptorSet{}

	if decompressedFds, err := schemaCompressionCodec.Decompress(*s.descriptor); err != nil {
		return rv, xerrors.NewSentinel("Failed to decompress schema").Wrap(err)
	} else if err := proto.Unmarshal(decompressedFds, fds); err != nil {
		return rv, xerrors.NewSentinel("descriptor unmarshall failed").Wrap(err)
	}

	if s.annotations == nil {
		return rv, xerrors.New("Empty \"annotations\"")
	}
	annotations := &structpb.Struct{}
	if err := annotations.UnmarshalJSON(*s.annotations); err != nil {
		return rv, xerrors.NewSentinel("Annotations unmarshall failed").Wrap(err)
	}

	rv.User = *s.author
	rv.Hash = *s.hash
	rv.Timestamp = *s.timestamp
	rv.Fds = fds
	rv.Annotations = annotations
	return rv, nil
}

//goland:noinspection SqlNoDataSourceInspection
var InsertSchemaTemplate = template.Must(
	template.New("sr-ist").Parse(
		// language=SQL
		`
		DECLARE ${{ .Hash }} AS String;
		DECLARE ${{ .Author }} AS String;
		DECLARE ${{ .Ctime }} AS Int64;
		DECLARE ${{ .Descriptor }} AS String;
		DECLARE ${{ .Annotations }} AS JsonDocument;

		REPLACE INTO {{ .SchemaRegistryTable }}
			({{ .Hash }}, {{ .Author }}, {{ .Ctime }}, {{ .Descriptor }}, {{ .Annotations }})
		VALUES
			(${{ .Hash }}, ${{ .Author }}, ${{ .Ctime }}, ${{ .Descriptor }}, ${{ .Annotations }})
		;
		`,
	),
)

func (s *Storage) EnsureSchema(ctx context.Context, req common.SchemaRecord) (common.SchemaRecord, error) {
	consts := &SchemaRegistrySchemaConstants

	// insert
	insertQuery := s.client.QueryPrefix()
	insertQuery += Tables[SchemaRegistryTable].Queries[consts.InsertSchema]

	var dumpedFds []byte

	if v, err := proto.Marshal(req.Fds); err != nil {
		return req, err
	} else if buf, err := schemaCompressionCodec.Compress(v); err != nil {
		return req, err
	} else {
		dumpedFds = buf
	}

	var dumpedAnnotations []byte
	if v, err := req.Annotations.MarshalJSON(); err != nil {
		return req, err
	} else {
		dumpedAnnotations = v
	}

	insertQueryParams := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+consts.Hash, ydbTypes.StringValueFromString(req.Hash)),
		ydbTable.ValueParam("$"+consts.Author, ydbTypes.StringValueFromString(req.User)),
		ydbTable.ValueParam("$"+consts.Ctime, ydbTypes.Int64Value(req.Timestamp)),
		ydbTable.ValueParam("$"+consts.Descriptor, ydbTypes.StringValue(dumpedFds)),
		ydbTable.ValueParam("$"+consts.Annotations, ydbTypes.JSONDocumentValueFromBytes(dumpedAnnotations)),
	)

	insertOperation := func(txCtx context.Context, tx ydbTable.TransactionActor) error {
		ctxlog.Infof(ctx, s.l, "Query: %v", insertQuery)
		res, err := tx.Execute(
			txCtx,
			insertQuery,
			insertQueryParams,
		)
		if err != nil {
			return err
		}
		defer func() {
			_ = res.Close()
		}()
		return res.Err()
	}

	// precondition
	preconditionQuery := s.client.QueryPrefix()
	preconditionQuery += Tables[SchemaRegistryTable].Queries[consts.SelectSchemaByHash]
	preconditionQueryParams := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+consts.Hash, ydbTypes.StringValueFromString(req.Hash)),
	)

	// payload buffers
	resultCount := 0
	preconditionQueryBuf := schemaRowBuffer{}

	preconditionOperation := func(txCtx context.Context, tx ydbTable.TransactionActor) error {
		ctxlog.Infof(txCtx, s.l, "Query: %v", preconditionQuery)
		res, err := tx.Execute(
			txCtx,
			preconditionQuery,
			preconditionQueryParams,
		)
		if err != nil {
			return err
		}
		defer func() {
			_ = res.Close()
		}()

		for res.NextResultSet(txCtx) {
			for res.NextRow() {
				resultCount += 1
				err = res.ScanNamed(
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Hash, &preconditionQueryBuf.hash),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Author, &preconditionQueryBuf.author),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Ctime, &preconditionQueryBuf.timestamp),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Descriptor, &preconditionQueryBuf.descriptor),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Annotations, &preconditionQueryBuf.annotations),
				)

				if err != nil {
					return err
				}
			}
		}
		return res.Err()
	}

	// transaction

	txOperation := func(txCtx context.Context, tx ydbTable.TransactionActor) error {
		if err := preconditionOperation(txCtx, tx); err != nil {
			return err
		}
		if resultCount > 0 {
			// NB: schema exists. nothing to do
			return nil
		}

		return insertOperation(txCtx, tx)
	}

	if err := s.client.DB.Table().DoTx(
		ctx, txOperation, ydbTable.WithTxSettings(
			ydbTable.TxSettings(
				ydbTable.WithSerializableReadWrite(),
			),
		),
	); err != nil {
		return req, err
	}

	if resultCount == 0 {
		return req, nil
	}

	// parse queried data
	return preconditionQueryBuf.Parse()
}

//goland:noinspection SqlNoDataSourceInspection
var SelectSchemaTemplate = template.Must(
	template.New("sr-sst").Parse(
		// language=SQL
		`
		DECLARE ${{ .Hash }} AS String;
		SELECT
			{{ .Hash }}, {{ .Author }}, {{ .Ctime }}, {{ .Descriptor }}, {{ .Annotations }}
		FROM
			{{ .SchemaRegistryTable }}
		WHERE
			{{ .Hash }} == ${{ .Hash }}
		;
		`,
	),
)

func (s *Storage) GetSchema(ctx context.Context, hash string) (common.SchemaRecord, error) {
	query := s.client.QueryPrefix()
	query += Tables[SchemaRegistryTable].Queries[SchemaRegistrySchemaConstants.SelectSchemaByHash]
	ctxlog.Infof(ctx, s.l, "Query: %v", query)

	queryParams := ydbTable.NewQueryParameters(
		ydbTable.ValueParam("$"+SchemaRegistrySchemaConstants.Hash, ydbTypes.StringValueFromString(hash)),
	)

	// payload buffers
	resultCount := 0
	queryBuf := schemaRowBuffer{}

	doQuery := func(opCtx context.Context, session ydbTable.Session) error {
		_, res, err := session.Execute(
			opCtx,
			s.client.ReadTxControl,
			query,
			queryParams,
		)
		if err != nil {
			return err
		}
		defer func() {
			_ = res.Close()
		}()

		for res.NextResultSet(opCtx) {
			for res.NextRow() {
				resultCount += 1
				err = res.ScanNamed(
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Hash, &queryBuf.hash),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Author, &queryBuf.author),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Ctime, &queryBuf.timestamp),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Descriptor, &queryBuf.descriptor),
					ydbNamed.Optional(SchemaRegistrySchemaConstants.Annotations, &queryBuf.annotations),
				)

				if err != nil {
					return err
				}
			}
		}
		return res.Err()
	}

	if err := s.client.DB.Table().Do(ctx, doQuery, ydbTable.WithIdempotent()); err != nil {
		return common.SchemaRecord{}, err
	}

	if resultCount == 0 {
		return common.SchemaRecord{}, common.ErrObjectNotFound
	}

	return queryBuf.Parse()

}

// Schema registry consts

type TSchemaRegistrySchemaConstants struct {
	// Table
	SchemaRegistryTable string
	// Columns
	Hash        string
	Author      string
	Ctime       string
	Descriptor  string
	Annotations string

	// queries
	SelectSchemaByHash xydb.Query
	InsertSchema       xydb.Query
}

var SchemaRegistrySchemaConstants = TSchemaRegistrySchemaConstants{
	// tables
	SchemaRegistryTable: SchemaRegistryTable,

	// columns
	Hash:        "hash",
	Author:      "author",
	Ctime:       "ctime",
	Descriptor:  "descriptor",
	Annotations: "annotations",

	// queries
	SelectSchemaByHash: "select_by_hash",
	InsertSchema:       "insert_schema",
}

func initSchemaRegistry() {
	opts := SchemaRegistrySchemaConstants
	table := xydb.TableSchema{
		Name: SchemaRegistryTable,
		Columns: []xydb.YdbColumn{
			{
				Name:       opts.Hash,
				ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
				PrimaryKey: true,
			},
			{
				Name:      opts.Author,
				ValueType: ydbTypes.Optional(ydbTypes.TypeString),
			},
			{
				Name:      opts.Ctime,
				ValueType: ydbTypes.Optional(ydbTypes.TypeInt64),
			},
			{
				Name:      opts.Descriptor,
				ValueType: ydbTypes.Optional(ydbTypes.TypeString),
			},
			{
				Name:      opts.Annotations,
				ValueType: ydbTypes.Optional(ydbTypes.TypeJSONDocument), // NB: Maybe support queries?
			},
		},
		SecondaryIndexes: []xydb.SecondaryIndex{},
		Queries:          make(map[xydb.Query]string),
	}
	table.Queries[opts.SelectSchemaByHash] = render(SelectSchemaTemplate, opts)
	table.Queries[opts.InsertSchema] = render(InsertSchemaTemplate, opts)

	Tables[table.Name] = table

}
