package ydbstore

import (
	"context"
	"fmt"
	"text/template"

	"github.com/ydb-platform/ydb-go-sdk/v3/table"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	ydbNamed "github.com/ydb-platform/ydb-go-sdk/v3/table/result/named"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/types"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

//goland:noinspection SqlNoDataSourceInspection
var InsertTaskletTemplate = template.Must(
	template.New("itt").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .Namespace }} AS String;
		DECLARE ${{ .P }} AS JsonDocument;

		INSERT INTO {{ .Table }} ({{ .ID }}, {{ .Name }}, {{ .Namespace }}, {{ .P }})
		VALUES (${{ .ID }}, ${{ .Name }}, ${{ .Namespace }}, ${{ .P }});
		`,
	),
)

func (s *Storage) AddTasklet(ctx context.Context, t *taskletv2.Tasklet) error {
	serialized, err := protojson.Marshal(t)
	if err != nil {
		ctxlog.Infof(ctx, s.l, "Marshall failed: %v", err)
		return err
	}

	selectQuery := s.client.QueryPrefix() + Tables[TaskletTable].Queries[SelectByNameQuery]
	s.logQuery(ctx, "add_tasklet_precondition", selectQuery)
	updateQuery := s.client.QueryPrefix() + Tables[TaskletTable].Queries[InsertQuery]
	s.logQuery(ctx, "add_tasklet", updateQuery)

	meta := t.GetMeta()

	queryErr := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			txControl := table.TxControl(table.BeginTx(table.WithSerializableReadWrite()))

			tx, res, err := session.Execute(
				c, txControl, selectQuery, table.NewQueryParameters(
					table.ValueParam("$"+NameColumn, types.StringValueFromString(meta.Name)),
					table.ValueParam("$"+NamespaceColumn, types.StringValueFromString(meta.Namespace)),
				),
			)
			ctxlog.Infof(c, s.l, "Execute1 result. txOk: %v, , err: %v", tx != nil, err)
			defer func() {
				if tx != nil {
					_ = tx.Rollback(c)
				}
			}()
			defer func() {
				_ = res.Close()
			}()

			if err != nil {
				return err
			}

			for res.NextResultSet(c, PayloadColumn) {
				for res.NextRow() {
					return common.ErrObjectExists
				}
			}
			_, err = tx.Execute(
				c, updateQuery, table.NewQueryParameters(
					table.ValueParam("$"+IDColumn, types.StringValueFromString(meta.Id)),
					table.ValueParam("$"+NameColumn, types.StringValueFromString(meta.Name)),
					table.ValueParam("$"+NamespaceColumn, types.StringValueFromString(meta.Namespace)),
					table.ValueParam("$"+PayloadColumn, types.JSONDocumentValueFromBytes(serialized)),
				),
			)
			ctxlog.Infof(c, s.l, "Execute2 result. err: %v", err)

			if err != nil {
				return err
			}
			_, err = tx.CommitTx(c, options.WithCommitCollectStatsModeBasic())
			return err
		},
	)
	return queryErr
}

//goland:noinspection SqlNoDataSourceInspection
var SelectTaskletByNameTemplate = template.Must(
	template.New("stt").Parse(
		// language=SQL
		`
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .Namespace }} AS String;
		SELECT
			{{ .P }}
		FROM
			{{ .Table }} VIEW {{ .NamespaceNameView }}
		WHERE
			{{ .Name }} == ${{ .Name }} AND
			{{ .Namespace }} == ${{ .Namespace }}
		;
		`,
	),
)

func (s *Storage) GetTaskletByName(ctx context.Context, name, namespace string) (*taskletv2.Tasklet, error) {
	q := s.client.QueryPrefix() + Tables[TaskletTable].Queries[SelectByNameQuery]
	ctxlog.Info(ctx, s.l, q)
	var payload *[]byte = nil
	err := s.tableDo(
		ctx, func(opCtx context.Context, session table.Session) error {
			_, res, err := session.Execute(
				opCtx, s.client.ReadTxControl, q, table.NewQueryParameters(
					table.ValueParam("$"+NameColumn, types.StringValueFromString(name)),
					table.ValueParam("$"+NamespaceColumn, types.StringValueFromString(namespace)),
				),
			)
			ctxlog.Infof(opCtx, s.l, "QRES: r: %v, err: %v", res, err)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			itemsProcessed := 0
			for res.NextResultSet(opCtx, PayloadColumn) {
				for res.NextRow() {
					err := res.Scan(&payload)
					if err != nil {
						return err
					}
					itemsProcessed += 1
					if itemsProcessed > 1 {
						return fmt.Errorf("multiple results. Name: %q", name)
					}
				}
			}
			return res.Err()
		},
	)
	if err != nil {
		return nil, err
	}
	if payload == nil {
		return nil, common.ErrObjectNotFound
	}
	ctxlog.Infof(ctx, s.l, "RET RET err: %v, pl: %q", err, string(*payload))

	rv := &taskletv2.Tasklet{}
	if err := protojson.Unmarshal(*payload, rv); err != nil {
		return nil, err
	}
	return rv, nil
}

func (s *Storage) GetTaskletByID(ctx context.Context, id consts.TaskletID) (*taskletv2.Tasklet, error) {
	query := Tables[TaskletTable].Queries[SelectByIDQuery]
	var payload []byte = nil
	err := s.client.ReadOneRow(
		ctx,
		query,
		xydb.ReadValues{ydbNamed.OptionalWithDefault(PayloadColumn, &payload)},
		table.ValueParam("$"+IDColumn, types.StringValueFromString(id.String())),
	)
	if xerrors.Is(err, xydb.ErrNoRows) {
		return nil, common.ErrObjectNotFound
	}
	rv := &taskletv2.Tasklet{}
	if err := protojson.Unmarshal(payload, rv); err != nil {
		return nil, err
	}
	return rv, nil
}

//goland:noinspection SqlNoDataSourceInspection
var UpdateTaskletTemplate = template.Must(
	template.New("itt").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .Namespace }} AS String;
		DECLARE ${{ .P }} AS JsonDocument;

		$getRevision = ($pp) -> {
			RETURN Unwrap(
				JSON_VALUE(
					$pp,
					"$.spec.revision"
					RETURNING Uint32
					DEFAULT 0 ON EMPTY
					ERROR ON ERROR
				)
			);
		};

		DISCARD SELECT
			Ensure(
				0,
				$getRevision(${{ .P }})  == $getRevision({{ .P }}) + 1,
				"Invalid revision. Object ID: '" || Unwrap({{ .ID }}) || "', Revision: " || CAST(Unwrap($getRevision({{ .P }})) AS String)
			)
		FROM {{ .Table }}
		WHERE
			{{ .ID }} == ${{ .ID }}
		;

		UPSERT INTO {{ .Table }} ({{ .ID }}, {{ .Name }}, {{ .Namespace }}, {{ .P }})
		VALUES (${{ .ID }}, ${{ .Name }}, ${{ .Namespace }}, ${{ .P }});
		`,
	),
)

func (s *Storage) UpdateTasklet(ctx context.Context, t *taskletv2.Tasklet) (*taskletv2.Tasklet, error) {
	if t.GetMeta().GetId() == "" {
		panic("missing object ID")
	}
	meta := t.GetMeta()
	query := s.client.QueryPrefix() + Tables[TaskletTable].Queries[UpdateQuery]
	ctxlog.Infof(ctx, s.l, "Update query: %v", query)
	serialized, err := protojson.Marshal(t)
	if err != nil {
		ctxlog.Infof(ctx, s.l, "Marshall failed: %v", err)
		return nil, err
	}

	errQ := s.client.ExecuteWriteQuery(
		ctx,
		query,
		table.ValueParam("$"+IDColumn, types.StringValueFromString(meta.Id)),
		table.ValueParam("$"+NameColumn, types.StringValueFromString(meta.Name)),
		table.ValueParam("$"+NamespaceColumn, types.StringValueFromString(meta.Namespace)),
		table.ValueParam("$"+PayloadColumn, types.JSONDocumentValueFromBytes(serialized)),
	)
	if errQ != nil {
		return nil, errQ
	}
	return t, nil
}

//goland:noinspection SqlNoDataSourceInspection
var SelectTaskletByIDTemplate = template.Must(
	template.New("stbit").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		SELECT
			{{ .P }}
		FROM
			{{ .Table }}
		WHERE
			{{ .ID }} == ${{ .ID }}
		;
		`,
	),
)

func (s *Storage) loadTaskletByIDTx(ctx context.Context, tx table.TransactionActor, id string) (
	*taskletv2.Tasklet,
	error,
) {

	query := s.client.QueryPrefix() + Tables[TaskletTable].Queries[SelectByIDQuery]
	ctxlog.Infof(ctx, s.l, "Query: %v", query)
	res, err := tx.Execute(
		ctx,
		query,
		table.NewQueryParameters(
			table.ValueParam("$"+IDColumn, types.StringValueFromString(id)),
		),
	)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = res.Close()
	}()

	var payload *[]byte = nil
	resultCount := 0
	for res.NextResultSet(ctx) {
		for res.NextRow() {
			resultCount += 1
			if resultCount > 1 {
				panic(fmt.Sprintf("multiple results for by PK: %q", id))
			}
			err = res.ScanNamed(
				ydbNamed.Optional(PayloadColumn, &payload),
			)

			if err != nil {
				return nil, err
			}
		}
	}
	if res.Err() != nil {
		return nil, res.Err()
	}
	if payload == nil {
		return nil, common.ErrObjectNotFound
	}
	rv := &taskletv2.Tasklet{}
	if err := protojson.Unmarshal(*payload, rv); err != nil {
		return nil, err
	}
	return rv, nil
}

//goland:noinspection SqlNoDataSourceInspection
var UpdateTaskletPayloadTemplate = template.Must(
	template.New("utpt").Parse(
		// language=SQL
		`
		DECLARE ${{ .ID }} AS String;
		DECLARE ${{ .P }} AS JsonDocument;

		UPDATE {{ .Table }}
		SET {{ .P }} = ${{ .P }}
		WHERE {{ .ID }} == ${{ .ID }}
		;
		`,
	),
)

func (s *Storage) updateTaskletByIDTx(ctx context.Context, tx table.TransactionActor, id string, payload []byte) error {
	query := s.client.QueryPrefix() + Tables[TaskletTable].Queries[TaskletUpdatePayloadQuery]
	ctxlog.Infof(ctx, s.l, "Update query: %v", query)

	_, err := tx.Execute(
		ctx,
		query,
		table.NewQueryParameters(
			table.ValueParam("$"+IDColumn, types.StringValueFromString(id)),
			table.ValueParam("$"+PayloadColumn, types.JSONDocumentValueFromBytes(payload)),
		),
	)
	if err != nil {
		return err
	}
	return nil

}

// UpdateTaskletOp apply any update to Tasklet in transaction
// This does not imply Spec.Version check and does not mutate it
// NB! most Meta fields ARE immutable by semantics. Not checks implied
func (s *Storage) UpdateTaskletOp(
	ctx context.Context,
	id string,
	updates ...common.TaskletUpdateOperation,
) (*taskletv2.Tasklet, error) {
	if id == "" {
		panic("missing object ID")
	}

	var rv *taskletv2.Tasklet = nil
	txOperation := func(txCtx context.Context, tx table.TransactionActor) error {
		tl, err := s.loadTaskletByIDTx(txCtx, tx, id)
		if err != nil {
			return err
		}
		for _, update := range updates {
			if err := update(tl); err != nil {
				return err
			}
		}
		serialized, err := protojson.Marshal(tl)
		if err != nil {
			return xerrors.Errorf("Marshal failed: %w", err)
		}

		rv = tl
		return s.updateTaskletByIDTx(txCtx, tx, id, serialized)
	}

	if err := s.tableDoTx(
		ctx, txOperation, table.WithTxSettings(
			table.TxSettings(
				table.WithSerializableReadWrite(),
			),
		),
	); err != nil {
		return nil, err
	}
	return rv, nil
}

//goland:noinspection SqlNoDataSourceInspection
var ListTaskletsTemplate = template.Must(
	template.New("ltt").Parse(
		// language=SQL
		`
		DECLARE ${{ .Name }} AS String;
		DECLARE ${{ .Namespace }} AS String;

		SELECT
			{{ .Name }}, {{ .Namespace }}, {{ .P }}
		FROM
			{{ .Table }} VIEW {{ .NamespaceNameView }}
		WHERE
			IF(${{ .Namespace }} == "", true, {{ .Namespace }} == ${{ .Namespace }}) AND
			{{ .Name }} < IF(${{ .Name }} == "" , "}love", ${{ .Name }})
		ORDER BY {{ .Namespace }}, {{ .Name }} DESC
		LIMIT 50
		;
		`,
	),
)

func (s *Storage) ListTasklets(ctx context.Context, namespace string) ([]*taskletv2.Tasklet, error) {
	query := s.client.QueryPrefix() + Tables[TaskletTable].Queries[ListQuery]
	ctxlog.Infof(ctx, s.l, "List query: %v", query)
	response := make([]*taskletv2.Tasklet, 0)

	queryErr := s.tableDo(
		ctx, func(c context.Context, session table.Session) error {
			_, res, err := session.Execute(
				c, s.client.ReadTxControl, query, table.NewQueryParameters(
					table.ValueParam("$"+NameColumn, types.StringValueFromString("")),
					table.ValueParam("$"+NamespaceColumn, types.StringValueFromString(namespace)),
				),
			)
			ctxlog.Infof(ctx, s.l, "QRES: r: %v, err: %v", res, err)
			if err != nil {
				return err
			}
			defer func() {
				_ = res.Close()
			}()

			for res.NextResultSet(ctx, PayloadColumn) {
				for res.NextRow() {
					var payload *[]byte
					if err := res.Scan(&payload); err != nil {
						return err
					}
					item := &taskletv2.Tasklet{}
					if err := protojson.Unmarshal(*payload, item); err != nil {
						return err
					}
					response = append(response, item)

				}
			}
			return res.Err()
		},
	)

	if queryErr != nil {
		return nil, queryErr
	}
	return response, nil
}

var TaskletUpdatePayloadQuery xydb.Query = "tasklet_update_payload"

func initTasklets() {
	tl := xydb.TableSchema{
		Name: TaskletTable,
		Columns: []xydb.YdbColumn{
			{
				Name:       IDColumn,
				ValueType:  types.Optional(types.TypeString),
				PrimaryKey: true,
			},
			{
				Name:      NameColumn,
				ValueType: types.Optional(types.TypeString),
			},
			{
				Name:      NamespaceColumn,
				ValueType: types.Optional(types.TypeString),
			},
			{
				Name:      PayloadColumn,
				ValueType: types.Optional(types.TypeJSONDocument),
			},
		},
		SecondaryIndexes: []xydb.SecondaryIndex{NamespaceNameIndex},
		Queries:          make(map[xydb.Query]string),
	}
	thisOpts := GenericTemplateOptions
	thisOpts.Table = tl.Name
	tl.Queries[SelectByIDQuery] = render(SelectTaskletByIDTemplate, thisOpts)
	tl.Queries[SelectByNameQuery] = render(SelectTaskletByNameTemplate, thisOpts)
	tl.Queries[InsertQuery] = render(InsertTaskletTemplate, thisOpts)
	tl.Queries[UpdateQuery] = render(UpdateTaskletTemplate, thisOpts)
	tl.Queries[ListQuery] = render(ListTaskletsTemplate, thisOpts)
	tl.Queries[TaskletUpdatePayloadQuery] = render(UpdateTaskletPayloadTemplate, thisOpts)
	Tables[tl.Name] = tl

}
