package testutils

import (
	"fmt"
	"time"

	"github.com/c2h5oh/datasize"
	"github.com/gofrs/uuid"
	"google.golang.org/grpc/codes"
	"google.golang.org/protobuf/encoding/prototext"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/handler/xmodels"
)

var timestamp = time.Unix(1652360629, 1543)

const (
	DefaultUserPayloadResourceID     = 1543
	DefaultTaskletTaskResourceID     = 1001
	DefaultTaskletExecutorResourceID = 1002
)

const (
	NamespaceArcadia = "arcadia"
	NamespaceTaxi    = "taxi"
)

const (
	TaskletTestTasklet     = "test_tasklet"
	TaskletNewTasklet      = "new_tasklet"
	TaskletTestTaxiTasklet = "taxi_test_tasklet"
)

var taskletPrototype = &taskletv2.Tasklet{
	Meta: &taskletv2.TaskletMeta{
		Id:        "uuid",
		Name:      "name",
		AccountId: "abc:123",
		CreatedAt: timestamppb.New(timestamp),
		Namespace: "name",
	},
	Spec: &taskletv2.TaskletSpec{
		Revision:      0,
		Catalog:       "/home/abash",
		TrackingLabel: "default",
	},
}

var buildPrototype = &taskletv2.Build{
	Meta: &taskletv2.BuildMeta{
		Id:        "uid1",
		CreatedAt: timestamppb.New(timestamp),
		Tasklet:   "tl",
		Namespace: "ns",
		TaskletId: "uid2",
		Revision:  0,
	},
	Spec: &taskletv2.BuildSpec{
		Description: "blah",
		ComputeResources: &taskletv2.ComputeResources{
			VcpuLimit:   1000,
			MemoryLimit: 1024 * 1024 * 100,
		},
		Payload: &taskletv2.BuildSpec_Payload{
			SandboxResourceId: DefaultUserPayloadResourceID,
		},
		LaunchSpec: &taskletv2.LaunchSpec{Type: consts.LaunchTypeBinary},
		Workspace: &taskletv2.BuildSpec_Workspace{
			StorageClass: taskletv2.EStorageClass_E_STORAGE_CLASS_HDD,
			StorageSize:  int64(datasize.GB.Bytes() * 10),
		},
		Environment: &taskletv2.Environment{
			Porto:  &taskletv2.PortoEnvironment{Enabled: true},
			Docker: nil,
			Java:   &taskletv2.JavaEnvironment{},
		},
		Schema: &taskletv2.IOSchema{
			SimpleProto: &taskletv2.IOSimpleSchemaProto{
				SchemaHash:    "deadbeef",
				InputMessage:  "tasklet.api.v2.GenericBinary",
				OutputMessage: "tasklet.api.v2.GenericBinary",
			},
		},
	},
	Status: nil,
}

var executionPrototype = &taskletv2.Execution{
	Meta: &taskletv2.ExecutionMeta{
		Id:        "uuid",
		TaskletId: "uuid",
		BuildId:   "uuid",
		CreatedAt: timestamppb.New(timestamp),
	},
	Spec: &taskletv2.ExecutionSpec{
		Author:          "abash",
		ReferencedLabel: "default",
		Requirements: &taskletv2.ExecutionRequirements{
			AccountId: "ACCOUNT_NAME",
		},
		Input: &taskletv2.ExecutionInput{
			SerializedData: nil,
		},
	},
	Status: nil,
}

// ObjectGenerator generates dummy messages for components testing. No data validation
type ObjectGenerator struct {
	namespaces map[string]*taskletv2.Namespace
	tasklets   map[string]*taskletv2.Tasklet
	builds     map[string]*taskletv2.Build
	executions map[string]*taskletv2.Execution
	seqNo      map[ObjectKind]int
}

func NewObjectGenerator() *ObjectGenerator {
	return &ObjectGenerator{
		namespaces: make(map[string]*taskletv2.Namespace),
		tasklets:   make(map[string]*taskletv2.Tasklet),
		builds:     make(map[string]*taskletv2.Build),
		executions: make(map[string]*taskletv2.Execution),
		seqNo:      make(map[ObjectKind]int),
	}
}

type ObjectKind int

const (
	ONamespace ObjectKind = iota
	OTasklet
	OLabel
	OBuild
	OExecution
)

func (og *ObjectGenerator) getUUID(kind ObjectKind) string {
	og.seqNo[kind] = og.seqNo[kind] + 1
	seq := og.seqNo[kind]
	rv := fmt.Sprintf("%08x-0000-4000-8000-%012x", 10+kind, seq)
	parsed, err := uuid.FromString(rv)
	if err != nil {
		panic(err)
	}
	if parsed.Version() != uuid.V4 {
		panic(parsed.Version())
	}
	if parsed.Variant() != uuid.VariantRFC4122 {
		panic(parsed.Variant())
	}

	return rv
}

func (og *ObjectGenerator) NewNamespace(name string) *taskletv2.Namespace {
	for _, ns := range og.namespaces {
		if ns.Meta.Name == name {
			panic(name)
		}
	}
	ns := &taskletv2.Namespace{
		Meta: &taskletv2.NamespaceMeta{
			Id:        og.getUUID(ONamespace),
			Name:      name,
			AccountId: "abc:123",
			CreatedAt: timestamppb.New(timestamp),
		},
	}
	if st := xmodels.ValidateNamespace(ns); st.Code() != codes.OK {
		panic(prototext.Format(st.Proto()))
	}
	og.namespaces[ns.Meta.Id] = ns
	return ns
}
func (og *ObjectGenerator) NewTasklet(name string, ns *taskletv2.Namespace) *taskletv2.Tasklet {
	for _, t := range og.tasklets {
		if t.Meta.Name == name {
			panic(name)
		}
	}

	tl, ok := proto.Clone(taskletPrototype).(*taskletv2.Tasklet)
	if !ok {
		panic(ok)
	}
	tl.Meta.Id = og.getUUID(OTasklet)
	tl.Meta.Name = name
	tl.Meta.Namespace = ns.Meta.Name
	if st := xmodels.ValidateTasklet(tl); st.Code() != codes.OK {
		panic(prototext.Format(st.Proto()))
	}

	og.tasklets[tl.Meta.Id] = tl
	return tl
}

func (og *ObjectGenerator) NewBuild(tl *taskletv2.Tasklet) *taskletv2.Build {
	build, ok := proto.Clone(buildPrototype).(*taskletv2.Build)
	if !ok {
		panic(ok)
	}
	build.Meta.Id = og.getUUID(OBuild)
	build.Meta.Tasklet = tl.Meta.Name
	build.Meta.TaskletId = tl.Meta.Id
	build.Meta.Namespace = tl.Meta.Namespace
	if st := xmodels.ValidateBuild(build); st.Code() != codes.OK {
		panic(prototext.Format(st.Proto()))
	}

	og.builds[build.Meta.Id] = build
	return build
}

func (og *ObjectGenerator) NewExecution(b *taskletv2.Build) *taskletv2.Execution {
	ex, ok := proto.Clone(executionPrototype).(*taskletv2.Execution)
	if !ok {
		panic(ok)
	}
	ex.Meta.Id = og.getUUID(OExecution)
	ex.Meta.TaskletId = b.Meta.TaskletId
	ex.Meta.BuildId = b.Meta.Id
	if st := xmodels.ValidateExecution(ex); st.Code() != codes.OK {
		panic(prototext.Format(st.Proto()))
	}
	og.executions[ex.Meta.Id] = ex
	return ex
}

func (og *ObjectGenerator) CreateTestObjectsInMemory() ([]*taskletv2.Namespace, []*taskletv2.Tasklet) {
	arcadiaNs := og.NewNamespace(NamespaceArcadia)
	CreatePermissions(&arcadiaNs.Meta.Permissions, NamespaceRoles[arcadiaNs.Meta.Name])

	arcadiaTestTl := og.NewTasklet(TaskletTestTasklet, arcadiaNs)
	CreatePermissions(&arcadiaTestTl.Meta.Permissions, TaskletRoles[arcadiaTestTl.Meta.Name])
	arcadiaNewTl := og.NewTasklet(TaskletNewTasklet, arcadiaNs)

	taxiNs := og.NewNamespace(NamespaceTaxi)
	CreatePermissions(&taxiNs.Meta.Permissions, NamespaceRoles[taxiNs.Meta.Name])
	taxiTestTl := og.NewTasklet(TaskletTestTaxiTasklet, taxiNs)
	CreatePermissions(&taxiTestTl.Meta.Permissions, TaskletRoles[taxiTestTl.Meta.Name])
	return []*taskletv2.Namespace{arcadiaNs, taxiNs}, []*taskletv2.Tasklet{
		arcadiaTestTl,
		arcadiaNewTl,
		taxiTestTl,
	}
}
