package xgrpc

import (
	"bufio"
	"bytes"
	"context"
	"runtime/debug"

	"github.com/grpc-ecosystem/go-grpc-middleware/recovery"
	"google.golang.org/genproto/googleapis/rpc/errdetails"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
)

func PanicHandler(l log.Logger, expose bool) grpc_recovery.RecoveryHandlerFuncContext {
	return func(ctx context.Context, p interface{}) error {
		err := xerrors.Errorf("Panic: %+v", p)
		stack := debug.Stack()
		fields := []log.Field{
			log.Error(err),
			log.Any("trace", string(stack)),
		}
		ctxlog.Error(ctx, l, "request panic", fields...)
		if !expose {
			return ErrGeneralError
		} else {
			st, ok := status.FromError(ErrGeneralError)
			if !ok {
				return ErrGeneralError
			}
			details := &errdetails.DebugInfo{
				Detail: err.Error(),
			}
			sc := bufio.NewScanner(bytes.NewReader(stack))
			for sc.Scan() {
				details.StackEntries = append(details.StackEntries, sc.Text())
			}
			if rv, err := st.WithDetails(details); err != nil {
				return ErrGeneralError
			} else {
				return rv.Err()
			}
		}
	}
}
