package abc

import (
	"context"
	"crypto/tls"
	"fmt"
	"net/url"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/auth"
)

// API spec: https://abc-back.yandex-team.ru/api/swagger/

const (
	ProductionAPIUrl = "https://abc-back.yandex-team.ru"

	UserAgentName = "tasklets-abc-client"
)

type Client struct {
	httpClient  *resty.Client
	logger      log.Logger
	authFactory auth.ServicePolicyFactory
}

func NewClient(authFactory auth.ServicePolicyFactory, logger log.Logger) (*Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("failed to get cert pool")
	}

	httpClient := resty.New().
		SetBaseURL(ProductionAPIUrl).
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetHeader(consts.UserAgentHeader, UserAgentName).
		SetLogger(logger)

	return &Client{httpClient, logger, authFactory}, nil
}

func (c *Client) GetServicesBySlugs(ctx context.Context, slugs []string) (ServicesBySlugs, error) {
	var (
		results      = ServicesBySlugs{}
		unknownSlugs []string
	)

	for _, slug := range slugs {
		item := abcServiceBySlugCache.Get(slug)
		if item != nil && !item.Expired() {
			results[slug] = item.Value().(*Service)
		} else {
			unknownSlugs = append(unknownSlugs, slug)
		}
	}

	if newServices, err := c.getServicesBySlugsNoCache(ctx, unknownSlugs); err != nil {
		return nil, err
	} else {
		for _, service := range newServices {
			results[service.Slug] = service
			abcServiceBySlugCache.Set(service.Slug, service, cachedServiceTTL)
		}
	}

	return results, nil
}

func (c *Client) getServicesBySlugsNoCache(ctx context.Context, slugs []string) (ServicesBySlugs, error) {
	results := ServicesBySlugs{}
	if len(slugs) == 0 {
		return results, nil
	}
	ctxlog.Debugf(ctx, c.logger, "Fetching ABC-services by slugs %v", slugs)

	nextQueryParams := url.Values{
		"fields":            []string{serviceMembersListRequestFields},
		"service__slug__in": []string{strings.Join(slugs, ",")},
	}

	authPolicy, err := c.authFactory.NewAuthPolicy(ctx)
	if err != nil {
		return nil, xerrors.Errorf("failed to get auth policy: %w", err)
	}

	for len(nextQueryParams) > 0 {
		servicesResponse := &servicesMembersListResponse{}
		request := c.httpClient.R().
			SetContext(ctx).
			SetResult(servicesResponse).
			SetQueryParamsFromValues(nextQueryParams)

		if err := authPolicy.Apply(request); err != nil {
			return nil, err
		}

		resp, err := request.Get("/api/v4/services/members/")

		if err != nil {
			return nil, xerrors.Errorf("Failed to fetch services for slugs %v: %w", slugs, err)
		}
		if !resp.IsSuccess() {
			return nil, xerrors.Errorf(
				"Failed to fetch services for slugs %v. ABC responded with code %v: %v",
				slugs, resp.StatusCode(), resp,
			)
		}
		buildServicesFromMembers(servicesResponse.Results, results)

		next, err := url.Parse(servicesResponse.Next)
		if err != nil {
			return nil, fmt.Errorf("failed to parse 'next' url '%v'", servicesResponse.Next)
		}
		nextQueryParams = next.Query()
	}
	return results, nil
}

func buildServicesFromMembers(members []serviceMemberResponse, results ServicesBySlugs) {
	for _, member := range members {
		service := results[member.Service.Slug]
		if service == nil {
			service = &Service{
				ID:       member.Service.ID,
				Slug:     member.Service.Slug,
				ParentID: member.Service.ParentID,
				Members:  map[string]*ServiceMember{},
			}
			results[member.Service.Slug] = service
		}

		serviceMember := service.Members[member.Person.Login]
		if serviceMember == nil {
			serviceMember = &ServiceMember{
				Login:   member.Person.Login,
				IsRobot: member.Person.IsRobot,
			}
			service.Members[member.Person.Login] = serviceMember
		}

		serviceMember.Roles = append(
			serviceMember.Roles,
			ServiceRole{ID: member.Role.ID, Slug: member.Role.Scope.Slug, Code: member.Role.Code},
		)
	}
}
