package app

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"strconv"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/library/go/yandex/solomon/reporters/puller/httppuller"
	"a.yandex-team.ru/library/go/yandex/tvm"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/cmd/idm/configs"
	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/idmclient"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/services"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	staffmodel "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/model"
)

type HandlerType int

const (
	Info HandlerType = iota
	GetRoles
	AddRole
	RemoveRole
)

type IdmHandlerProps struct {
	Config         *configs.Config
	Logger         log.Logger
	IdmTreeBuilder *services.IdmTreeBuilder
	RoleBuilder    *services.RoleBuilder
	DB             storage.IStorage
	StaffCache     *staff.StaffGroupsCache
	StaffClient    staff.IClient
	TVMClient      tvm.Client
}

type IdmHandler struct {
	Handler HandlerType
	Props   IdmHandlerProps
}

var (
	// ErrFatal means mutation parameters are malformed. Mutation has do effect and must be discarded.
	ErrFatal = xerrors.NewSentinel("idm: fatal error")

	// ErrWarning means mutation encountered inconsistencies that were fixed.
	ErrWarning = xerrors.NewSentinel("idm: warning")
)

type IdmServer struct {
	Config     *configs.Config
	Logger     log.Logger
	DB         storage.IStorage
	HTTPServer *http.Server
}

func setupSolomon(rootMux *http.ServeMux, s *solomon.Registry) func(next http.Handler) http.Handler {
	if s != nil {
		rootMux.Handle("/_solomon/", httppuller.NewHandler(s))
		return httpmetrics.New(
			s.WithPrefix("idm"),
			// TODO: httpmetrics.WithEndpointKey()
		)
	}
	return func(next http.Handler) http.Handler {
		return next
	}
}

func NewIdmServer(
	config *configs.Config,
	logger log.Logger,
	db storage.IStorage,
	s *solomon.Registry,
	staffClient staff.IClient,
	staffCache *staff.StaffGroupsCache,
	auth AuthorizationInterface,
) *IdmServer {
	rootMux := http.NewServeMux()
	apiMux := http.NewServeMux()

	props := IdmHandlerProps{
		Config:         config,
		Logger:         logger,
		IdmTreeBuilder: &services.IdmTreeBuilder{Logger: logger, DB: db},
		RoleBuilder:    &services.RoleBuilder{Logger: logger, DB: db},
		DB:             db,
		StaffCache:     staffCache,
		StaffClient:    staffClient,
	}
	apiMux.Handle(
		"/info/", &IdmHandler{
			Handler: Info,
			Props:   props,
		},
	)
	apiMux.Handle(
		"/get-roles/", &IdmHandler{
			Handler: GetRoles,
			Props:   props,
		},
	)
	apiMux.Handle(
		"/add-role/", &IdmHandler{
			Handler: AddRole,
			Props:   props,
		},
	)
	apiMux.Handle(
		"/remove-role/", &IdmHandler{
			Handler: RemoveRole,
			Props:   props,
		},
	)

	solomonMiddleware := setupSolomon(rootMux, s)
	rootMux.Handle("/", auth.AuthorizeHandler()(solomonMiddleware(apiMux)))

	srv := &http.Server{
		Addr:    ":" + strconv.Itoa(config.IdmConfig.API.Port),
		Handler: rootMux,
	}
	return &IdmServer{
		Config:     config,
		Logger:     logger,
		HTTPServer: srv,
	}
}

func (h *IdmHandler) replyError(ctx context.Context, w http.ResponseWriter, code int, msg string) {
	w.WriteHeader(code)
	ctxlog.Infof(ctx, h.Props.Logger, "Reply error. Status: %v, Msg: %v", code, msg)
	_, err := w.Write([]byte(msg))
	if err != nil {
		ctxlog.Warn(ctx, h.Props.Logger, "Error writing reply", log.Error(err))
	}
}

func (h *IdmHandler) reply(ctx context.Context, w http.ResponseWriter, rsp Response) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(http.StatusOK)

	e := json.NewEncoder(w)
	e.SetIndent("", "    ")
	if err := e.Encode(rsp); err != nil {
		ctxlog.Warn(ctx, h.Props.Logger, "error writing reply", log.Error(err))
	}
}

func (h *IdmHandler) replyIDMError(ctx context.Context, w http.ResponseWriter, err error) {
	var fields []log.Field
	fields = append(fields, ctxlog.ContextFields(ctx)...)
	fields = append(fields, log.Error(err))

	rsp := &BaseResponse{}
	switch {
	case errors.Is(err, ErrWarning):
		ctxlog.Error(ctx, h.Props.Logger, "API warning", fields...)
		rsp.Warning = fmt.Sprintf("%s", err)

	case errors.Is(err, ErrFatal):
		ctxlog.Error(ctx, h.Props.Logger, "fatal API error", fields...)
		rsp.Code = 1
		rsp.Fatal = fmt.Sprintf("%s", err)

	default:
		ctxlog.Error(ctx, h.Props.Logger, "API error", fields...)
		rsp.Code = 1
		rsp.Error = fmt.Sprintf("%s", err)
	}

	h.reply(ctx, w, rsp)
}

func parseFormRole(r *http.Request) (*idmclient.Role, error) {
	if err := r.ParseForm(); err != nil {
		return nil, err
	}

	role := &idmclient.Role{}
	user := r.FormValue("login")
	if user != "" {
		role.Login = user
	}
	group, err := strconv.Atoi(r.FormValue("group"))
	if err == nil {
		role.Group = group
	}
	if user == "" && err != nil {
		err = errors.New("role change request must specify either login or group")
		return nil, err
	}
	role.Path = r.FormValue("path")
	return role, nil
}

func (h *IdmHandler) Info(ctx context.Context, w http.ResponseWriter) {
	tree, err := h.Props.IdmTreeBuilder.IdmRoleNodes(ctx)
	if err != nil {
		h.replyIDMError(ctx, w, err)
		return
	}

	treeInfo := tree.BuildIdmTreeInfo().Values[""].Roles
	response := &InfoResponse{Roles: treeInfo}
	h.reply(context.Background(), w, response)
}

func (h *IdmHandler) GetRoles(ctx context.Context, w http.ResponseWriter) {
	roles, err := h.Props.RoleBuilder.IdmRolesList(ctx, h.Props.StaffCache)
	if err != nil {
		h.replyIDMError(ctx, w, err)
		return
	}

	response := &RolesResponse{Roles: roles}
	h.reply(context.Background(), w, response)
}

func (h *IdmHandler) processRoleOperation(
	ctx context.Context, role *idmclient.Role, roleOp acmodel.RoleOperation,
) error {
	roleNode := lib.NewRoleNode(role.Path)
	namespaceName := roleNode.NamespaceName
	source := role.RoleType()
	roleName := acmodel.RoleType(roleNode.Slug)
	var name string

	switch source {
	case taskletv2.PermissionsSubject_E_SOURCE_USER:
		name = role.Login
	case taskletv2.PermissionsSubject_E_SOURCE_ABC:
		if v, ok := h.Props.StaffCache.GroupIDCache.Load(role.Group); ok {
			val, o := v.(*staffmodel.GroupInfo)
			if !o {
				return xerrors.New(fmt.Sprintf("Bad staff group type stored in cache for group %v", role.Group))
			}
			name = val.URL
		} else {
			g, err := h.Props.StaffClient.GroupByID(ctx, role.Group)
			if err != nil {
				return xerrors.Errorf("Unable to resolve staff GID %v. Error: %w", role.Group, err)
			}
			h.Props.StaffCache.AddGroup(g)
			name = g.URL
		}
	case taskletv2.PermissionsSubject_E_SOURCE_INVALID:
		return xerrors.Errorf("Unknown role source for role %s", role.Path)
	}

	switch roleNode.Type {
	case lib.NamespaceRoleNode:
		namespace, err := h.Props.DB.GetNamespaceByName(ctx, namespaceName)
		if err != nil {
			return xerrors.Errorf("Namespace %s not found. Error: %w", namespaceName, err)
		}
		opErr := roleOp(roleName, name, source, &namespace.Meta.Permissions)
		if opErr != nil {
			return xerrors.Errorf(
				"Can't edit permissions for namespace %s. Error: %w",
				namespaceName,
				opErr,
			)
		}
		_, upErr := h.Props.DB.UpdateNamespace(
			ctx, namespace.Meta.Id, func(value *taskletv2.Namespace) error {
				value.Meta.Permissions = namespace.Meta.Permissions
				return nil
			},
		)
		if upErr != nil {
			return xerrors.Errorf("Can't update namespace %s. Error: %w", namespaceName, upErr)
		}
	case lib.TaskletRoleNode:
		taskletName := roleNode.TaskletName
		tasklet, err := h.Props.DB.GetTaskletByName(ctx, taskletName, namespaceName)
		if err != nil {
			return xerrors.Errorf("Tasklet %s:%s not found. Error: %w", namespaceName, taskletName, err)
		}
		opErr := roleOp(roleName, name, source, &tasklet.Meta.Permissions)
		if opErr != nil {
			return xerrors.Errorf(
				"Can't edit permissions for tasklet %s:%s. Error: %w",
				namespaceName,
				taskletName,
				opErr,
			)
		}
		_, upErr := h.Props.DB.UpdateTaskletOp(
			ctx, tasklet.Meta.Id, func(value *taskletv2.Tasklet) error {
				value.Meta.Permissions = tasklet.Meta.Permissions
				return nil
			},
		)
		if upErr != nil {
			return xerrors.Errorf("Can't update tasklet %s:%s. Error: %w", namespaceName, taskletName, upErr)
		}
	case lib.OtherNode:
		return xerrors.Errorf("Unknown role path %s", role.Path)
	}
	return nil
}

func (h *IdmHandler) AddRole(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	role, err := parseFormRole(r)
	if err != nil {
		h.replyIDMError(ctx, w, err)
		return
	}
	updErr := h.processRoleOperation(ctx, role, acmodel.AddRole)
	if updErr != nil {
		h.replyIDMError(ctx, w, updErr)
	}
	response := &BaseResponse{Code: 0}
	h.reply(context.Background(), w, response)
}

func (h *IdmHandler) RemoveRole(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	role, err := parseFormRole(r)
	if err != nil {
		h.replyIDMError(ctx, w, err)
		return
	}
	updErr := h.processRoleOperation(ctx, role, acmodel.RemoveRole)
	if updErr != nil {
		h.replyIDMError(ctx, w, updErr)
	}
	response := &BaseResponse{Code: 0}
	h.reply(context.Background(), w, response)
}

type RequestIDType string

const RequestIDHeader = RequestIDType("RequestId")

func (h *IdmHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	ctx := context.WithValue(context.Background(), RequestIDHeader, r.Header.Get("X-IDM-Request-Id"))
	ctxlog.Debugf(ctx, h.Props.Logger, "API IDM request with path %v.", r.URL.Path)

	switch h.Handler {
	case Info:
		h.Info(ctx, w)
	case GetRoles:
		h.GetRoles(ctx, w)
	case AddRole:
		h.AddRole(ctx, w, r)
	case RemoveRole:
		h.RemoveRole(ctx, w, r)
	default:
		h.replyError(context.Background(), w, http.StatusBadRequest, "Unbound method.")
	}
}

func (is *IdmServer) Serve() error {
	is.Logger.Debugf("Starting IDM server with config %+v", is.Config)
	return is.HTTPServer.ListenAndServe()
}

func (is *IdmServer) Stop(ctx context.Context) error {
	return is.HTTPServer.Shutdown(ctx)
}
