package app

import (
	"context"
	"encoding/json"
	"net/http"
	"sort"
	"strconv"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"
	"go.uber.org/goleak"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	taskletApi "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/cmd/idm/configs"
	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/idmclient"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/services"
	idmtestutils "a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/testutils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	staffmodel "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/model"
	stafftestutls "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/testutils"

	"golang.org/x/net/context/ctxhttp"
)

type APISuite struct {
	idmtestutils.BaseSuite
	IdmServer        *IdmServer
	StaffGroupsCache *staff.StaffGroupsCache
}

func (es *APISuite) TearDownTest() {
	if es.IdmServer != nil {
		ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
		defer cancel()
		es.StaffGroupsCache.Stop()
		err := es.IdmServer.Stop(ctx)
		if err != nil {
			panic(err)
		}
	}
}

type AuthorizationMock struct {
}

func (auth *AuthorizationMock) AuthorizeHandler() func(next http.Handler) http.Handler {
	return func(next http.Handler) http.Handler {
		return next
	}
}

type TestIdmClient struct {
	Logger     log.Logger
	URL        string
	HTTPClient *http.Client
}

func (c *TestIdmClient) Do(ctx context.Context, req *http.Request) (*http.Response, error) {
	fields := ctxlog.ContextFields(ctx)

	fields = append(fields, log.String("url", req.URL.String()))
	fields = append(fields, log.String("method", req.Method))

	reqID := consts.NewRequestID().String()
	fields = append(fields, log.String("api_request_id", reqID))
	req.Header.Add("X-System-Request-Id", reqID)

	ctxlog.Debug(ctx, c.Logger, "sending request", fields...)
	rsp, err := ctxhttp.Do(ctx, c.HTTPClient, req)

	fields = ctxlog.ContextFields(ctx)

	if err == nil {
		fields = append(
			fields,
			log.Int("status_code", rsp.StatusCode),
		)
	}

	ctxlog.Debug(ctx, c.Logger, "received response", fields...)

	return rsp, err
}

func (c *TestIdmClient) Info() (*InfoResponse, error) {
	requestURL := c.URL + "/info/"
	rsp, err := lib.Get(c, context.Background(), requestURL)

	if err != nil {
		return nil, err
	}

	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return nil, err
	}

	var reply InfoResponse
	if err = json.NewDecoder(rsp.Body).Decode(&reply); err != nil {
		return nil, err
	}
	return &reply, nil
}

func (c *TestIdmClient) GetRoles() (*RolesResponse, error) {
	requestURL := c.URL + "/get-roles/"
	rsp, err := lib.Get(c, context.Background(), requestURL)

	if err != nil {
		return nil, err
	}

	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return nil, err
	}

	var reply RolesResponse
	if err = json.NewDecoder(rsp.Body).Decode(&reply); err != nil {
		return nil, err
	}
	return &reply, nil
}

func (c *TestIdmClient) changeRole(requestURL string, role *idmclient.Role) (*BaseResponse, error) {
	if role.Login != "" {
		requestURL += "&login=" + role.Login
	} else {
		requestURL += "&group=" + strconv.Itoa(role.Group)
	}
	rsp, err := lib.Post(c, context.Background(), requestURL, []byte{})
	if err != nil {
		return nil, err
	}
	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return nil, err
	}

	var reply BaseResponse
	if err = json.NewDecoder(rsp.Body).Decode(&reply); err != nil {
		return nil, err
	}
	return &reply, nil
}

func (c *TestIdmClient) AddRole(role *idmclient.Role) (*BaseResponse, error) {
	requestURL := c.URL + "/add-role/?path=" + role.Path
	return c.changeRole(requestURL, role)
}

func (c *TestIdmClient) RemoveRole(role *idmclient.Role) (*BaseResponse, error) {
	requestURL := c.URL + "/remove-role/?path=" + role.Path
	return c.changeRole(requestURL, role)
}

func (es *APISuite) BootstrapContextWithServer(ctx context.Context) (
	*testutils.ObjectGenerator,
	*configs.IdmServiceConfig,
	[]*taskletApi.Namespace,
	[]*taskletApi.Tasklet,
) {
	r, c, n, t := es.BootstrapContext(ctx)
	staffCLient := &stafftestutls.StaffClientMock{Logger: es.Logger}
	es.StaffGroupsCache = staff.NewStaffGroupsCache(staffCLient)
	es.IdmServer = NewIdmServer(
		&configs.Config{IdmConfig: c},
		es.Logger,
		es.TestStorage,
		nil,
		staffCLient,
		es.StaffGroupsCache,
		&AuthorizationMock{},
	)
	go func() {
		// NB: let grpc server initialize
		es.Logger.Infof("Starting IDM Server")
		err := es.IdmServer.Serve()
		if err != nil {
			es.Logger.Warnf("Server stopped. Err: %v", err)
		}
	}()
	es.Require().NoError(testutils.WaitForPort(time.Second, c.API.Port))

	return r, c, n, t
}

func (es *APISuite) makeIdmClient(conf *configs.IdmServiceConfig) *TestIdmClient {
	return &TestIdmClient{
		Logger:     es.Logger,
		URL:        "http://localhost:" + strconv.Itoa(conf.API.Port),
		HTTPClient: &http.Client{},
	}
}

func (es *APISuite) TestInfoHandler() {
	ctx := context.Background()
	_, conf, _, _ := es.BootstrapContextWithServer(ctx)
	c := es.makeIdmClient(conf)
	apiInfo, err := c.Info()
	es.NoError(err)
	apiInfoJSON, err := json.Marshal(apiInfo.Roles)
	es.NoError(err)
	es.Equal(0, apiInfo.Code, apiInfo.Error)
	treeBuilder := services.IdmTreeBuilder{
		Logger: es.Logger,
		DB:     es.TestStorage,
	}
	idmTreeMap, err := treeBuilder.IdmRoleNodes(ctx)
	es.NoError(err)
	funcInfo := idmTreeMap.BuildIdmTreeInfo()
	funcInfoJSON, err := json.Marshal(funcInfo.Values[""].Roles)
	es.NoError(err)
	es.JSONEq(string(funcInfoJSON), string(apiInfoJSON))
}

func (es *APISuite) TestGetRoles() {
	ctx := context.Background()
	_, conf, _, _ := es.BootstrapContextWithServer(ctx)
	c := es.makeIdmClient(conf)
	apiRoles, err := c.GetRoles()
	es.NoError(err)
	es.Equal(0, apiRoles.Code, apiRoles.Error)
	apiRolesJSON, err := json.Marshal(apiRoles.Roles)
	es.NoError(err)
	roleBuilder := services.RoleBuilder{Logger: es.Logger, DB: es.TestStorage}
	funcRoles, err := roleBuilder.IdmRolesList(
		ctx, es.StaffGroupsCache,
	)
	es.NoError(err)
	funcRolesJSON, err := json.Marshal(funcRoles)
	es.NoError(err)
	es.JSONEq(string(funcRolesJSON), string(apiRolesJSON))
}

func checkPermission(
	role string,
	name string,
	source taskletApi.PermissionsSubject_ESource,
	permissions *taskletApi.Permissions,
) int {
	count := 0
	p := acmodel.GetPermissionsSubject(name, source, permissions)
	if p == nil {
		return 0
	}
	for _, r := range p.Roles {
		if r == role {
			count += 1
		}
	}
	return count
}

func (es *APISuite) formatRoleRequest(login string, group *staffmodel.GroupInfo, path string) *idmclient.Role {
	roleRequest := &idmclient.Role{
		Path: path,
	}
	if login != "" {
		roleRequest.Login = login
	} else {
		roleRequest.Group = group.ID
	}
	return roleRequest
}

func (es *APISuite) checkExpectedPermission(
	ctx context.Context, login string, group *staffmodel.GroupInfo, path string, expectedCount int,
) {
	var permissions *taskletApi.Permissions
	roleNode := lib.NewRoleNode(path)

	if roleNode.Type == lib.NamespaceRoleNode {
		ns, err := es.TestStorage.GetNamespaceByName(ctx, roleNode.NamespaceName)
		es.NoError(err)
		permissions = ns.Meta.Permissions
	} else {
		tl, err := es.TestStorage.GetTaskletByName(ctx, roleNode.TaskletName, roleNode.NamespaceName)
		es.NoError(err)
		permissions = tl.Meta.Permissions
	}
	if login != "" {
		es.Equal(
			expectedCount,
			checkPermission(
				roleNode.Slug,
				login,
				taskletApi.PermissionsSubject_E_SOURCE_USER,
				permissions,
			),
		)
	} else {
		es.Equal(
			expectedCount,
			checkPermission(
				roleNode.Slug,
				group.URL,
				taskletApi.PermissionsSubject_E_SOURCE_ABC,
				permissions,
			),
		)
	}
}

func (es *APISuite) processAddRole(
	ctx context.Context,
	c *TestIdmClient,
	login string,
	group *staffmodel.GroupInfo,
	path string,
) *idmclient.Role {
	roleRequest := es.formatRoleRequest(login, group, path)

	resp, err := c.AddRole(roleRequest)
	es.NoError(err)
	es.Equal(0, resp.Code, resp.Error)

	es.checkExpectedPermission(ctx, login, group, path, 1)
	return roleRequest
}

func (es *APISuite) processRemoveRole(
	ctx context.Context,
	c *TestIdmClient,
	login string,
	group *staffmodel.GroupInfo,
	path string,
) *idmclient.Role {
	roleRequest := es.formatRoleRequest(login, group, path)

	resp, err := c.RemoveRole(roleRequest)
	es.NoError(err)
	es.Equal(0, resp.Code, resp.Error)

	es.checkExpectedPermission(ctx, login, group, path, 0)
	return roleRequest
}

func sortRoles(roles []*idmclient.Role) {
	sort.Slice(
		roles, func(i, j int) bool {
			if roles[i].Path != roles[j].Path {
				return roles[i].Path < roles[j].Path
			}
			if roles[i].Login != roles[j].Login {
				return roles[i].Login < roles[j].Login
			}
			return roles[i].Group < roles[j].Group
		},
	)
}

func (es *APISuite) TestAddRole() {
	ctx := context.Background()
	_, conf, _, _ := es.BootstrapContextWithServer(ctx)
	c := es.makeIdmClient(conf)
	roleBuilder := services.RoleBuilder{Logger: es.Logger, DB: es.TestStorage}
	rolesBeforeChanges, err := roleBuilder.IdmRolesList(
		ctx, es.StaffGroupsCache,
	)
	es.NoError(err)

	var newRoles []*idmclient.Role
	// Check add user role to namespace
	newRoles = append(
		newRoles, es.processAddRole(
			ctx, c, testutils.UserMark, nil, "/namespace/arcadia/type/roles/role/NamespaceRead/",
		),
	)
	// Check duplicate user role to namespace
	es.processAddRole(
		ctx, c, testutils.UserMark, nil, "/namespace/arcadia/type/roles/role/NamespaceRead/",
	)
	// Check add group role to namespace
	newRoles = append(
		newRoles, es.processAddRole(
			ctx, c, "", testutils.GroupDungeon, "/namespace/taxi/type/roles/role/NamespaceOwner/",
		),
	)
	// Check duplicate group role to namespace
	es.processAddRole(
		ctx, c, "", testutils.GroupDungeon, "/namespace/taxi/type/roles/role/NamespaceOwner/",
	)

	// Check add user role to tasklet
	newRoles = append(
		newRoles, es.processAddRole(
			ctx, c, testutils.UserVan, nil,
			"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletRead/",
		),
	)
	// Check duplicate user role to tasklet
	es.processAddRole(
		ctx, c, testutils.UserVan, nil,
		"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletRead/",
	)
	// Check add group role to tasklet
	newRoles = append(
		newRoles, es.processAddRole(
			ctx, c, "", testutils.GroupDungeon,
			"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletWrite/",
		),
	)
	// Check duplicate group role to tasklet
	es.processAddRole(
		ctx, c, "", testutils.GroupDungeon,
		"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletWrite/",
	)
	// Check add role to tasklet with empty permissions
	newRoles = append(
		newRoles, es.processAddRole(
			ctx, c, "", testutils.GroupJym,
			"/namespace/arcadia/type/tasklet/name/new_tasklet/role/TaskletWrite/",
		),
	)
	rolesAfterChanges, err := roleBuilder.IdmRolesList(
		ctx, es.StaffGroupsCache,
	)
	es.NoError(err)
	es.Equal(len(rolesBeforeChanges)+len(newRoles), len(rolesAfterChanges))
	expectedRoles := append(rolesBeforeChanges, newRoles...)

	sortRoles(expectedRoles)
	sortRoles(rolesAfterChanges)

	expectedRolesJSON, err := json.Marshal(expectedRoles)
	es.NoError(err)
	rolesAfterChangesJSON, err := json.Marshal(rolesAfterChanges)
	es.NoError(err)
	es.JSONEq(string(expectedRolesJSON), string(rolesAfterChangesJSON))
}

func (es *APISuite) TestRemoveRole() {
	ctx := context.Background()
	_, conf, _, _ := es.BootstrapContextWithServer(ctx)
	c := es.makeIdmClient(conf)
	roleBuilder := services.RoleBuilder{Logger: es.Logger, DB: es.TestStorage}
	rolesBeforeChanges, err := roleBuilder.IdmRolesList(
		ctx, es.StaffGroupsCache,
	)
	es.NoError(err)

	var removedRoles []*idmclient.Role
	// Check remove user role from namespace
	removedRoles = append(
		removedRoles, es.processRemoveRole(
			ctx, c, testutils.UserBilly, nil, "/namespace/arcadia/type/roles/role/NamespaceOwner/",
		),
	)
	// Check return user role and remove it in namespace
	es.processAddRole(
		ctx, c, testutils.UserBilly, nil, "/namespace/arcadia/type/roles/role/NamespaceOwner/",
	)
	es.processRemoveRole(
		ctx, c, testutils.UserBilly, nil, "/namespace/arcadia/type/roles/role/NamespaceOwner/",
	)
	// Check remove not exist user role from namespace
	es.processRemoveRole(
		ctx, c, testutils.UserBilly, nil, "/namespace/arcadia/type/roles/role/NamespaceOwner/",
	)
	// Check remove group role from namespace
	removedRoles = append(
		removedRoles, es.processRemoveRole(
			ctx, c, "", testutils.GroupDungeon, "/namespace/arcadia/type/roles/role/NamespaceRead/",
		),
	)

	// Check return group role and remove it in namespace
	es.processAddRole(
		ctx, c, "", testutils.GroupDungeon, "/namespace/arcadia/type/roles/role/NamespaceRead/",
	)
	es.processRemoveRole(
		ctx, c, "", testutils.GroupDungeon, "/namespace/arcadia/type/roles/role/NamespaceRead/",
	)
	// Check remove not exist group role from namespace
	es.processRemoveRole(
		ctx, c, "", testutils.GroupDungeon, "/namespace/arcadia/type/roles/role/NamespaceRead/",
	)

	// Check remove user role from tasklet
	removedRoles = append(
		removedRoles, es.processRemoveRole(
			ctx, c, testutils.UserMark, nil,
			"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletWrite/",
		),
	)
	// Check return group role and remove it in tasklet
	es.processAddRole(
		ctx, c, testutils.UserMark, nil,
		"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletWrite/",
	)
	es.processRemoveRole(
		ctx, c, testutils.UserMark, nil,
		"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletWrite/",
	)
	// Check remove not exist user role from tasklet
	es.processRemoveRole(
		ctx, c, testutils.UserMark, nil,
		"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletWrite/",
	)

	// Check remove group role from tasklet
	removedRoles = append(
		removedRoles, es.processRemoveRole(
			ctx, c, "", testutils.GroupJym,
			"/namespace/taxi/type/tasklet/name/taxi_test_tasklet/role/TaskletOwner/",
		),
	)
	// Check remove not exist group role from tasklet
	es.processRemoveRole(
		ctx, c, "", testutils.GroupJym,
		"/namespace/taxi/type/tasklet/name/taxi_test_tasklet/role/TaskletOwner/",
	)

	rolesAfterChanges, err := roleBuilder.IdmRolesList(
		ctx, es.StaffGroupsCache,
	)

	es.NoError(err)
	es.Equal(len(rolesBeforeChanges)-len(removedRoles), len(rolesAfterChanges))
	expectedRoles := append(rolesAfterChanges, removedRoles...)
	sortRoles(expectedRoles)
	sortRoles(rolesBeforeChanges)

	expectedRolesJSON, err := json.Marshal(expectedRoles)
	es.NoError(err)
	rolesBeforeChangesJSON, err := json.Marshal(rolesBeforeChanges)
	es.NoError(err)
	es.JSONEq(string(expectedRolesJSON), string(rolesBeforeChangesJSON))
}

func TestRoles(t *testing.T) {
	defer goleak.VerifyNone(t, goleak.IgnoreCurrent())
	s := &APISuite{}
	suite.Run(t, s)
}
