package idmclient

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"

	"golang.org/x/net/context/ctxhttp"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
)

type requester struct{}

var requesterKey requester

// func WithRequester(ctx context.Context, requester string) context.Context {
// 	return context.WithValue(ctx, &requesterKey, requester)
// }
//
// func WithoutRequester(ctx context.Context) context.Context {
// 	return context.WithValue(ctx, &requesterKey, "")
// }

func ContextRequester(ctx context.Context) (requester string, ok bool) {
	if v := ctx.Value(&requesterKey); v != nil {
		requester = v.(string)
		ok = requester != ""
	}
	return
}

type sessionID struct{}

var sessionIDKey sessionID

// func WithSessionID(ctx context.Context, sessionID string) context.Context {
//	return context.WithValue(ctx, &sessionIDKey, sessionID)
// }

func ContextSessionID(ctx context.Context) (sessionID string, ok bool) {
	if v := ctx.Value(&sessionIDKey); v != nil {
		sessionID = v.(string)
		ok = true
	}
	return
}

type Client struct {
	Logger      log.Logger
	TVMClient   tvm.Client
	TVMClientID tvm.ClientID
	URL         string
	Limit       int
	Silent      bool
	NoMeta      bool

	// MasterOnly makes IDM client execute queries via master.
	MasterOnly bool

	HTTPClient *http.Client
}

func (q *RolesQuery) Build(limit int) (url.Values, error) {
	if q.System == "" {
		return nil, xerrors.New(`idm: "system" parameter is required`)
	}

	if q.User == "" && q.Group == 0 && len(q.FieldsData) == 0 && q.Path == "" {
		return nil, xerrors.New(`idm: either "user", "group" or "fields_data" parameter is required`)
	}

	values := url.Values{}
	values.Add("system", q.System)

	if q.Path != "" {
		values.Add("path", q.Path)
	}
	if q.User != "" {
		values.Add("user", q.User)
		values.Add("ownership", "personal")
	}
	if q.Group != 0 {
		values.Add("group", fmt.Sprint(q.Group))
		values.Add("ownership", "group")
	}
	if q.State != "" {
		values.Add("state", q.State)
	}

	if len(q.FieldsData) != 0 {
		jsonFields, err := json.Marshal(q.FieldsData)
		if err != nil {
			return nil, err
		}

		values.Add("fields_data", string(jsonFields))
	}

	if limit != 0 {
		values.Add("limit", fmt.Sprint(limit))
	}

	return values, nil
}

type Reply struct {
	Meta struct {
		Limit      int     `json:"limit"`
		TotalCount int     `json:"total_count"`
		Next       *string `json:"next"`
	} `json:"meta"`

	Objects []RoleInfo `json:"objects"`
}

func (c *Client) Do(ctx context.Context, req *http.Request) (*http.Response, error) {
	fields := ctxlog.ContextFields(ctx)

	requester, ok := ContextRequester(ctx)
	if ok {
		fields = append(fields, log.String("requester", requester))
	}

	fields = append(fields, log.String("url", req.URL.String()))
	fields = append(fields, log.String("method", req.Method))

	reqID := consts.NewRequestID().String()
	fields = append(fields, log.String("idm_api_request_id", reqID))
	req.Header.Add("X-System-Request-Id", reqID)

	if c.MasterOnly {
		req.Header.Add("X-Replicated-State", "master")
		fields = append(fields, log.String("replica", "master"))
	}

	sessionID, ok := ContextSessionID(ctx)
	if ok {
		req.AddCookie(&http.Cookie{Name: "Session_id", Value: sessionID})
		fields = append(fields, log.Bool("stolen_cookies", true))
	} else {
		ticket, err := c.TVMClient.GetServiceTicketForID(ctx, c.TVMClientID)
		if err != nil {
			return nil, xerrors.Errorf("Error on getting TVM service ticket. Error: %w", err)
		}
		req.Header.Add("X-Ya-Service-Ticket", ticket)
	}

	start := time.Now()

	ctxlog.Debug(ctx, c.Logger, "sending IDM request", fields...)
	rsp, err := ctxhttp.Do(ctx, c.HTTPClient, req)

	fields = ctxlog.ContextFields(ctx)
	fields = append(
		fields,
		log.Error(err),
		log.Duration("duration", time.Since(start)),
	)

	if err == nil {
		fields = append(
			fields,
			log.Int("status_code", rsp.StatusCode),
		)
	}

	ctxlog.Debug(ctx, c.Logger, "received IDM response", fields...)

	return rsp, err
}

func (c *Client) fetchRoles(ctx context.Context, requestURL string) (roles []RoleInfo, next *string, err error) {
	var rsp *http.Response
	rsp, err = lib.Get(c, ctx, requestURL)
	if err != nil {
		return
	}
	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return
	}

	var reply Reply
	if err = json.NewDecoder(rsp.Body).Decode(&reply); err != nil {
		return
	}
	return reply.Objects, reply.Meta.Next, nil
}

func (c *Client) SelectRoles(ctx context.Context, query RolesQuery) (roles []RoleInfo, err error) {
	var requestQuery url.Values
	if requestQuery, err = query.Build(c.Limit); err != nil {
		return
	}

	requestURL := fmt.Sprintf("%s/api/v1/roles/?%s", c.URL, requestQuery.Encode())
	for {
		batch, next, err := c.fetchRoles(ctx, requestURL)
		if err != nil {
			return nil, err
		}

		roles = append(roles, batch...)
		if next != nil {
			requestURL = fmt.Sprint(c.URL, *next)
		} else {
			break
		}
	}

	return
}

func (c *Client) RequestRole(ctx context.Context, request *RoleRequest) (role *RoleInfo, err error) {
	if requester, ok := ContextRequester(ctx); ok {
		request.Requester = requester
	}

	request.Silent = c.Silent
	if c.NoMeta {
		request.NoMeta = true
	}

	reqJSON, err := json.Marshal(request)
	if err != nil {
		return
	}

	rsp, err := lib.Post(
		c,
		ctxlog.WithFields(ctx, log.Any("role_request", request)),
		fmt.Sprintf("%s/api/v1/rolerequests/", c.URL),
		reqJSON,
	)
	if err != nil {
		return nil, err
	}
	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return
	}

	role = &RoleInfo{}
	err = json.NewDecoder(rsp.Body).Decode(role)
	return
}

func (c *Client) DeleteRole(ctx context.Context, roleID int, comment string) (err error) {
	body := map[string]string{
		"comment": comment,
	}

	if requester, ok := ContextRequester(ctx); ok {
		body["_requester"] = requester
	}

	bodyJS, _ := json.Marshal(body)

	rsp, err := lib.Delete(c, ctx, fmt.Sprintf("%s/api/v1/roles/%d/", c.URL, roleID), bodyJS)
	if err != nil {
		return err
	}
	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return
	}

	return
}

func (c *Client) EditRoleNode(ctx context.Context, system, path string, node RoleValue) (err error) {
	node.Slug = ""
	request := struct {
		RoleValue
		Create bool `json:"create"`
	}{
		node,
		true,
	}

	reqJSON, err := json.Marshal(request)
	if err != nil {
		return
	}

	rsp, err := lib.Put(c, ctx, fmt.Sprintf("%s/api/v1/rolenodes/%s%s", c.URL, system, path), reqJSON)
	if err != nil {
		return
	}
	defer func() { _ = rsp.Body.Close() }()

	err = lib.CheckStatusCode(rsp)
	return
}

func (c *Client) RemoveRoleNode(ctx context.Context, system, path string) (err error) {
	rsp, err := lib.Delete(c, ctx, fmt.Sprintf("%s/api/v1/rolenodes/%s%s", c.URL, system, path), nil)
	if err != nil {
		return
	}
	defer func() { _ = rsp.Body.Close() }()

	if rsp.StatusCode != http.StatusNotFound {
		err = lib.CheckStatusCode(rsp)
	}
	return
}

func (c *Client) DoBatchRequest(ctx context.Context, queries ...*BatchQuery) (err error) {
	var requests []*BatchRequest
	var responses []*BatchResponse
	for _, q := range queries {
		requests = append(requests, q.Request)
		responses = append(responses, q.Response)
	}

	reqJSON, err := json.Marshal(requests)
	if err != nil {
		return
	}

	requestURL := fmt.Sprintf("%s/api/v1/batch/", c.URL)
	rsp, err := lib.Post(c, ctx, requestURL, reqJSON)
	if err != nil {
		return
	}
	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return
	}

	resp := &struct {
		Responses []*BatchResponse
	}{
		Responses: responses,
	}

	err = json.NewDecoder(rsp.Body).Decode(&resp)
	return
}

func (c *Client) ListRoleNodes(ctx context.Context, system, slugPath string) (nodes []*RoleNodeInfo, err error) {
	requestQuery := &url.Values{}
	requestQuery.Add("system", system)
	requestQuery.Add("slug_path", slugPath)

	requestURL := fmt.Sprintf("%s/api/v1/rolenodes/?%s", c.URL, requestQuery.Encode())
	for {
		reply, err := c.fetchRoleNodes(ctx, requestURL)
		if err != nil {
			return nil, err
		}

		nodes = append(nodes, reply.Objects...)

		if next := reply.Meta.Next; next != nil {
			requestURL = fmt.Sprint(c.URL, *next)
		} else {
			break
		}
	}

	return
}

func (c *Client) fetchRoleNodes(ctx context.Context, url string) (reply ListRoleNodesReply, err error) {
	rsp, err := lib.Get(c, ctx, url)
	if err != nil {
		return
	}
	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return
	}

	err = json.NewDecoder(rsp.Body).Decode(&reply)
	if err != nil {
		return ListRoleNodesReply{}, xerrors.Errorf("unable to decode response: %w", err)
	}

	return
}

func (c *Client) GetRoleNode(ctx context.Context, systemSlug, roleSlug string) (v *RoleValue, err error) {
	requestURL := fmt.Sprintf("%s/api/v1/rolenodes/%s%s", c.URL, systemSlug, roleSlug)
	rsp, err := lib.Get(c, ctx, requestURL)
	if err != nil {
		return nil, err
	}
	defer func() { _ = rsp.Body.Close() }()

	if err := lib.CheckStatusCode(rsp); err != nil {
		return nil, err
	}

	var reply RoleValue
	err = json.NewDecoder(rsp.Body).Decode(&reply)
	if err != nil {
		return nil, fmt.Errorf("unable to decode role node: %w", err)
	}
	return &reply, nil
}
