package idmclient

import (
	"reflect"

	taskletApi "a.yandex-team.ru/tasklet/api/v2"
)

type LocalizedString struct {
	En string `json:"en"`
	Ru string `json:"ru"`
}

func (l *LocalizedString) Equal(other *LocalizedString) bool {
	return l.En == other.En && l.Ru == other.Ru
}

func NewLocalizedString(s string) LocalizedString {
	return LocalizedString{
		En: s,
		Ru: s,
	}
}

type RolesQuery struct {
	System string

	Path       string
	User       string
	Group      int
	State      string
	FieldsData map[string]string
}

type RoleInfo struct {
	ID int `json:"id"`

	Group *struct {
		ID int `json:"id"`
	} `json:"group"`

	User *struct {
		Username string `json:"username"`
	} `json:"user"`

	System struct {
		Slug string `json:"slug"`
	} `json:"system"`

	State string `json:"state"`

	Node struct {
		ValuePath string            `json:"value_path"`
		Data      map[string]string `json:"data"`
	} `json:"node"`

	ExpireAt string `json:"expire_at"`

	ReviewAt string `json:"review_at"`

	FieldsData map[string]string `json:"fields_data"`

	SystemSpecific map[string]string `json:"system_specific"`
}

type RoleRequest struct {
	System           string            `json:"system"`
	User             string            `json:"user,omitempty"`
	Group            int               `json:"group,omitempty"`
	Path             string            `json:"path"`
	FieldsData       map[string]string `json:"fields_data,omitempty"`
	Comment          string            `json:"comment,omitempty"`
	Silent           bool              `json:"silent,omitempty"`
	NoMeta           bool              `json:"no_meta,omitempty"`
	DepriveAt        string            `json:"deprive_at,omitempty"`
	DepriveAfterDays *int              `json:"deprive_after_days,omitempty"`

	Requester string `json:"_requester,omitempty"`
}

type RoleField struct {
	Slug     string          `json:"slug"`
	Name     LocalizedString `json:"name"`
	Type     string          `json:"type"`
	Required bool            `json:"required"`
}

type Responsible struct {
	Username string `json:"username"`
	Notify   bool   `json:"notify"`
}

type RoleTreeNode struct {
	Slug   string                `json:"slug"`
	Name   LocalizedString       `json:"name"`
	Help   LocalizedString       `json:"help"`
	Values map[string]*RoleValue `json:"values"`
}

type RoleValue struct {
	Name        LocalizedString `json:"name"`
	Slug        string          `json:"slug,omitempty"`
	Help        LocalizedString `json:"help"`
	UniqueID    string          `json:"unique_id,omitempty"`
	Visibility  *bool           `json:"visibility,omitempty"`
	Fields      []RoleField     `json:"fields,omitempty"`
	Responsible []Responsible   `json:"responsibilities,omitempty"`

	Roles *RoleTreeNode `json:"roles,omitempty"`
}

type BatchRequest struct {
	ID     string      `json:"id,omitempty"`
	Method string      `json:"method"`
	Path   string      `json:"path"`
	Body   interface{} `json:"body"`
}

type BatchResponse struct {
	ID         string            `json:"id,omitempty"`
	StatusCode int               `json:"status_code"`
	Headers    map[string]string `json:"headers"`
	Body       interface{}
}

type BatchQuery struct {
	Request  *BatchRequest
	Response *BatchResponse
}

type RoleNodeInfo struct {
	System struct {
		Slug string `json:"slug"`
	} `json:"system"`

	Name      LocalizedString `json:"name"`
	Slug      string          `json:"slug"`
	SlugPath  string          `json:"slug_path"`
	ValuePath string          `json:"value_path"`

	State    string `json:"state"`
	UniqueID string `json:"unique_id,omitempty"`

	Visibility *bool `json:"visibility,omitempty"`

	Fields           []RoleField   `json:"fields,omitempty"`
	Responsibilities []Responsible `json:"responsibilities,omitempty"`
}

func (i *RoleNodeInfo) Equal(other *RoleNodeInfo) bool {
	return i.SlugPath == other.SlugPath && i.ContentsEqual(other)
}

func (i *RoleNodeInfo) ContentsEqual(other *RoleNodeInfo) bool {
	return i.UniqueID == other.UniqueID &&
		i.Name.En == other.Name.En &&
		i.ValuePath == other.ValuePath &&
		i.Slug == other.Slug &&
		i.compareFields(other) &&
		i.compareResponsibilities(other)
}

func (i *RoleNodeInfo) compareFields(other *RoleNodeInfo) bool {
	if len(i.Fields) != len(other.Fields) {
		return false
	}

	fields := make(map[string]RoleField)
	for _, f := range i.Fields {
		fields[f.Slug] = f
	}

	for _, o := range other.Fields {
		f, ok := fields[o.Slug]
		if !ok {
			return false
		}
		if !reflect.DeepEqual(f, o) {
			return false
		}
	}

	return true
}

func (i *RoleNodeInfo) compareResponsibilities(other *RoleNodeInfo) bool {
	if len(i.Responsibilities) != len(other.Responsibilities) {
		return false
	}

	responsibilities := make(map[string]Responsible)
	for _, r := range i.Responsibilities {
		responsibilities[r.Username] = r
	}

	for _, o := range other.Responsibilities {
		r, ok := responsibilities[o.Username]
		if !ok {
			return false
		}
		if !reflect.DeepEqual(r, o) {
			return false
		}
	}

	return true
}

type ListRoleNodesReply struct {
	Meta struct {
		Limit      int     `json:"limit"`
		TotalCount int     `json:"total_count"`
		Next       *string `json:"next"`
	} `json:"meta"`

	Objects []*RoleNodeInfo `json:"objects"`
}

func (i *ListRoleNodesReply) CastToMap() map[string]*RoleNodeInfo {
	result := make(map[string]*RoleNodeInfo, 0)
	for _, info := range i.Objects {
		result[info.SlugPath] = info
	}
	return result
}

type Role struct {
	Path  string `json:"path"`
	Login string `json:"login,omitempty"`
	Group int    `json:"group,omitempty"`
}

func (r *Role) RoleType() taskletApi.PermissionsSubject_ESource {
	if r.Login != "" {
		return taskletApi.PermissionsSubject_E_SOURCE_USER
	}
	if r.Group != 0 {
		return taskletApi.PermissionsSubject_E_SOURCE_ABC
	}

	return taskletApi.PermissionsSubject_E_SOURCE_INVALID
}
