package lib

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
)

type HTTPError struct {
	ErrorCode string
	Msg       string
	Errors    map[string]interface{}
}

func (s *HTTPError) Error() string {
	if len(s.Errors) != 0 {
		return fmt.Sprintf("Message: %s: errors: %v", s.Msg, s.Errors)
	} else {
		return fmt.Sprintf("Message: %s", s.Msg)
	}
}

func CheckStatusCode(rsp *http.Response) error {
	if rsp.StatusCode/100 != 2 {
		body, _ := io.ReadAll(rsp.Body)

		var msg struct {
			Inner     map[string]interface{} `json:"errors"`
			Msg       string                 `json:"message"`
			ErrorCode string                 `json:"error_code"`
		}

		if json.Unmarshal(body, &msg) == nil && msg.Msg != "" {
			return &HTTPError{ErrorCode: msg.ErrorCode, Msg: msg.Msg, Errors: msg.Inner}
		}

		const limit = 4 * 1024
		if len(body) > limit {
			body = body[:limit]
		}

		return &HTTPError{Msg: fmt.Sprintf("Request failed: code=%d, body=%q", rsp.StatusCode, body)}
	}

	return nil
}

type Client interface {
	Do(ctx context.Context, req *http.Request) (*http.Response, error)
}

func Get(c Client, ctx context.Context, url string) (*http.Response, error) {
	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}
	return c.Do(ctx, req)
}

func Post(c Client, ctx context.Context, url string, body []byte) (*http.Response, error) {
	req, err := http.NewRequest(http.MethodPost, url, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/json")
	return c.Do(ctx, req)
}

func Put(c Client, ctx context.Context, url string, body []byte) (*http.Response, error) {
	req, err := http.NewRequest(http.MethodPut, url, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/json")
	return c.Do(ctx, req)
}

func Delete(c Client, ctx context.Context, url string, body []byte) (*http.Response, error) {
	req, err := http.NewRequest(http.MethodDelete, url, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}

	if len(body) != 0 {
		req.Header.Add("Content-Type", "application/json")
	}

	return c.Do(ctx, req)
}
