package model

import (
	"context"

	"a.yandex-team.ru/library/go/core/xerrors"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
)

type PermissionNode struct {
	Name   string
	Source taskletv2.PermissionsSubject_ESource
}

type TaskletNode struct {
	Tasklet     *taskletv2.Tasklet
	Permissions map[string][]*PermissionNode
}

type NamespaceNode struct {
	Namespace   *taskletv2.Namespace
	Tasklets    map[string]*TaskletNode
	Permissions map[string][]*PermissionNode
}

type DBObjects struct {
	Tasklets   []*taskletv2.Tasklet
	Namespaces []*taskletv2.Namespace
	Tree       map[string]*NamespaceNode
}

func NewDatabaseObjects(ctx context.Context, DB storage.IStorage) (*DBObjects, error) {
	namespaces, err := DB.ListNamespaces(ctx, "")
	if err != nil {
		return nil, xerrors.Errorf("Error on getting namespaces. Err: %w", err)
	}

	var tasklets []*taskletv2.Tasklet

	for _, ns := range namespaces {
		tls, err := DB.ListTasklets(ctx, ns.Meta.Name)
		if err != nil {
			return nil, xerrors.Errorf("Error on getting tasklets for namespace %v. Err: %w", ns.Meta.Name, err)
		}
		tasklets = append(tasklets, tls...)
	}
	objs := &DBObjects{Tasklets: tasklets, Namespaces: namespaces}
	objs.buildDatabaseObjectsTree()

	return objs, nil
}

func createPermissionNodes(permissions *taskletv2.Permissions) map[string][]*PermissionNode {
	pmNodes := map[string][]*PermissionNode{}
	for _, subj := range permissions.GetSubjects() {
		for _, role := range subj.Roles {
			p := &PermissionNode{
				Name:   subj.Name,
				Source: subj.Source,
			}
			if v, ok := pmNodes[role]; ok {
				pmNodes[role] = append(v, p)
			} else {
				pmNodes[role] = []*PermissionNode{p}
			}
		}
	}
	return pmNodes
}

func (objs *DBObjects) buildDatabaseObjectsTree() {
	objs.Tree = map[string]*NamespaceNode{}
	for _, namespace := range objs.Namespaces {
		nsNode := &NamespaceNode{
			Namespace:   namespace,
			Tasklets:    map[string]*TaskletNode{},
			Permissions: createPermissionNodes(namespace.Meta.Permissions),
		}
		objs.Tree[namespace.Meta.Name] = nsNode
	}

	for _, tasklet := range objs.Tasklets {
		tlNode := &TaskletNode{Tasklet: tasklet, Permissions: createPermissionNodes(tasklet.Meta.Permissions)}
		objs.Tree[tasklet.Meta.Namespace].Tasklets[tasklet.Meta.Name] = tlNode
	}
}
