package services

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/idmclient"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/model"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	staffmodel "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/model"
)

type RoleBuilder struct {
	Logger log.Logger
	DB     storage.IStorage
}

func buildObjectRoles(
	permissions *taskletv2.Permissions,
	roleSlugParent string,
	staffCache *staff.StaffGroupsCache,
) ([]*idmclient.Role, error) {
	var idmRoles []*idmclient.Role
	for _, p := range permissions.GetSubjects() {
		roleTemplate := idmclient.Role{}
		switch p.Source {
		case taskletv2.PermissionsSubject_E_SOURCE_USER:
			roleTemplate.Login = p.Name
		case taskletv2.PermissionsSubject_E_SOURCE_ABC:
			if v, ok := staffCache.GroupNameCache.Load(p.Name); ok {
				val, o := v.(*staffmodel.GroupInfo)
				if !o {
					return nil, xerrors.Errorf("Bad staff group type stored in cache for group %v", p.Name)
				}
				roleTemplate.Group = val.ID
			} else {
				continue
			}
		case taskletv2.PermissionsSubject_E_SOURCE_INVALID:
			continue
		}

		for _, roleName := range p.Roles {
			role := &idmclient.Role{
				Path:  roleSlugParent + roleName + "/",
				Login: roleTemplate.Login,
				Group: roleTemplate.Group,
			}
			idmRoles = append(idmRoles, role)
		}
	}
	return idmRoles, nil
}

func (rb *RoleBuilder) IdmRolesList(ctx context.Context, staffCache *staff.StaffGroupsCache) (
	[]*idmclient.Role,
	error,
) {
	dbObjects, dbErr := model.NewDatabaseObjects(ctx, rb.DB)
	if dbErr != nil {
		return nil, dbErr
	}

	err := staffCache.UpdateGroups(ctx, dbObjects)
	if err != nil {
		return nil, xerrors.Errorf("Error on updating staff cache. Err: %w", err)
	}

	var roles []*idmclient.Role
	for _, namespace := range dbObjects.Namespaces {
		extRoles, buildErr := buildObjectRoles(
			namespace.Meta.Permissions,
			lib.NamespaceRoleSlugParent(namespace),
			staffCache,
		)
		if buildErr != nil {
			return nil, xerrors.Errorf("Error on build namespaces object roles. Error: %w", buildErr)
		}
		roles = append(roles, extRoles...)
	}

	for _, tasklet := range dbObjects.Tasklets {
		extRoles, buildErr := buildObjectRoles(
			tasklet.Meta.Permissions,
			lib.TaskletRoleParent(tasklet),
			staffCache,
		)
		if buildErr != nil {
			return nil, xerrors.Errorf("Error on build tasklets object roles. Error: %w", buildErr)
		}
		roles = append(roles, extRoles...)
	}
	return roles, nil
}
