package services

import (
	"context"
	"fmt"
	"sort"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/tasklet/api/v2"
	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/idmclient"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/model"
)

type Role struct {
	Slug string
	Name string
}

var NamespaceRoles = []Role{
	{string(acmodel.NamespaceRead), "Namespace Read"},
	{string(acmodel.NamespaceOwner), "Namespace Owner"},
	{string(acmodel.CreateTasklet), "Create Tasklet"},
}

var TaskletRoles = []Role{
	{string(acmodel.TaskletRead), "Tasklet Read"},
	{string(acmodel.TaskletWrite), "Tasklet Write"},
	{string(acmodel.TaskletRun), "Tasklet Run"},
	{string(acmodel.TaskletOwner), "Tasklet Owner"},
}

type IdmTreeBuilder struct {
	Logger log.Logger
	DB     storage.IStorage
}

type IdmTree struct {
	Tree      map[string]*idmclient.RoleNodeInfo
	DBObjects *model.DBObjects
}

func (itb *IdmTreeBuilder) createNode(
	roleNodes map[string]*idmclient.RoleNodeInfo,
	name string,
	slugPath string,
	uniqueID string,
) {
	valuePath, slug := lib.ValueAndSlugFromSlugPath(slugPath)
	roleNodes[slugPath] = &idmclient.RoleNodeInfo{
		System: struct {
			Slug string `json:"slug"`
		}{},
		Name:             idmclient.NewLocalizedString(name),
		Slug:             slug,
		SlugPath:         slugPath,
		ValuePath:        valuePath,
		State:            "active",
		UniqueID:         uniqueID,
		Visibility:       ptr.Bool(true),
		Fields:           []idmclient.RoleField{},
		Responsibilities: []idmclient.Responsible{},
	}
}

func (itb *IdmTreeBuilder) NamespaceRoleNodes(
	roleNodes map[string]*idmclient.RoleNodeInfo,
	namespace *taskletv2.Namespace,
) error {
	itb.createNode(
		roleNodes,
		namespace.Meta.Name,
		fmt.Sprintf("/namespace/%s/", namespace.Meta.Name),
		"namespace_"+namespace.Meta.Id,
	)
	itb.createNode(roleNodes, "Type", fmt.Sprintf("/namespace/%s/type/", namespace.Meta.Name), "")
	itb.createNode(roleNodes, "Tasklet", fmt.Sprintf("/namespace/%s/type/tasklet/", namespace.Meta.Name), "")
	itb.createNode(roleNodes, "Tasklet name", fmt.Sprintf("/namespace/%s/type/tasklet/name/", namespace.Meta.Name), "")
	itb.createNode(roleNodes, "Namespace roles", fmt.Sprintf("/namespace/%s/type/roles/", namespace.Meta.Name), "")
	itb.createNode(roleNodes, "Role", fmt.Sprintf("/namespace/%s/type/roles/role/", namespace.Meta.Name), "")
	for _, role := range NamespaceRoles {
		itb.createNode(
			roleNodes,
			role.Name,
			lib.NamespaceRoleSlug(namespace, role.Slug),
			"",
		)
	}

	return nil
}

func (itb *IdmTreeBuilder) TaskletRoleNodes(
	roleNodes map[string]*idmclient.RoleNodeInfo,
	tasklet *taskletv2.Tasklet,
) error {
	itb.createNode(
		roleNodes,
		tasklet.Meta.Name,
		fmt.Sprintf("/namespace/%s/type/tasklet/name/%s/", tasklet.Meta.Namespace, tasklet.Meta.Name),
		"tasklet_"+tasklet.Meta.Id,
	)
	itb.createNode(
		roleNodes,
		"Role",
		fmt.Sprintf("/namespace/%s/type/tasklet/name/%s/role/", tasklet.Meta.Namespace, tasklet.Meta.Name),
		"",
	)

	for _, role := range TaskletRoles {
		itb.createNode(
			roleNodes,
			role.Name,
			lib.TaskletRoleSlug(tasklet, role.Slug),
			"",
		)
	}

	return nil
}

func (itb *IdmTreeBuilder) RootRoleNodes(roleNodes map[string]*idmclient.RoleNodeInfo) error {
	itb.createNode(roleNodes, "", "/", "")
	itb.createNode(roleNodes, "Namespace", "/namespace/", "")
	return nil
}

func (itb *IdmTreeBuilder) IdmRoleNodes(ctx context.Context) (*IdmTree, error) {
	dbObjects, dbErr := model.NewDatabaseObjects(ctx, itb.DB)
	if dbErr != nil {
		return nil, dbErr
	}
	var roleNodes = map[string]*idmclient.RoleNodeInfo{}
	err := itb.RootRoleNodes(roleNodes)
	if err != nil {
		ctxlog.Error(ctx, itb.Logger, "Error on creating root node", log.Error(err))
		return nil, err
	}

	for _, namespace := range dbObjects.Namespaces {
		err := itb.NamespaceRoleNodes(roleNodes, namespace)
		if err != nil {
			ctxlog.Errorf(
				ctx,
				itb.Logger,
				"Error on creating namespace '%s' node. Err: %v",
				namespace.Meta.Name,
				err,
			)
			return nil, err
		}
	}

	for _, tasklet := range dbObjects.Tasklets {
		err := itb.TaskletRoleNodes(roleNodes, tasklet)
		if err != nil {
			ctxlog.Errorf(ctx, itb.Logger, "Error on creating tasklet '%s' node. Err: %v", tasklet.Meta.Name, err)
			return nil, err
		}
	}
	return &IdmTree{Tree: roleNodes, DBObjects: dbObjects}, nil
}

type ComparedIdmTree struct {
	NodesToCreate []*idmclient.RoleNodeInfo
	NodesToEdit   []*idmclient.RoleNodeInfo
	NodesToDelete []*idmclient.RoleNodeInfo
}

func compareExternalNodes(
	tree map[string]*idmclient.RoleNodeInfo,
	commonNodes map[string]bool,
) []*idmclient.RoleNodeInfo {
	var result []*idmclient.RoleNodeInfo
	for k, v := range tree {
		if _, ok := commonNodes[k]; !ok {
			result = append(result, v)
		}
	}
	sort.Slice(
		result, func(i, j int) bool {
			return result[i].SlugPath < result[j].SlugPath
		},
	)
	return result
}

func (tree *IdmTree) CompareTree(otherTree map[string]*idmclient.RoleNodeInfo) *ComparedIdmTree {
	commonNodes := map[string]bool{}
	for k := range tree.Tree {
		if _, ok := otherTree[k]; ok {
			commonNodes[k] = true
		}
	}

	var result ComparedIdmTree
	result.NodesToCreate = compareExternalNodes(tree.Tree, commonNodes)
	result.NodesToDelete = compareExternalNodes(otherTree, commonNodes)
	result.NodesToEdit = []*idmclient.RoleNodeInfo{}

	for k := range commonNodes {
		v1 := tree.Tree[k]
		v2 := otherTree[k]
		if !v1.Equal(v2) {
			result.NodesToEdit = append(result.NodesToEdit, v1)
		}
	}
	sort.Slice(
		result.NodesToEdit, func(i, j int) bool {
			return result.NodesToEdit[i].SlugPath < result.NodesToEdit[j].SlugPath
		},
	)
	return &result
}

func (tree *IdmTree) BuildIdmTreeInfo() *idmclient.RoleTreeNode {
	var slugPathes []string
	for k := range tree.Tree {
		slugPathes = append(slugPathes, k)
	}
	sort.Strings(slugPathes)
	root := &idmclient.RoleTreeNode{Values: map[string]*idmclient.RoleValue{}}
	var lastValue *idmclient.RoleValue
	for _, slugPath := range slugPathes {
		splitSlugPath := strings.Split(slugPath, "/")
		splitSlugPath = splitSlugPath[:len(splitSlugPath)-2]
		treeNode := tree.Tree[slugPath]

		if len(splitSlugPath)&1 == 0 {
			node := root
			for idx := 0; idx < len(splitSlugPath); idx += 2 {
				node = node.Values[splitSlugPath[idx]].Roles
			}

			value := &idmclient.RoleValue{
				Name:        treeNode.Name,
				Slug:        treeNode.Slug,
				Help:        treeNode.Name,
				UniqueID:    treeNode.UniqueID,
				Visibility:  treeNode.Visibility,
				Fields:      treeNode.Fields,
				Responsible: treeNode.Responsibilities,
				Roles:       nil,
			}
			node.Values[treeNode.Slug] = value
			lastValue = value
		} else {
			lastValue.Roles = &idmclient.RoleTreeNode{
				Slug:   treeNode.Slug,
				Name:   treeNode.Name,
				Help:   treeNode.Name,
				Values: map[string]*idmclient.RoleValue{},
			}
		}
	}
	return root
}
