package services

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/cmd/idm/configs"
	"a.yandex-team.ru/tasklet/experimental/internal/locks"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/idmclient"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/model"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	staffmodel "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/model"
)

type IdmPusherClient interface {
	ListRoleNodes(ctx context.Context, system, slugPath string) (nodes []*idmclient.RoleNodeInfo, err error)
	EditRoleNode(ctx context.Context, system, path string, node idmclient.RoleValue) (err error)
	RemoveRoleNode(ctx context.Context, system, path string) (err error)
	RequestRole(ctx context.Context, request *idmclient.RoleRequest) (role *idmclient.RoleInfo, err error)
}

type TreePusher struct {
	Config         *configs.IdmServiceConfig
	Logger         log.Logger
	Client         IdmPusherClient
	IdmTreeBuilder *IdmTreeBuilder
	StaffCache     *staff.StaffGroupsCache
	locker         locks.Locker
	stop           chan struct{}
}

func NewTreePusher(
	conf *configs.IdmServiceConfig,
	logger log.Logger,
	client IdmPusherClient,
	treeBuilder *IdmTreeBuilder,
	staffCache *staff.StaffGroupsCache,
	locker locks.Locker,
) (*TreePusher, error) {
	return &TreePusher{
		Config:         conf,
		Logger:         logger,
		Client:         client,
		IdmTreeBuilder: treeBuilder,
		StaffCache:     staffCache,
		stop:           make(chan struct{}),
		locker:         locker,
	}, nil
}

type OperationType string

const (
	NodeCreate = OperationType("Create")
	NodeUpdate = OperationType("Update")
)

func (p *TreePusher) getRoleRequest(
	ctx context.Context,
	permNode *model.PermissionNode,
	roleNode *lib.RoleNode,
	DBObjects *model.DBObjects,
) (*idmclient.RoleRequest, error) {
	req := &idmclient.RoleRequest{
		System:  p.Config.Common.System,
		Path:    roleNode.ValuePath,
		Comment: "Automatic generated role for new object.",
	}
	switch permNode.Source {
	case taskletv2.PermissionsSubject_E_SOURCE_USER:
		req.User = permNode.Name
	case taskletv2.PermissionsSubject_E_SOURCE_ABC:
		v, ok := p.StaffCache.GroupNameCache.Load(permNode.Name)

		if !ok {
			err := p.StaffCache.UpdateGroups(ctx, DBObjects)
			if err != nil {
				return nil, err
			}
			v, _ = p.StaffCache.GroupNameCache.Load(permNode.Name)
		}
		req.Group = v.(*staffmodel.GroupInfo).ID
	}
	return req, nil
}

func (p *TreePusher) editIdmNode(
	ctx context.Context, node *idmclient.RoleNodeInfo, operation OperationType, DBObjects *model.DBObjects,
) error {
	ctxlog.Infof(ctx, p.IdmTreeBuilder.Logger, "%v idm node %s", operation, node.SlugPath)
	err := p.Client.EditRoleNode(
		ctx, p.Config.Common.System, node.SlugPath, idmclient.RoleValue{
			Name:        node.Name,
			Slug:        node.Slug,
			Help:        node.Name,
			UniqueID:    node.UniqueID,
			Visibility:  node.Visibility,
			Fields:      node.Fields,
			Responsible: node.Responsibilities,
			Roles:       nil,
		},
	)
	if err != nil {
		return xerrors.Errorf("Unable to %v idm node %v. Error: %w", operation, node.SlugPath, err)
	}

	if operation == NodeCreate {
		roleNode := lib.NewRoleNode(node.SlugPath)
		var permNodesArray []*model.PermissionNode
		exists := false
		switch roleNode.Type {
		case lib.NamespaceRoleNode:
			permNodesArray, exists = DBObjects.Tree[roleNode.NamespaceName].Permissions[node.Slug]
		case lib.TaskletRoleNode:
			permNodesArray, exists = DBObjects.Tree[roleNode.NamespaceName].Tasklets[roleNode.TaskletName].Permissions[node.Slug]
		}
		if exists {
			for _, permNode := range permNodesArray {
				req, err := p.getRoleRequest(ctx, permNode, roleNode, DBObjects)
				if err != nil {
					return xerrors.Errorf("Error on processing role %v. Error: %w", node.SlugPath, err)
				}
				_, err = p.Client.RequestRole(ctx, req)
				if err != nil {
					return xerrors.Errorf("Error on requesting role %v. Error: %w", node.SlugPath, err)
				}
			}
		}
	}

	return nil
}

func (p *TreePusher) UpdateIdmTree(ctx context.Context) error {
	idmTreeNodes, err := p.Client.ListRoleNodes(ctx, p.Config.Common.System, "/")
	if err != nil {
		return xerrors.Errorf("Can't get IDM role nodes from IDM API. Err: %w", err)
	}
	idmTree := map[string]*idmclient.RoleNodeInfo{}
	for _, node := range idmTreeNodes {
		idmTree[node.SlugPath] = node
	}

	localTree, idmTreeErr := p.IdmTreeBuilder.IdmRoleNodes(ctx)
	if idmTreeErr != nil {
		return xerrors.Errorf("\"Can't build IDM role nodes from database. Err: %w", idmTreeErr)
	}

	compareResult := localTree.CompareTree(idmTree)

	for _, node := range compareResult.NodesToDelete {
		ctxlog.Infof(ctx, p.IdmTreeBuilder.Logger, "Delete idm node %s", node.SlugPath)
		reqErr := p.Client.RemoveRoleNode(ctx, p.Config.Common.System, node.SlugPath)
		if reqErr != nil {
			return xerrors.Errorf("Unable to Delete idm node %v. Error: %w", node.SlugPath, reqErr)
		}
	}

	for _, node := range compareResult.NodesToCreate {
		reqErr := p.editIdmNode(ctx, node, NodeCreate, localTree.DBObjects)
		if reqErr != nil {
			return reqErr
		}
	}

	for _, node := range compareResult.NodesToEdit {
		reqErr := p.editIdmNode(ctx, node, NodeUpdate, localTree.DBObjects)
		if reqErr != nil {
			return reqErr
		}
	}
	return nil
}

func (p *TreePusher) Tick(ctx context.Context) <-chan struct{} {
	ctxlog.Info(ctx, p.Logger, "New tick")
	defer func() {
		ctxlog.Info(ctx, p.Logger, "Tick finished")
	}()
	tickCtx, tickCancel := context.WithCancel(ctx)

	go func() {
		defer tickCancel()
		err := p.UpdateIdmTree(ctx)
		if err != nil {
			ctxlog.Error(tickCtx, p.Logger, "Tick failed", log.Error(err))
		} else {
			ctxlog.Debug(tickCtx, p.Logger, "Tick done")
		}
	}()
	return tickCtx.Done()
}

func (p *TreePusher) Run() error {
	if !p.Config.IdmTree.Enabled {
		<-p.stop
		return nil
	}
	p.locker.Start()
	defer p.locker.Stop()
	timer := time.NewTimer(time.Nanosecond)

MainLoop:
	for {
		select {
		case <-p.stop:
			break MainLoop
		case <-timer.C:
			// respawn limit
			timer.Reset(time.Second * 30)
		}
		if !p.locker.IsLocked() {
			continue
		}

		tickCtx, cancel := context.WithCancel(context.Background())
		tickChan := p.Tick(tickCtx)
		select {
		case <-tickChan:
			cancel()
			// noop
		case <-p.stop:
			cancel()
			break MainLoop
		}

		select {
		case <-time.After(time.Second * time.Duration(p.Config.IdmTree.SyncPeriod)):
			// noop
		case <-p.stop:
			break MainLoop
		}
	}
	return nil
}

func (p *TreePusher) Stop() {
	close(p.stop)
}
