package services

import (
	"context"
	"encoding/json"
	"fmt"
	"sort"
	"testing"

	"a.yandex-team.ru/library/go/test/canon"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	stafftestutls "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/testutils"
	"github.com/stretchr/testify/suite"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	taskletApi "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/idmclient"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/services/testdata"
	idmtestutils "a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/testutils"
)

type RolesTreeSuite struct {
	idmtestutils.BaseSuite
}

type NamespaceNode struct {
	Namespace *taskletApi.Namespace
	Node      *idmclient.RoleNodeInfo
}

type TaskletNode struct {
	Tasklet *taskletApi.Tasklet
	Node    *idmclient.RoleNodeInfo
}

func getNamespaceNodes(namespaces []*taskletApi.Namespace, tree map[string]*idmclient.RoleNodeInfo) []*NamespaceNode {
	var result []*NamespaceNode
	for _, ns := range namespaces {
		slugPath := fmt.Sprintf("/namespace/%s/", ns.Meta.Name)
		if v, ok := tree[slugPath]; ok {
			result = append(result, &NamespaceNode{Namespace: ns, Node: v})
		}
	}
	return result
}

func setNamespaceUniqueID(namespaces []*taskletApi.Namespace, tree map[string]*idmclient.RoleNodeInfo) {
	for _, nsNode := range getNamespaceNodes(namespaces, tree) {
		(*nsNode.Node).UniqueID = "namespace_" + nsNode.Namespace.Meta.Id
	}
}

func getTaskletNodes(tasklets []*taskletApi.Tasklet, tree map[string]*idmclient.RoleNodeInfo) []*TaskletNode {
	var result []*TaskletNode
	for _, tasklet := range tasklets {
		slugPath := fmt.Sprintf("/namespace/%s/type/tasklet/name/%s/", tasklet.Meta.Namespace, tasklet.Meta.Name)
		if v, ok := tree[slugPath]; ok {
			result = append(result, &TaskletNode{Tasklet: tasklet, Node: v})
		}
	}
	return result
}

func setTaskletUniqueID(tasklets []*taskletApi.Tasklet, tree map[string]*idmclient.RoleNodeInfo) {
	for _, taskletNode := range getTaskletNodes(tasklets, tree) {
		(*taskletNode.Node).UniqueID = "tasklet_" + taskletNode.Tasklet.Meta.Id
	}
}

func (es *RolesTreeSuite) TestBuildIdmTreeCanon() {
	ctx := context.Background()
	_, _, namespaces, tasklets := es.BootstrapContext(ctx)
	treeNodesBuilder := IdmTreeBuilder{es.Logger, es.TestStorage}
	treeNodes, err := treeNodesBuilder.IdmRoleNodes(ctx)
	es.Suite.NoError(err)

	for _, nsNode := range getNamespaceNodes(namespaces, treeNodes.Tree) {
		es.Equal("namespace_"+nsNode.Namespace.Meta.Id, nsNode.Node.UniqueID)
		(*nsNode.Node).UniqueID = ""
	}
	for _, taskletNode := range getTaskletNodes(tasklets, treeNodes.Tree) {
		es.Equal("tasklet_"+taskletNode.Tasklet.Meta.Id, taskletNode.Node.UniqueID)
		(*taskletNode.Node).UniqueID = ""
	}

	var objects []*idmclient.RoleNodeInfo

	for _, node := range treeNodes.Tree {
		objects = append(objects, node)
	}
	sort.Slice(
		objects, func(i, j int) bool {
			return objects[i].SlugPath < objects[j].SlugPath
		},
	)

	canon.SaveJSON(es.T(), idmclient.ListRoleNodesReply{Objects: objects})
}

func (es *RolesTreeSuite) TestBuildIdmTreeInfoCanon() {
	ctx := context.Background()
	_, _, namespaces, tasklets := es.BootstrapContext(ctx)
	treeNodesBuilder := IdmTreeBuilder{es.Logger, es.TestStorage}
	treeNodes, err := treeNodesBuilder.IdmRoleNodes(ctx)
	es.Suite.NoError(err)

	for _, nsNode := range getNamespaceNodes(namespaces, treeNodes.Tree) {
		es.Equal("namespace_"+nsNode.Namespace.Meta.Id, nsNode.Node.UniqueID)
		(*nsNode.Node).UniqueID = ""
	}
	for _, taskletNode := range getTaskletNodes(tasklets, treeNodes.Tree) {
		es.Equal("tasklet_"+taskletNode.Tasklet.Meta.Id, taskletNode.Node.UniqueID)
		(*taskletNode.Node).UniqueID = ""
	}

	info := treeNodes.BuildIdmTreeInfo()

	canon.SaveJSON(es.T(), info)
}

func (es *RolesTreeSuite) TestBuildAndCompareIdmTree() {
	ctx := context.Background()
	_, _, namespaces, tasklets := es.BootstrapContext(ctx)
	treeNodesBuilder := IdmTreeBuilder{es.Logger, es.TestStorage}
	treeNodes, err := treeNodesBuilder.IdmRoleNodes(ctx)
	es.Suite.NoError(err)

	idmRespData := testdata.Get("test_idm_schema.json")
	var idmRespRoles idmclient.ListRoleNodesReply
	err = json.Unmarshal(idmRespData, &idmRespRoles)
	es.Suite.NoError(err)
	mappedIdmRespRoles := idmRespRoles.CastToMap()
	setNamespaceUniqueID(namespaces, mappedIdmRespRoles)
	setTaskletUniqueID(tasklets, mappedIdmRespRoles)

	compareResult := treeNodes.CompareTree(mappedIdmRespRoles)

	es.Suite.Equal(0, len(compareResult.NodesToCreate), "Nodes to create")
	es.Suite.Equal(0, len(compareResult.NodesToEdit), "Nodes to edit")
	es.Suite.Equal(0, len(compareResult.NodesToDelete), "Nodes to delete")

	wrongIdmRespData := testdata.Get("test_wrong_idm_schema.json")
	var wrongIdmRespRoles idmclient.ListRoleNodesReply
	es.Suite.NoError(json.Unmarshal(wrongIdmRespData, &wrongIdmRespRoles), "Load json")
	wrongMappedIdmRespRoles := wrongIdmRespRoles.CastToMap()
	setNamespaceUniqueID(namespaces, wrongMappedIdmRespRoles)
	setTaskletUniqueID(tasklets, wrongMappedIdmRespRoles)

	wrongCompareResult := treeNodes.CompareTree(wrongMappedIdmRespRoles)

	es.Suite.Equal(1, len(wrongCompareResult.NodesToCreate), "Nodes to Create")
	es.Suite.Equal(
		"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletOwner/",
		wrongCompareResult.NodesToCreate[0].SlugPath,
		"Created node",
	)

	es.Suite.Equal(1, len(wrongCompareResult.NodesToEdit), "Nodes to Edit")
	es.Suite.Equal(
		"/namespace/taxi/type/tasklet/name/taxi_test_tasklet/role/TaskletWrite/",
		wrongCompareResult.NodesToEdit[0].SlugPath,
		"Edited node",
	)

	es.Suite.Equal(1, len(wrongCompareResult.NodesToDelete), "Nodes to delete")
	es.Suite.Equal(
		"/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletFail/",
		wrongCompareResult.NodesToDelete[0].SlugPath,
		"Deleted nodes",
	)
}

type IdmCall struct {
	method      string
	system      string
	slugPath    string
	roleValue   *idmclient.RoleValue
	roleRequest *idmclient.RoleRequest
}

func (ic IdmCall) Equal(other *IdmCall) bool {
	if ic.method != other.method || ic.system != other.system || ic.slugPath != other.slugPath {
		return false
	}
	if ic.roleValue == nil || other.roleValue == nil {
		return ic.roleValue == other.roleValue
	}
	if ic.roleRequest == nil || other.roleRequest == nil {
		return ic.roleRequest == other.roleRequest
	}

	roleValueEq := ic.roleValue.Name.Equal(&other.roleValue.Name) && ic.roleValue.Slug == other.roleValue.Slug && ic.roleValue.Help == other.roleValue.Help
	roleRequestEq := ic.roleRequest.User == other.roleRequest.User && ic.roleRequest.Group == other.roleRequest.Group && ic.roleRequest.System == other.roleRequest.System && ic.roleRequest.Path == other.roleRequest.Path
	return roleValueEq && roleRequestEq
}

type MockIdmClient struct {
	Logger       log.Logger
	CallAudit    []*IdmCall
	TestTreePath string
	namespaces   []*taskletApi.Namespace
	tasklets     []*taskletApi.Tasklet
}

func (c *MockIdmClient) ListRoleNodes(_ context.Context, system, slugPath string) (
	nodes []*idmclient.RoleNodeInfo,
	err error,
) {
	c.CallAudit = append(c.CallAudit, &IdmCall{"ListRoleNodes", system, slugPath, nil, nil})
	var idmRespRoles idmclient.ListRoleNodesReply
	if c.TestTreePath != "" {
		idmRespData := testdata.Get(c.TestTreePath)
		err = json.Unmarshal(idmRespData, &idmRespRoles)
		if err != nil {
			return nil, err
		}
		mappedIdmRespRoles := idmRespRoles.CastToMap()
		setNamespaceUniqueID(c.namespaces, mappedIdmRespRoles)
		setTaskletUniqueID(c.tasklets, mappedIdmRespRoles)
	} else {
		idmRespRoles.Objects = []*idmclient.RoleNodeInfo{}
	}
	return idmRespRoles.Objects, nil
}

func (c *MockIdmClient) EditRoleNode(_ context.Context, system, path string, node idmclient.RoleValue) (err error) {
	c.CallAudit = append(c.CallAudit, &IdmCall{"EditRoleNode", system, path, &node, nil})
	return nil
}

func (c *MockIdmClient) RemoveRoleNode(_ context.Context, system, path string) (err error) {
	c.CallAudit = append(c.CallAudit, &IdmCall{"RemoveRoleNode", system, path, nil, nil})
	return nil
}

func (c *MockIdmClient) RequestRole(_ context.Context, request *idmclient.RoleRequest) (
	role *idmclient.RoleInfo,
	err error,
) {
	c.CallAudit = append(c.CallAudit, &IdmCall{"RequestRole", request.System, request.Path, nil, request})
	return nil, nil
}

func (es *RolesTreeSuite) TestTreePusher() {
	ctx := context.Background()
	_, testConfig, namespaces, tasklets := es.BootstrapContext(ctx)
	treeNodesBuilder := IdmTreeBuilder{es.Logger, es.TestStorage}
	audit := []*IdmCall{}
	idmMockClient := MockIdmClient{
		es.Logger,
		audit,
		"test_idm_schema.json",
		namespaces,
		tasklets,
	}

	staffClient := &stafftestutls.StaffClientMock{Logger: es.Logger}
	treePusher, tpErr := NewTreePusher(
		testConfig,
		es.Logger,
		&idmMockClient,
		&treeNodesBuilder,
		&staff.StaffGroupsCache{StaffClient: staffClient},
		nil,
	)
	es.Suite.NoError(tpErr)

	es.Suite.NoError(treePusher.UpdateIdmTree(ctx))
	es.Suite.Equal(1, len(idmMockClient.CallAudit))
	es.Suite.True(
		idmMockClient.CallAudit[0].Equal(
			&IdmCall{"ListRoleNodes", "tasklets", "/", nil, nil},
		),
	)

	audit = []*IdmCall{}
	idmMockClient = MockIdmClient{
		es.Logger,
		audit,
		"test_wrong_idm_schema.json",
		namespaces,
		tasklets,
	}
	nTreePusher, ntpErr := NewTreePusher(
		testConfig,
		es.Logger,
		&idmMockClient,
		&treeNodesBuilder,
		&staff.StaffGroupsCache{StaffClient: staffClient},
		nil,
	)
	es.Suite.NoError(ntpErr)

	es.Suite.NoError(nTreePusher.UpdateIdmTree(ctx))
	es.Suite.Equal(5, len(idmMockClient.CallAudit))
	es.Suite.True(
		idmMockClient.CallAudit[0].Equal(
			&IdmCall{"ListRoleNodes", "tasklets", "/", nil, nil},
		),
	)
	es.Suite.True(
		idmMockClient.CallAudit[1].Equal(
			&IdmCall{
				method:    "RemoveRoleNode",
				system:    "tasklets",
				slugPath:  "/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletFail/",
				roleValue: nil,
			},
		),
	)
	es.Suite.True(
		idmMockClient.CallAudit[2].Equal(
			&IdmCall{
				method:   "EditRoleNode",
				system:   "tasklets",
				slugPath: "/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletOwner/",
				roleValue: &idmclient.RoleValue{
					Name:        idmclient.NewLocalizedString("Tasklet Owner"),
					Slug:        "TaskletOwner",
					Help:        idmclient.NewLocalizedString("Tasklet Owner"),
					Visibility:  ptr.Bool(true),
					Fields:      nil,
					Responsible: nil,
					Roles:       nil,
				},
			},
		),
	)
	es.Suite.True(
		idmMockClient.CallAudit[3].Equal(
			&IdmCall{
				method:    "RequestRole",
				system:    "tasklets",
				slugPath:  "/arcadia/tasklet/test_tasklet/TaskletOwner/",
				roleValue: nil,
				roleRequest: &idmclient.RoleRequest{
					System: "tasklets",
					User:   testutils.UserBilly,
					Path:   "/namespace/arcadia/type/tasklet/name/test_tasklet/role/TaskletOwner/",
				},
			},
		),
	)
	es.Suite.True(
		idmMockClient.CallAudit[4].Equal(
			&IdmCall{
				method:   "EditRoleNode",
				system:   "tasklets",
				slugPath: "/namespace/taxi/type/tasklet/name/taxi_test_tasklet/role/TaskletWrite/",
				roleValue: &idmclient.RoleValue{
					Name:        idmclient.NewLocalizedString("Tasklet Write"),
					Slug:        "TaskletWrite",
					Help:        idmclient.NewLocalizedString("Tasklet Write"),
					Visibility:  ptr.Bool(true),
					Fields:      nil,
					Responsible: nil,
					Roles:       nil,
				},
			},
		),
	)

	idmMockClient = MockIdmClient{
		es.Logger,
		audit,
		"",
		namespaces,
		tasklets,
	}
	staffGroupsCache := staff.NewStaffGroupsCache(staffClient)
	defer staffGroupsCache.Stop()
	emptyTreePusher, etpErr := NewTreePusher(
		testConfig,
		es.Logger,
		&idmMockClient,
		&treeNodesBuilder,
		staffGroupsCache,
		nil,
	)
	es.Suite.NoError(etpErr)

	es.Suite.NoError(emptyTreePusher.UpdateIdmTree(ctx))
	es.Suite.Equal(50, len(idmMockClient.CallAudit))

	requestRoles := map[string][]*IdmCall{}

	fillRequestRoles := func(path string, name string, source taskletApi.PermissionsSubject_ESource) {
		roleRequest := &idmclient.RoleRequest{
			System: "tasklets",
			Path:   path,
		}
		if source == taskletApi.PermissionsSubject_E_SOURCE_USER {
			roleRequest.User = name
		} else {
			roleRequest.Group = testutils.GroupURLToID[name].ID
		}
		idcCall := &IdmCall{
			method:      "RequestRole",
			system:      "tasklets",
			slugPath:    path,
			roleValue:   nil,
			roleRequest: roleRequest,
		}
		v, ok := requestRoles[path]
		if !ok {
			v = []*IdmCall{}
			requestRoles[path] = []*IdmCall{}
		}
		requestRoles[path] = append(v, idcCall)
	}

	for _, namespace := range namespaces {
		for _, subj := range namespace.Meta.Permissions.GetSubjects() {
			for _, role := range subj.Roles {
				slug, _ := lib.ValueAndSlugFromSlugPath(lib.NamespaceRoleSlug(namespace, role))
				fillRequestRoles(slug, subj.Name, subj.Source)
			}
		}
	}

	for _, tasklet := range tasklets {
		for _, subj := range tasklet.Meta.Permissions.GetSubjects() {
			for _, role := range subj.Roles {
				slug, _ := lib.ValueAndSlugFromSlugPath(lib.TaskletRoleSlug(tasklet, role))
				fillRequestRoles(slug, subj.Name, subj.Source)
			}
		}
	}

	for _, auditItem := range idmMockClient.CallAudit {
		if auditItem.method != "RequestRole" {
			continue
		}
		id := -1
		audits := requestRoles[auditItem.slugPath]
		for idx, expectedItem := range audits {
			if expectedItem.Equal(auditItem) {
				id = idx
				break
			}
		}
		found := id >= 0
		es.True(found)
		audits[id] = audits[len(audits)-1]
		requestRoles[auditItem.slugPath] = audits[:len(audits)-1]
		if len(requestRoles[auditItem.slugPath]) == 0 {
			delete(requestRoles, auditItem.slugPath)
		}
	}
	es.Equal(0, len(requestRoles))
}

func TestRolesTree(t *testing.T) {
	s := &RolesTreeSuite{}
	suite.Run(t, s)
}
