package testutils

import (
	"context"
	"strings"
	"time"

	"github.com/stretchr/testify/suite"

	"a.yandex-team.ru/library/go/core/log"
	taskletApi "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/cmd/idm/configs"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/ydbstore"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

type BaseSuite struct {
	suite.Suite
	suiteClient *xydb.Client
	TestStorage *ydbstore.Storage
	root        string
	tmpdir      string
	Logger      log.Logger
}

func (es *BaseSuite) SetupSuite() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.suiteClient = xydb.MustGetYdbClient(ctx, nil, es.T().Name())
	es.root = es.suiteClient.Folder
	es.tmpdir = testutils.TwistTmpDir(es.T())
}

func (es *BaseSuite) SetupTest() {
	logger := testutils.TwistMakeLogger(es.tmpdir, strings.ReplaceAll(es.T().Name(), "/", "_")+".log")
	es.Logger = logger
	es.suiteClient.Folder = es.root + "/" + es.T().Name()
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.NoError(ydbstore.PurgeDatabase(ctx, es.suiteClient))
	es.NoError(ydbstore.CreateTables(ctx, es.suiteClient))
	es.TestStorage = ydbstore.NewStorage(es.suiteClient, logger)
	es.TestStorage.LogQueries = true
}

func (es *BaseSuite) TearDownSuite() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.NoError(es.suiteClient.Close(ctx))
}

func (es *BaseSuite) BootstrapContext(ctx context.Context) (
	*testutils.ObjectGenerator,
	*configs.IdmServiceConfig,
	[]*taskletApi.Namespace,
	[]*taskletApi.Tasklet,
) {
	r := es.Require()
	og := testutils.NewObjectGenerator()
	namespaces, tasklets := og.CreateTestObjectsInMemory()
	for _, ns := range namespaces {
		r.NoError(es.TestStorage.AddNamespace(ctx, ns))
	}
	for _, tl := range tasklets {
		r.NoError(es.TestStorage.AddTasklet(ctx, tl))
	}

	return og, &configs.IdmServiceConfig{
			Common: &configs.CommonConfig{System: "tasklets"},
			API: &configs.APIConfig{
				Port: 8001, // TODO: automatic choose free port
			},
		},
		namespaces, tasklets
}
