from __future__ import print_function

import argparse
import requests
import json
import time
import os
import workflow

from os.path import expanduser

from requests.exceptions import HTTPError

CLUSTERS = ["tasklets", "taskletstest"]

parser = argparse.ArgumentParser(description='Update IDM workflow.')
parser.add_argument('--cluster', type=str, action='append', default=["tasklets"], choices=CLUSTERS, help='Tasklet slug inside IDM')
parser.add_argument('--all', action="store_true", default=False, help='Update all clusters')
parser.add_argument('--confirm', action="store_true", default=False, help='Confirm workflow immediately')
parser.add_argument('--approve', type=str, help='JSON describing approve action')


def read_token():
    if "IDM_TOKEN" in os.environ:
        return os.environ["IDM_TOKEN"]

    return open(expanduser("~/.idm/token")).read().strip()


class Token(requests.auth.AuthBase):
    def __call__(self, r):
        r.headers["Authorization"] = "OAuth " + read_token()
        return r


def raise_for_status(res):
    try:
        res.raise_for_status()
    except HTTPError as e:
        if res.text:
            raise HTTPError('{} Error Message: {}'.format(e.message, res.text.encode('utf-8')))
        else:
            raise e


def url(path):
    return "https://idm-api.yandex-team.ru" + path


def get_workflow():
    prefix = "# {}\n".format(time.time())

    return {
        "workflow": prefix + workflow.WORKFLOW,
        "group_workflow": prefix + workflow.GROUP_WORKFLOW,
    }


def update_workflow(args, cluster, approve):
    rsp = requests.post(url("/api/frontend/systems/{}/clone_workflow/".format(cluster)), auth=Token())
    raise_for_status(rsp)

    workflow_id = rsp.json()["id"]
    approve[cluster] = workflow_id

    rsp = requests.post(
        url("/api/frontend/workflow/{}/".format(workflow_id)),
        auth=Token(),
        json=get_workflow())
    raise_for_status(rsp)

    rsp = requests.post(url("/api/frontend/workflow/{}/commit/".format(workflow_id)), json={"silent": True}, auth=Token())
    raise_for_status(rsp)


def approve_workflow(args, approve):
    for cluster, workflow_id in approve.items():
        rsp = requests.post(url("/api/frontend/workflow/{}/approve/".format(workflow_id)), json={"silent": True}, auth=Token())
        raise_for_status(rsp)


def main():
    args = parser.parse_args()

    approve = {}
    if args.all:
        for cluster in CLUSTERS:
            update_workflow(args, cluster, approve)
    else:
        for cluster in args.cluster:
            update_workflow(args, cluster, approve)

    if args.confirm:
        approve_workflow(args, approve)
    elif approve:
        print("Workflow updated, now run the following command:")
        print("tasklet-workflow {}--approve '{}'".format("--test " if args.test else "", json.dumps(approve)))

    if args.approve:
        approve_workflow(args, json.loads(args.approve))


if __name__ == "__main__":
    main()
