class AccessDenied(RuntimeError):
    pass


class AnyFrom(object):
    def __init__(self, users=None, priority=0):
        self.users = [(u, priority) for u in users] if users else []

    def __or__(self, other):
        result = AnyFrom()
        result.users = self.users + other.users
        return result

    def __eq__(self, other):
        return self.users == other.users

    def __repr__(self):
        return "AnyFrom({})".format(self.users)

    @property
    def approvers(self):
        return [u[0] for u in self.users]


def any_from(l, priority=None, notify=None):
    return AnyFrom(l, priority)


class User(object):
    def __init__(self, name, is_robot=False, is_tvm_app=False):
        self.username = name
        self.is_robot = is_robot
        self.is_tvm_app = is_tvm_app
        self.head_of_chains = []

    def __eq__(self, other):
        if other is None:
            return False

        return (self.username, self.is_robot) == (other.username, other.is_robot)

    def __repr__(self):
        return "User(\"{}\", is_robot={})".format(self.username, self.is_robot)

    @property
    def groups(self):
        result = []
        for g in KNOWN_GROUPS_MAP.values():
            if self in g.members:
                result.append(g)
                break
        return result


class Group(object):
    def __init__(self, _id, users=None, slug="", gtype="service"):
        users = users or []
        self.id = _id
        self.members = []
        self.type = gtype
        for user in users:
            self.members.append(User(user))
        self.slug = slug

    def __repr__(self):
        return "Group(slug={}, id={})".format(self.slug, self.id)

    def __eq__(self, other):
        self_users = sorted(self.members, key=lambda x: x.username)
        other_users = sorted(other.members, key=lambda x: x.username)
        return self_users == other_users

# Test users
USER_ALICE = User("alice")
USER_BOB = User("bob")
USER_CHARLY = User("charly")
USER_HERMES = User("robot-hermes", is_robot=True)

# System test users
ROBOT_TASKLETS = User("robot-tasklets", is_robot=True)
USER_ATOS = User("atos")
USER_PARTOS = User("partos")

USER_CHARLY.head_of_chains = any_from([USER_BOB])

TASKLET_GROUP = Group(249436, [USER_ATOS.username, USER_PARTOS.username], "tasklets")
KEK_GROUP = Group(1, [USER_ALICE.username, USER_BOB.username], "kek")

KNOWN_GROUPS = [TASKLET_GROUP, KEK_GROUP]
KNOWN_GROUPS_MAP = {g.id: g for g in KNOWN_GROUPS}


def groupify(name):
    if name in KNOWN_GROUPS_MAP:
        return KNOWN_GROUPS_MAP[name]
    else:
        raise RuntimeError("unknown group %s error" % str(name))


def userify(name):
    return User(name)
