package sandbox

import (
	"context"
	"time"

	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
)

type IGroupCache interface {
	Get(group string) (*UserGroup, error)
	Stop()
}

var (
	ErrNoGroup = xerrors.NewSentinel("Sandbox group does not exist")
)

type GroupCacheConfig struct {
	TTL       int   `yaml:"ttl" valid:"greater=0"`
	CacheSize int64 `yaml:"cache_size" valid:"greater=100"`
}

type GroupCache struct {
	cache *ccache.Cache
	ttl   time.Duration
	sbx   *Client
	l     log.Logger
}

func NewGroupCache(conf *GroupCacheConfig, sb *Client, l log.Logger) *GroupCache {
	c := ccache.New(
		ccache.Configure().MaxSize(conf.CacheSize),
	)
	return &GroupCache{
		cache: c,
		ttl:   time.Duration(conf.TTL) * time.Second,
		sbx:   sb,
		l:     l,
	}
}

func (c *GroupCache) Get(group string) (*UserGroup, error) {
	rv, err := c.cache.Fetch(
		group,
		c.ttl,
		func() (interface{}, error) {
			ctx, cancel := context.WithTimeout(context.TODO(), time.Duration(10)*time.Second)
			defer cancel()
			return c.sbx.GetGroup(ctx, group)
		},
	)
	if err != nil || rv == nil {
		if err != nil {
			c.l.Errorf("Failed fetching sandbox group: %+v", err)
		}

		// Maybe has expired value
		rv = c.cache.Get(group)
		if rv != nil {
			return rv.Value().(*UserGroup), nil
		} else {
			return nil, ErrNoGroup
		}

	}
	return rv.Value().(*UserGroup), nil
}

func (c *GroupCache) Stop() {
	if c.cache != nil {
		c.cache.Stop()
	}
}
