package sandbox

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"strings"
	"time"

	"github.com/go-openapi/runtime"
	httptransport "github.com/go-openapi/runtime/client"
	"github.com/go-openapi/strfmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/sandbox/common/go/clients/authenticate"
	"a.yandex-team.ru/sandbox/common/go/clients/user"
	"golang.org/x/exp/slices"

	sb "a.yandex-team.ru/sandbox/common/go/clients"
	"a.yandex-team.ru/sandbox/common/go/clients/batch"
	"a.yandex-team.ru/sandbox/common/go/clients/group"
	"a.yandex-team.ru/sandbox/common/go/clients/resource"
	"a.yandex-team.ru/sandbox/common/go/clients/task"
	"a.yandex-team.ru/sandbox/common/go/clients/yav"
	"a.yandex-team.ru/sandbox/common/go/models"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
)

var (
	defaultPriorityClass    = models.TaskPriorityClassSERVICE
	defaultPrioritySubclass = models.TaskPrioritySubclassNORMAL
	DefaultPriority         = models.TaskPriority{
		Class:    &defaultPriorityClass,
		Subclass: &defaultPrioritySubclass,
	}
)

var (
	ErrSandboxError         = xerrors.NewSentinel("sandbox call failed")
	ErrSandboxResponseParse = xerrors.NewSentinel("can not parse sandbox response")
	ErrSandboxNotFound      = xerrors.NewSentinel("object not found")
	// ErrSandboxTaskStart is permanent error. Task will not start under any circumstances
	ErrSandboxTaskStart = xerrors.NewSentinel("sandbox task error")
)

func validatePayload(payload runtime.Validatable) error {
	if payload == nil {
		return ErrSandboxResponseParse.Wrap(xerrors.New("nil payload"))
	}

	if err := payload.Validate(strfmt.Default); err != nil {
		return ErrSandboxResponseParse.Wrap(err)
	}
	return nil
}

type Client struct {
	conf   *Config
	logger log.Logger
	sbx    *sb.SandboxJSONAPI
	token  string
}

func New(conf *Config, token string, l log.Logger, mr metrics.Registry) (*Client, error) {
	var auth runtime.ClientAuthInfoWriter
	if token != "" {
		auth = httptransport.APIKeyAuth(
			consts.AuthorizationHeader,
			"header",
			string(consts.OAuthMethod)+" "+strings.TrimSpace(token),
		)
	}
	transport := httptransport.New(conf.Host, sb.DefaultBasePath, sb.DefaultSchemes)
	transport.DefaultAuthentication = auth
	transport.SetDebug(conf.TransportDebug)
	transport.Transport = newAugmentedTransport(conf, mr.WithPrefix("transport"))
	transport.SetLogger(&logWrapper{l.WithName("transport")})
	sbx := sb.New(transport, nil)
	return &Client{
		conf:   conf,
		logger: l,
		sbx:    sbx,
		token:  token,
	}, nil
}

// TryCreateTask constructs DRAFT task in idempotent way by using executionID as uniqueness key
func (c *Client) TryCreateTask(
	ctx context.Context,
	executionID consts.ExecutionID,
	data *models.TaskNew,
) (SandboxTaskID, error) {

	ctxlog.Infof(ctx, c.logger, "Creating task for execution %q", executionID.String())

	if err := validatePayload(data); err != nil {
		return 0, err
	}

	createParams := task.NewTaskListPostParams().
		WithDefaults().
		WithContext(ctx).
		WithData(data)

	createdTaskResp, err := c.sbx.Task.TaskListPost(createParams, nil)
	if err != nil {
		ctxlog.Errorf(ctx, c.logger, "Failed to create task. ExecutionId: %q", executionID)
		return 0, ErrSandboxError.Wrap(err)
	}
	if createdTaskResp.Payload == nil {
		ctxlog.Errorf(ctx, c.logger, "Null payload in create task response. ExecutionId: %q", executionID)
		return 0, ErrSandboxResponseParse.Wrap(xerrors.New("null payload"))
	}

	if createdTaskResp.Payload.ID == nil {
		ctxlog.Errorf(ctx, c.logger, "Null Task id in create task response. ExecutionId: %q", executionID)
		return 0, ErrSandboxResponseParse.Wrap(xerrors.New("null task id"))
	}

	ctxlog.Infof(ctx, c.logger, "Sandbox task #%v created for execution %q", *createdTaskResp.Payload.ID, executionID)

	return SandboxTaskID(*createdTaskResp.Payload.ID), nil
}

type batchOperation string

func (bo batchOperation) String() string {
	return string(bo)
}

var batchStart batchOperation = "start"
var batchStop batchOperation = "stop"

func (c *Client) StartTask(ctx context.Context, taskID SandboxTaskID) error {

	taskSpec, err := c.getSandboxTask(ctx, taskID)
	if err != nil {
		return ErrSandboxError.Wrap(err)
	}

	if taskSpec.Status != models.TaskAuditItemStatusDRAFT {
		return nil
	}
	return c.performBatchOpOnTask(ctx, taskID, batchStart, nil)
}

// NB: mining script:
// ${ARC}/sandbox/common/types$ ya py
// import sandbox.common.types.task as ctt
// for st in  ctt.Status:
//   print st, ctt.Status.can_switch(st, ctt.Status.STOPPING)

// ASSIGNED True
// ENQUEUED True
// ENQUEUING True
// EXECUTING True
// PREPARING True
// SUSPENDED True
// TEMPORARY True
// WAIT_OUT True
// WAIT_RES True
// WAIT_TASK True
// WAIT_TIME True

var sandboxStoppableStatuses = []string{
	models.TaskAuditItemStatusASSIGNED,
	models.TaskAuditItemStatusENQUEUED,
	models.TaskAuditItemStatusENQUEUING,
	models.TaskAuditItemStatusEXECUTING,
	models.TaskAuditItemStatusPREPARING,
	models.TaskAuditItemStatusSUSPENDED,
	models.TaskAuditItemStatusTEMPORARY,
	models.TaskAuditItemStatusWAITOUT,
	models.TaskAuditItemStatusWAITRES,
	models.TaskAuditItemStatusWAITTASK,
	models.TaskAuditItemStatusWAITTIME,
}

// https://a.yandex-team.ru/arc_vcs/sandbox/common/types/task.py?rev=r8088426#L400
var sandboxStatusToSuccess = map[string]bool{
	// Status.Group.SUCCEED
	models.TaskAuditItemStatusSUCCESS:     true,
	models.TaskAuditItemStatusRELEASING:   true,
	models.TaskAuditItemStatusRELEASED:    true,
	models.TaskAuditItemStatusNOTRELEASED: true,
	// Status.Group.SCHEDULER_FAILURE
	models.TaskAuditItemStatusFAILURE: false,
	// NB: is exception|no_res|timeout|expired final statuses?
	// https://a.yandex-team.ru/arcadia/sandbox/common/types/task.py?rev=r8088426#L293
	models.TaskAuditItemStatusEXCEPTION: false,
	models.TaskAuditItemStatusNORES:     false,
	models.TaskAuditItemStatusTIMEOUT:   false,
	models.TaskAuditItemStatusEXPIRED:   false,
	// S.P.E.C.I.A.L
	models.TaskAuditItemStatusDELETED: false, // NB: concurrency issues?
	// Status.Group.BREAK:
	// EXCEPTION = None -- already in FAILURE?
	// NO_RES = None -- already in FAILURE?
	// TIMEOUT = None  -- already in FAILURE?
	models.TaskAuditItemStatusSTOPPED: false,
	// EXPIRED = None -- already in FAILURE?
}

func (c *Client) StopTask(ctx context.Context, taskID SandboxTaskID, session SandboxExternalSession) error {

	taskSpec, err := c.getSandboxTask(ctx, taskID)
	if err != nil {
		return ErrSandboxError.Wrap(err)
	}

	if slices.Contains(
		[]string{
			models.TaskAuditItemStatusSTOPPED,
			models.TaskAuditItemStatusSTOPPING,
			models.TaskAuditNewExpectedStatusDRAFT,
		}, taskSpec.Status,
	) {
		return nil
	}

	if _, ok := sandboxStatusToSuccess[taskSpec.Status]; ok {
		// Observed sandbox terminal status
		return nil
	}

	// NB: Relevant sandbox code
	// https://a.yandex-team.ru/svn/trunk/arcadia/sandbox/yasandbox/api/json/batch.py?rev=r9639478#L176
	// https://a.yandex-team.ru/svn/trunk/arcadia/sandbox/yasandbox/api/json/batch.py?rev=r9639478#L144
	// https://a.yandex-team.ru/svn/trunk/arcadia/sandbox/yasandbox/controller/task.py?rev=r9649015#L138-148
	// https://a.yandex-team.ru/svn/trunk/arcadia/sandbox/yasandbox/controller/task.py?rev=r9665982#L1366-1371
	// Transitions in form from_status -> to_status
	//    assert to_status in transitions.get(from_status, [])
	// https://a.yandex-team.ru/svn/trunk/arcadia/sandbox/common/types/task.py?rev=r9685884#L261

	if !slices.Contains(sandboxStoppableStatuses, taskSpec.Status) {
		ctxlog.Warnf(ctx, c.logger, "Stopping sandbox task in unexpected state. State: %q", taskSpec.Status)
	}

	auth := httptransport.APIKeyAuth(
		consts.AuthorizationHeader,
		"header",
		string(consts.OAuthMethod)+" "+session.String(),
	)

	return c.performBatchOpOnTask(ctx, taskID, batchStop, auth)

}

// performBatchOpOnTask applies batchOperation on single task.
// Precondition: task exists and has valid state
func (c *Client) performBatchOpOnTask(
	ctx context.Context,
	taskID SandboxTaskID,
	op batchOperation,
	authInfo runtime.ClientAuthInfoWriter,
) error {
	ctxlog.Debugf(ctx, c.logger, "Batch operation %v on task #%v", op, taskID)
	req := batch.NewTasksOpPutParams().
		WithDefaults().
		WithContext(ctx).
		WithOperation(op.String()).
		WithData(
			&models.BatchData{
				ID: []int64{taskID.ToInt()},
			},
		)

	resp, err := c.sbx.Batch.TasksOpPut(req, authInfo)
	if err != nil {
		return ErrSandboxError.Wrap(err)
	}

	if resp == nil {
		return ErrSandboxResponseParse.Wrap(xerrors.New("null resp"))
	}

	if len(resp.Payload) != 1 {
		return ErrSandboxResponseParse.Wrap(
			xerrors.Errorf(
				"unexpected Payload length %v",
				len(resp.Payload),
			),
		)
	}
	item := resp.Payload[0]

	if err := validatePayload(item); err != nil {
		return err
	}
	switch *item.Status {
	case models.BatchResultStatusSUCCESS:
		return nil
	case models.BatchResultStatusWARNING:
		ctxlog.Infof(ctx, c.logger, "Got warning from sandbox: %q", item.Message)
		switch op {
		case batchStart:
			if strings.Contains(item.Message, "Task started successfully") {
				return nil
			}
		case batchStop:
			if strings.Contains(item.Message, "Task scheduled for stop") {
				return nil
			}
		}
		return ErrSandboxError.Wrap(
			xerrors.Errorf(
				"TaskID: %v, Status: %q, Message: %q",
				taskID,
				*item.Status,
				item.Message,
			),
		)
	default:
		return ErrSandboxTaskStart.Wrap(
			xerrors.Errorf(
				"TaskID: %v, Status: %q, Message: %q",
				taskID,
				*item.Status,
				item.Message,
			),
		)
	}
}

func (c *Client) getSandboxTask(ctx context.Context, taskID SandboxTaskID) (*models.Task, error) {
	ctxlog.Debugf(ctx, c.logger, "Get task #%v", taskID)
	taskGetResp, err := c.sbx.Task.TaskGet(
		task.NewTaskGetParams().
			WithDefaults().
			WithContext(ctx).
			WithID(taskID.ToInt()),
	)
	if err != nil {
		return nil, err
	}
	if taskGetResp == nil {
		return nil, xerrors.New("Nil response")
	}

	// NB: validation fails. Malformed sandbox swagger spec
	// Validate interesting fields by hand
	//
	// if err := validatePayload(taskGetResp.Payload); err != nil {
	// 	return "", false, false, err
	// }

	pl := taskGetResp.Payload
	if pl == nil {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("Nil payload"))
	}
	if pl.Status == "" {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("Nil status"))
	}

	if pl.ID == nil {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("Nil id"))
	}
	return pl, nil
}

func (c *Client) GetSandboxTaskStatus(ctx context.Context, taskID SandboxTaskID) (
	status string,
	finished bool,
	success bool,
	err error,
) {

	pl, err := c.getSandboxTask(ctx, taskID)
	if err != nil {
		return "", false, false, err
	}

	success, finished = sandboxStatusToSuccess[pl.Status]
	return pl.Status, finished, success, nil
}

func (c *Client) GetGroup(ctx context.Context, name string) (*UserGroup, error) {
	ctxlog.Debugf(ctx, c.logger, "Get group %v", name)
	getOk, err := c.sbx.Group.GroupGet(
		group.NewGroupGetParams().
			WithDefaults().
			WithContext(ctx).
			WithName(name),
		nil,
	)

	if err != nil {
		return nil, err
	}

	if getOk == nil {
		ctxlog.Errorf(ctx, c.logger, "Nil GetGroup response. GroupID: %q", name)
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("nil response"))
	}
	if err := validatePayload(getOk.Payload); err != nil {
		return nil, err
	}

	rv := &UserGroup{
		Name:      name,
		CacheTime: time.Now(),
		Members:   nil,
	}

	payload := getOk.GetPayload()
	rv.Members = append(rv.Members, payload.Members...)
	return rv, nil
}

type ResourceInfo struct {
	ID        int64
	ProxyLink string
	MD5       string // Not empty
}

func (c *Client) GetResourceInfo(ctx context.Context, resourceID int64) (*ResourceInfo, error) {
	ctxlog.Debugf(ctx, c.logger, "Get resource %v info", resourceID)
	resInfo, err := c.sbx.Resource.ResourceGet(
		resource.NewResourceGetParams().WithContext(ctx).WithID(resourceID),
	)

	rv := ResourceInfo{
		ID: resourceID,
	}
	if err != nil {
		return nil, err
	}

	if resInfo == nil {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("nil resInfo"))
	} else if payload := resInfo.Payload; payload == nil {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("nil resInfo.payload"))
	}
	payload := resInfo.Payload

	// Acquired payload. Checking attributes
	if payload.State == nil || *payload.State != models.ResourceStateREADY {
		state := "nil"
		if payload.State != nil {
			state = *payload.State
		}
		return nil, xerrors.Errorf("Resource has invalid state. Id: %v, State: %q", resourceID, state)
	}

	if payload.Multifile != nil && *payload.Multifile {
		return nil, xerrors.Errorf(
			"Resource has invalid attribute. Id: %v, Multifile: %v",
			resourceID,
			*payload.Multifile,
		)
	}

	// Locate proxy link
	if dataLinks := payload.HTTP; dataLinks == nil {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("nil resInfo.payload.http"))
	} else if proxyLink := dataLinks.Proxy; proxyLink == "" {
		return nil, ErrSandboxResponseParse.Wrap(
			xerrors.Errorf(
				"Empty proxy link. ResourceId: %v, Status: %v",
				resourceID,
				payload.State,
			),
		)
	} else {
		rv.ProxyLink = proxyLink
	}

	// Locate MD5
	if md5 := payload.Md5; md5 == "" {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("No md5 for resource"))
	} else {
		rv.MD5 = md5
	}

	return &rv, nil
}

// CopyResource copies resource from sandbox proxy to out Writer
// Note: method does not use swagger transport
func (c *Client) CopyResource(ctx context.Context, resource ResourceInfo, out io.Writer) error {
	req, err := http.NewRequest(http.MethodGet, resource.ProxyLink, nil)
	if err != nil {
		return err
	}
	if c.token != "" {
		req.Header.Add(consts.AuthorizationHeader, string(consts.OAuthMethod)+" "+c.token)
	}

	client := http.Client{}
	resp, err := client.Do(req.WithContext(ctx))
	if err != nil {
		return err
	}
	defer func() { _ = resp.Body.Close() }()

	n, err := io.Copy(out, resp.Body)
	ctxlog.Infof(ctx, c.logger, "Resource copy info. Success: %v, BytesProcessed: %v", err == nil, n)
	return err
}

func (c *Client) GetSecretData(ctx context.Context, secRequest SecretDataRequest) (
	map[string]SecretKeyValue,
	error,
) {

	sbSecretsReq := []*models.SecretAndVersion{
		{
			ID:      ptr.String(secRequest.ID),
			Version: secRequest.Version,
		},
	}

	reqParams := yav.NewSecretsDataPostParams().
		WithTimeout(time.Second * 10).
		WithContext(ctx).
		WithData(&models.SecretAndVersionList{Secrets: sbSecretsReq})

	ctxlog.Debugf(ctx, c.logger, "Get secret")
	resp, err := c.sbx.Yav.SecretsDataPost(reqParams, nil)
	if err != nil {
		msg := fmt.Sprintf("Failed to get secret %v:%v", secRequest.ID, secRequest.Version)
		newErr := xerrors.Errorf("%v: %w", msg, err)
		ctxlog.Error(ctx, c.logger, msg, log.Error(err))
		return nil, ErrSandboxError.Wrap(newErr)
	} else if resp.GetPayload() == nil {
		return nil, ErrSandboxError.Wrap(xerrors.New("invalid sandbox response"))
	}

	respItems := resp.Payload.Items
	if len(respItems) != 1 {
		return nil, ErrSandboxError.Wrap(
			xerrors.Errorf("got %v secrets instead of 1 requested", len(respItems)),
		)
	}
	rv := make(map[string]SecretKeyValue, len(respItems[0].Values))

	for _, value := range respItems[0].Values {
		if value.Key == nil || value.Value == nil {
			err := xerrors.Errorf(
				"Missing required fields in response. SecID: %q, SecVer: %q",
				secRequest.ID,
				secRequest.Version,
			)
			return nil, ErrSandboxError.Wrap(err)
		}
		rv[*value.Key] = SecretKeyValue{
			Value:    *value.Value,
			Encoding: value.Encoding,
		}
	}
	return rv, nil
}

func (c *Client) LookupResource(ctx context.Context, resType string, owner string, attrs map[string]string) (
	*ResourceInfo,
	error,
) {
	var attrsStr *string
	if len(attrs) != 0 {
		buf, err := json.Marshal(attrs)
		if err != nil {
			return nil, err
		}
		if len(buf) != 0 {
			attrsStr = ptr.String(string(buf))
		}
	}

	req := resource.NewResourceListGetParams().
		WithContext(ctx).
		WithType([]string{resType}).
		WithState([]string{models.ResourceStateREADY}).
		WithLimit(1).
		WithOwner(ptr.String(owner)).
		WithAttrs(attrsStr)

	ctxlog.Debugf(ctx, c.logger, "Lookup resource")
	resp, err := c.sbx.Resource.ResourceListGet(req)
	if err != nil {
		return nil, err
	}
	payload := resp.Payload
	if payload == nil {
		ctxlog.Error(ctx, c.logger, "Empty payload in response")
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("nil payload"))
	}
	if payload.Limit == nil {
		ctxlog.Warn(ctx, c.logger, "nil \"Limit\" in payload")
	} else if payload.Offset == nil {
		ctxlog.Warn(ctx, c.logger, "nil \"Offset\" in payload")
	} else if payload.Total == nil {
		ctxlog.Warn(ctx, c.logger, "nil \"Total\" in payload")
	} else {
		ctxlog.Infof(ctx, c.logger, "Limit: %v Offset: %v Total: %v", *payload.Limit, *payload.Offset, *payload.Total)
	}

	parseItem := func(itemPtr *models.ResourceListItem) (int64, error) {
		if itemPtr == nil {
			return 0, xerrors.New("nil item ptr")
		}
		item := *itemPtr
		if item.ID == nil {
			return 0, xerrors.New("nil ID")
		}
		resourceID := *item.ID

		if item.Type == nil {
			return 0, xerrors.New("nil Type")
		}
		if *item.Type != resType {
			return 0, xerrors.Errorf("type mismatch. Type: %q, ID: %v", *item.Type, resourceID)
		}
		if item.State == nil {
			return 0, xerrors.Errorf("nil state")
		}
		if *item.State != models.ResourceStateREADY {
			return 0, xerrors.Errorf("state mismatch. Type: %q, ID: %v", *item.State, resourceID)
		}
		if item.Owner == nil {
			return 0, xerrors.Errorf("nil owner")
		}
		if *item.Owner != owner {
			return 0, xerrors.Errorf("owner mismatch. Type: %q, ID: %v", *item.State, resourceID)
		}
		// No attrs check

		return resourceID, nil
	}

	if len(payload.Items) == 0 {
		return nil, xerrors.New("not found")
	}
	if len(payload.Items) > 1 {
		return nil, ErrSandboxResponseParse.Wrap(xerrors.New("multiple results"))
	}

	resourceID, err := parseItem(payload.Items[0])
	if err != nil {
		return nil, ErrSandboxResponseParse.Wrap(err)
	}
	return c.GetResourceInfo(ctx, resourceID)
}

func (c *Client) GetCurrentUser(ctx context.Context) (string, error) {
	ctxlog.Debugf(ctx, c.logger, "Get current user")
	resp, err := c.sbx.User.UserCurrentGet(
		user.NewUserCurrentGetParams().
			WithDefaults().
			WithContext(ctx),
	)
	if err != nil {
		return "", ErrSandboxError.Wrap(err)
	}

	if err := validatePayload(resp.GetPayload()); err != nil {
		return "", err
	}

	return *resp.Payload.Login, nil

}

func (c *Client) CreateExternalSession(
	ctx context.Context,
	execution consts.ExecutionID,
	login string,
	taskID SandboxTaskID,
) (SandboxExternalSession, error) {
	createRequest := &models.ExternalOAuthCreate{
		JobID:   ptr.String(execution.String()),
		Login:   ptr.String(login),
		Service: ptr.String(taskletServiceTesting),
		TTL:     ptr.Int64(int64((time.Hour * 24).Seconds())),
	}
	// NB: YT task unbound sessions
	if taskID != 0 {
		createRequest.SandboxTaskID = taskID.ToInt()
	}
	if err := createRequest.Validate(strfmt.Default); err != nil {
		return "", xerrors.Errorf("Request validation failed: %w", err)
	}
	ctxlog.Debugf(ctx, c.logger, "Create external session")
	resp, err := c.sbx.Authenticate.ExternalOauthTokenPost(
		authenticate.NewExternalOauthTokenPostParams().
			WithDefaults().
			WithContext(ctx).
			WithBody(createRequest),
		nil,
	)
	if err != nil {
		return "", ErrSandboxError.Wrap(err)
	}
	if err := validatePayload(resp.GetPayload()); err != nil {
		return "", err
	}

	return SandboxExternalSession(*resp.Payload.Token), nil
}

func (c *Client) GetExternalSession(
	ctx context.Context,
	session SandboxExternalSession,
) (ExternalSessionInfo, error) {
	request := &models.ExternalOAuthGet{
		Service: ptr.String(taskletServiceTesting),
		Token:   session.String(),
	}
	if err := request.Validate(strfmt.Default); err != nil {
		return ExternalSessionInfo{}, xerrors.Errorf("Request validation failed: %w", err)
	}
	ctxlog.Debugf(ctx, c.logger, "Get external session")
	resp, err := c.sbx.Authenticate.ExternalOauthSessionSearchPost(
		authenticate.NewExternalOauthSessionSearchPostParams().
			WithDefaults().
			WithContext(ctx).
			WithBody(request),
		nil,
	)
	if err != nil {
		return ExternalSessionInfo{}, ErrSandboxError.Wrap(err)
	}
	if err := validatePayload(resp.GetPayload()); err != nil {
		return ExternalSessionInfo{}, err
	}

	if *resp.Payload.Token != session.String() {
		return ExternalSessionInfo{}, xerrors.Errorf(
			"Invalid sandbox response. PK mismatch",
		)
	}

	return ExternalSessionInfo{
		Token:       session,
		TaskID:      SandboxTaskID(resp.Payload.SandboxTaskID),
		ExecutionID: consts.ExecutionID(*resp.Payload.JobID),
	}, nil
}

func (c *Client) SearchExternalSession(
	ctx context.Context,
	executionID consts.ExecutionID,
) (ExternalSessionInfo, error) {
	request := &models.ExternalOAuthGet{
		Service: ptr.String(taskletServiceTesting),
		JobID:   executionID.String(),
	}
	if err := request.Validate(strfmt.Default); err != nil {
		return ExternalSessionInfo{}, xerrors.Errorf("Request validation failed: %w", err)
	}
	ctxlog.Debugf(ctx, c.logger, "Search external session")
	resp, err := c.sbx.Authenticate.ExternalOauthSessionSearchPost(
		authenticate.NewExternalOauthSessionSearchPostParams().
			WithDefaults().
			WithContext(ctx).
			WithBody(request),
		nil,
	)
	if err != nil {
		switch err.(type) {
		case *authenticate.ExternalOauthSessionSearchPostNotFound:
			return ExternalSessionInfo{}, ErrSandboxNotFound
		default:
			return ExternalSessionInfo{}, ErrSandboxError.Wrap(err)
		}
	}
	if err := validatePayload(resp.GetPayload()); err != nil {
		return ExternalSessionInfo{}, err
	}

	if *resp.Payload.JobID != executionID.String() {
		return ExternalSessionInfo{}, xerrors.Errorf(
			"Invalid sandbox response. %q != %q",
			*resp.Payload.JobID,
			executionID.String(),
		)
	}

	return ExternalSessionInfo{
		Token:       SandboxExternalSession(*resp.Payload.Token),
		TaskID:      SandboxTaskID(resp.Payload.SandboxTaskID),
		ExecutionID: executionID,
	}, nil
}

func (c *Client) DeleteExternalSession(
	ctx context.Context,
	session SandboxExternalSession,
) error {
	request := &models.ExternalOAuthDelete{
		Service: ptr.String(taskletServiceTesting),
		Token:   ptr.String(session.String()),
	}
	if err := request.Validate(strfmt.Default); err != nil {
		return xerrors.Errorf("Request validation failed: %w", err)
	}
	ctxlog.Debugf(ctx, c.logger, "Delete external session")
	resp, err := c.sbx.Authenticate.ExternalOauthTokenDelete(
		authenticate.NewExternalOauthTokenDeleteParams().
			WithDefaults().
			WithContext(ctx).
			WithBody(request),
		nil,
	)

	if resp != nil {
		// NB: 204 -- no content
		return nil
	}
	if _, ok := err.(*authenticate.ExternalOauthTokenDeleteNotFound); ok {
		ctxlog.Warn(ctx, c.logger, "External session not found")
		return nil
	}
	return ErrSandboxError.Wrap(err)
}

func (c *Client) SearchSession(
	ctx context.Context,
	session SandboxSession,
) (SessionInfo, error) {
	request := &models.TaskSessionGet{
		Token: ptr.String(session.String()),
	}
	if err := request.Validate(strfmt.Default); err != nil {
		return SessionInfo{}, xerrors.Errorf("Request validation failed: %w", err)
	}
	ctxlog.Debugf(ctx, c.logger, "Search session")
	resp, err := c.sbx.Authenticate.TaskSessionSearchPost(
		authenticate.NewTaskSessionSearchPostParams().
			WithDefaults().
			WithContext(ctx).
			WithBody(request),
		nil,
	)
	if err != nil {
		switch err.(type) {
		case *authenticate.TaskSessionSearchPostNotFound:
			return SessionInfo{}, ErrSandboxNotFound
		default:
			return SessionInfo{}, ErrSandboxError.Wrap(err)
		}
	}
	if err := validatePayload(resp.GetPayload()); err != nil {
		return SessionInfo{}, err
	}

	if *resp.Payload.Token != session.String() {
		return SessionInfo{}, xerrors.New(
			"Invalid sandbox response: token mismatch",
		)
	}

	return SessionInfo{
		Token:  SandboxSession(*resp.Payload.Token),
		Login:  *resp.Payload.Login,
		TaskID: SandboxTaskID(*resp.Payload.TaskID),
	}, nil
}
