package sandbox

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"strings"
	"testing"

	httptransport "github.com/go-openapi/runtime/client"
	"github.com/go-openapi/strfmt"
	"github.com/jarcoal/httpmock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"

	"a.yandex-team.ru/library/go/core/log/nop"
	sb "a.yandex-team.ru/sandbox/common/go/clients"
	"a.yandex-team.ru/sandbox/common/go/clients/authenticate"
	"a.yandex-team.ru/sandbox/common/go/clients/batch"
	"a.yandex-team.ru/sandbox/common/go/clients/resource"
	"a.yandex-team.ru/sandbox/common/go/clients/task"
	"a.yandex-team.ru/sandbox/common/go/models"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox/corpus"
)

func NewCorpusResponder(t *testing.T, status int, request string, response string) httpmock.Responder {
	var inputBytes []byte
	var inputJSON any
	if request != "" {
		inputBytes = corpus.MustGet(request)
		if err := json.Unmarshal(inputBytes, &inputJSON); err != nil {
			panic(err)
		}
	}

	var httpResponse *http.Response
	if response != "" {
		outputBytes := corpus.MustGet(response)
		httpResponse = httpmock.NewBytesResponse(status, outputBytes)
	} else {
		httpResponse = httpmock.NewStringResponse(status, "")
	}

	return func(req *http.Request) (rsp *http.Response, err error) {
		httpResponse.Request = req
		defer func() {
			errClose := req.Body.Close()
			if errClose != nil {
				err = errClose
			}
		}()

		var requestJSON any
		if requestBytes, errRead := io.ReadAll(req.Body); errRead != nil {
			return nil, errRead
		} else if len(requestBytes) > 0 {
			errParse := json.Unmarshal(requestBytes, &requestJSON)
			if errParse != nil {
				return nil, errParse
			}
		}
		assert.Equal(t, inputJSON, requestJSON)
		return httpResponse, nil
	}
}

const sandboxHost = "sbx.yandex.ru"

func callPath(suffix string) string {
	if !strings.HasPrefix(suffix, "/") {
		suffix = "/" + suffix
	}
	return fmt.Sprintf("%s://%s%s%s", sb.DefaultSchemes[0], sandboxHost, sb.DefaultBasePath, suffix)
}

type AuthenticateTestSuite struct {
	suite.Suite
	client  *Client
	runtime *httptransport.Runtime
	tr      *httpmock.MockTransport
}

func (as *AuthenticateTestSuite) SetupSuite() {
	as.client = &Client{
		conf:   nil,
		logger: &nop.Logger{},
		sbx: &sb.SandboxJSONAPI{
			Authenticate: authenticate.New(nil, strfmt.Default),
		},
		token: "",
	}

}

func (as *AuthenticateTestSuite) SetupTest() {
	tmpdir := testutils.TwistTmpDir(as.T())
	logger := testutils.TwistMakeLogger(tmpdir, "client.log")
	as.client.logger = logger
	as.tr = httpmock.NewMockTransport()
	as.runtime = httptransport.New(sandboxHost, sb.DefaultBasePath, sb.DefaultSchemes)
	as.runtime.Transport = as.tr
	as.runtime.SetDebug(true)
	as.runtime.SetLogger(&logWrapper{logger.WithName("transport")})

	as.client.sbx.Authenticate.SetTransport(as.runtime)
}

func (as *AuthenticateTestSuite) TearDownTest() {
	as.client.logger = nil
	as.tr = nil
	as.runtime = nil
}

func (as *AuthenticateTestSuite) TestCreateExternalSessionOK() {
	tr := as.tr
	tr.RegisterResponder(
		http.MethodPost,
		callPath("/authenticate/external/session"),
		NewCorpusResponder(
			as.T(),
			http.StatusOK,
			"authenticate_external_session_post.json",
			"authenticate_external_session_post_ok.json",
		),
	)

	rv, err := as.client.CreateExternalSession(
		context.Background(),
		"0198d50c-4a23-4159-bee0-8e87e6456eff",
		"alximik",
		1315697012,
	)
	as.NoError(err)
	as.Equal(SandboxExternalSession("5e66dd513d2846dca9f01f0dfacb860f"), rv)
	as.Equal(1, tr.GetTotalCallCount())
}

func (as *AuthenticateTestSuite) TestDeleteExternalSessionOK() {
	tr := as.tr
	tr.RegisterResponder(
		http.MethodDelete,
		callPath("/authenticate/external/session"),
		NewCorpusResponder(
			as.T(),
			http.StatusNoContent,
			"authenticate_external_session_delete.json",
			"",
		),
	)
	err := as.client.DeleteExternalSession(
		context.Background(),
		"5e66dd513d2846dca9f01f0dfacb860f",
	)
	as.NoError(err)
	as.Equal(1, tr.GetTotalCallCount())
}

func (as *AuthenticateTestSuite) TestDeleteExternalSessionNotFound() {
	tr := as.tr
	tr.RegisterResponder(
		http.MethodDelete,
		callPath("/authenticate/external/session"),
		NewCorpusResponder(
			as.T(),
			http.StatusNotFound,
			"authenticate_external_session_delete.json",
			"",
		),
	)
	err := as.client.DeleteExternalSession(
		context.Background(),
		"5e66dd513d2846dca9f01f0dfacb860f",
	)
	as.NoError(err)
	as.Equal(1, tr.GetTotalCallCount())
}

func (as *AuthenticateTestSuite) TestGetExternalSession() {
	tr := as.tr
	tr.RegisterResponder(
		http.MethodPost,
		callPath("/authenticate/external/session/search"),
		NewCorpusResponder(
			as.T(),
			http.StatusOK,
			"authenticate_external_session_search_post_v2.json",
			"authenticate_external_session_search_post_v2_ok.json",
		),
	)
	rv, err := as.client.GetExternalSession(
		context.Background(),
		"f7dcdfba72ef4606bed2c57af421b8e6",
	)
	as.NoError(err)
	as.Equal(rv.Token, SandboxExternalSession("f7dcdfba72ef4606bed2c57af421b8e6"))
	as.Equal(rv.TaskID, SandboxTaskID(1315697012))
	as.Equal(rv.ExecutionID, consts.ExecutionID("0198d50c-4a23-4159-bee0-8e87e6456eff"))
	as.Equal(1, tr.GetTotalCallCount())
}

func (as *AuthenticateTestSuite) TestSearchExternalSession() {
	tr := as.tr
	tr.RegisterResponder(
		http.MethodPost,
		callPath("/authenticate/external/session/search"),
		NewCorpusResponder(
			as.T(),
			http.StatusOK,
			"authenticate_external_session_search_post.json",
			"authenticate_external_session_search_post_ok.json",
		),
	)
	rv, err := as.client.SearchExternalSession(
		context.Background(),
		"0198d50c-4a23-4159-bee0-8e87e6456eff",
	)
	as.NoError(err)
	as.Equal(rv.Token, SandboxExternalSession("f7dcdfba72ef4606bed2c57af421b8e6"))
	as.Equal(rv.TaskID, SandboxTaskID(1315697012))
	as.Equal(rv.ExecutionID, consts.ExecutionID("0198d50c-4a23-4159-bee0-8e87e6456eff"))
	as.Equal(1, tr.GetTotalCallCount())
}

func (as *AuthenticateTestSuite) TestSearchExternalSessionNotFound() {
	tr := as.tr
	tr.RegisterResponder(
		http.MethodPost,
		callPath("/authenticate/external/session/search"),
		NewCorpusResponder(
			as.T(),
			http.StatusNotFound,
			"authenticate_external_session_search_post.json",
			"",
		),
	)
	rv, err := as.client.SearchExternalSession(
		context.Background(),
		"0198d50c-4a23-4159-bee0-8e87e6456eff",
	)
	as.ErrorIs(err, ErrSandboxNotFound)
	as.Empty(rv)
	as.Equal(1, tr.GetTotalCallCount())
}

func TestAuthenticateGroup(t *testing.T) {
	s := &AuthenticateTestSuite{}
	suite.Run(t, s)
}

type TaskTestSuite struct {
	suite.Suite
	client  *Client
	runtime *httptransport.Runtime
	tr      *httpmock.MockTransport
}

func (ts *TaskTestSuite) SetupSuite() {
	ts.client = &Client{
		conf:   nil,
		logger: &nop.Logger{},
		sbx: &sb.SandboxJSONAPI{
			Task: task.New(nil, strfmt.Default),
		},
		token: "",
	}

}

func (ts *TaskTestSuite) SetupTest() {
	tmpdir := testutils.TwistTmpDir(ts.T())
	logger := testutils.TwistMakeLogger(tmpdir, "client.log")
	ts.client.logger = logger
	ts.tr = httpmock.NewMockTransport()
	ts.runtime = httptransport.New(sandboxHost, sb.DefaultBasePath, sb.DefaultSchemes)
	ts.runtime.Transport = ts.tr
	ts.runtime.SetDebug(true)
	ts.runtime.SetLogger(&logWrapper{logger.WithName("transport")})

	ts.client.sbx.Task.SetTransport(ts.runtime)
}

func (ts *TaskTestSuite) TestGetTask() {
	tr := ts.tr
	tr.RegisterResponder(
		http.MethodGet,
		callPath("task/1315697012"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"",
			"task_get_ok.json",
		),
	)
	status, finished, success, err := ts.client.GetSandboxTaskStatus(
		context.Background(),
		SandboxTaskID(1315697012),
	)
	ts.NoError(err)
	ts.Equal(status, models.TaskAuditItemStatusSUCCESS)
	ts.True(finished)
	ts.True(success)
	ts.Equal(1, tr.GetTotalCallCount())
}

func (ts *TaskTestSuite) TestCreateTask() {
	tr := ts.tr
	tr.RegisterResponder(
		http.MethodPost,
		callPath("task"),
		NewCorpusResponder(
			ts.T(),
			http.StatusCreated,
			"task_post.json",
			"task_post_ok.json",
		),
	)
	data := &models.TaskNew{}
	ts.NoError(data.UnmarshalBinary(corpus.MustGet("task_post.json")))
	taskID, err := ts.client.TryCreateTask(
		context.Background(),
		"59da1ac0-4cd2-4d52-a928-6e654b20e335",
		data,
	)
	ts.NoError(err)
	ts.EqualValues(1320446088, taskID)
	ts.Equal(1, tr.GetTotalCallCount())
}

func TestTaskGroup(t *testing.T) {
	s := &TaskTestSuite{}
	suite.Run(t, s)
}

// Resource API

type ResourceTestSuite struct {
	suite.Suite
	client  *Client
	runtime *httptransport.Runtime
	tr      *httpmock.MockTransport
}

func (ts *ResourceTestSuite) SetupSuite() {
	ts.client = &Client{
		conf:   nil,
		logger: &nop.Logger{},
		sbx: &sb.SandboxJSONAPI{
			Resource: resource.New(nil, strfmt.Default),
		},
		token: "",
	}

}

func (ts *ResourceTestSuite) SetupTest() {
	tmpdir := testutils.TwistTmpDir(ts.T())
	logger := testutils.TwistMakeLogger(tmpdir, "client.log")
	ts.client.logger = logger
	ts.tr = httpmock.NewMockTransport()
	ts.runtime = httptransport.New(sandboxHost, sb.DefaultBasePath, sb.DefaultSchemes)
	ts.runtime.Transport = ts.tr
	ts.runtime.SetDebug(true)
	ts.runtime.SetLogger(&logWrapper{logger.WithName("transport")})

	ts.client.sbx.Resource.SetTransport(ts.runtime)
}

func (ts *ResourceTestSuite) TestLookupResource() {
	tr := ts.tr
	tr.RegisterResponder(
		http.MethodGet,
		callPath("resource"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"",
			"resource_lookup_ok.json",
		),
	)
	tr.RegisterResponder(
		http.MethodGet,
		callPath("resource/3127375053"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"",
			"resource_get_ok.json",
		),
	)

	// FIXME: check query string
	resInfo, err := ts.client.LookupResource(
		context.Background(),
		consts.ArcClientResourceType.String(),
		"ARC",
		map[string]string{"platform": "linux", "released": "stable"},
	)
	ts.NoError(err)
	ts.EqualValues(3127375053, resInfo.ID)
	ts.Equal("https://proxy.sandbox.yandex-team.ru/3127375053", resInfo.ProxyLink)
	ts.Equal("be621d6d57e54189df5632d94fc775f1", resInfo.MD5)
	ts.Equal(2, tr.GetTotalCallCount())

}

func (ts *ResourceTestSuite) TestResourceGet() {
	tr := ts.tr
	tr.RegisterResponder(
		http.MethodGet,
		callPath("resource/3127375053"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"",
			"resource_get_ok.json",
		),
	)
	resInfo, err := ts.client.GetResourceInfo(
		context.Background(),
		3127375053,
	)
	ts.NoError(err)
	ts.EqualValues(3127375053, resInfo.ID)
	ts.Equal("https://proxy.sandbox.yandex-team.ru/3127375053", resInfo.ProxyLink)
	ts.Equal("be621d6d57e54189df5632d94fc775f1", resInfo.MD5)
	ts.Equal(1, tr.GetTotalCallCount())

}

func TestResourceGroup(t *testing.T) {
	s := &ResourceTestSuite{}
	suite.Run(t, s)
}

// Batch API

type BatchTestSuite struct {
	suite.Suite
	client  *Client
	runtime *httptransport.Runtime
	tr      *httpmock.MockTransport
}

func (ts *BatchTestSuite) SetupSuite() {
	ts.client = &Client{
		conf:   nil,
		logger: &nop.Logger{},
		sbx: &sb.SandboxJSONAPI{
			Batch: batch.New(nil, strfmt.Default),
			Task:  task.New(nil, strfmt.Default),
		},
		token: "",
	}

}

func (ts *BatchTestSuite) SetupTest() {
	tmpdir := testutils.TwistTmpDir(ts.T())
	logger := testutils.TwistMakeLogger(tmpdir, "client.log")
	ts.client.logger = logger
	ts.tr = httpmock.NewMockTransport()
	ts.runtime = httptransport.New(sandboxHost, sb.DefaultBasePath, sb.DefaultSchemes)
	ts.runtime.Transport = ts.tr
	ts.runtime.SetDebug(true)
	ts.runtime.SetLogger(&logWrapper{logger.WithName("transport")})

	ts.client.sbx.Batch.SetTransport(ts.runtime)
	ts.client.sbx.Task.SetTransport(ts.runtime)
}

func (ts *BatchTestSuite) TestTaskStart() {
	tr := ts.tr
	tr.RegisterResponder(
		http.MethodGet,
		callPath("task/1320446088"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"",
			"task_draft_get_ok.json",
		),
	)
	tr.RegisterResponder(
		http.MethodPut,
		callPath("batch/tasks/start"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"batch_tasks_start.json",
			"batch_tasks_start_ok.json",
		),
	)
	err := ts.client.StartTask(
		context.Background(),
		1320446088,
	)
	ts.NoError(err)
	ts.Equal(2, tr.GetTotalCallCount())
}

func (ts *BatchTestSuite) TestTaskStop() {
	tr := ts.tr
	tr.RegisterResponder(
		http.MethodGet,
		callPath("task/1320446088"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"",
			"task_executing_get_ok.json",
		),
	)
	tr.RegisterResponder(
		http.MethodPut,
		callPath("batch/tasks/stop"),
		NewCorpusResponder(
			ts.T(),
			http.StatusOK,
			"batch_tasks_start.json",
			"batch_tasks_start_ok.json",
		),
	)
	err := ts.client.StopTask(
		context.Background(),
		1320446088,
		"",
	)
	ts.NoError(err)
	ts.Equal(2, tr.GetTotalCallCount())
}

func TestBatchGroup(t *testing.T) {
	s := &BatchTestSuite{}
	suite.Run(t, s)
}
