package sandbox

import (
	"context"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
)

type logWrapper struct {
	l log.Logger
}

func (l *logWrapper) Printf(format string, args ...interface{}) {
	l.l.Infof(format, args...)
}
func (l *logWrapper) Debugf(format string, args ...interface{}) {
	l.l.Debugf(format, args...)
}

type httpMetrics struct {
	calls1xx          metrics.Counter
	calls2xx          metrics.Counter
	calls3xx          metrics.Counter
	calls4xx          metrics.Counter
	calls5xx          metrics.Counter
	fails             metrics.Counter
	canceled          metrics.Counter
	panics            metrics.Counter
	durationHistogram metrics.Timer
}

func newHTTPMetrics(mr metrics.Registry) *httpMetrics {
	rv := &httpMetrics{
		calls1xx: mr.WithTags(map[string]string{"http_code": "1xx"}).Counter("calls"),
		calls2xx: mr.WithTags(map[string]string{"http_code": "2xx"}).Counter("calls"),
		calls3xx: mr.WithTags(map[string]string{"http_code": "3xx"}).Counter("calls"),
		calls4xx: mr.WithTags(map[string]string{"http_code": "4xx"}).Counter("calls"),
		calls5xx: mr.WithTags(map[string]string{"http_code": "5xx"}).Counter("calls"),
		fails:    mr.Counter("fails"),
		canceled: mr.Counter("canceled"),
		panics:   mr.Counter("panics"),
		durationHistogram: mr.DurationHistogram(
			"duration_buckets",
			metrics.NewDurationBuckets(httpmetrics.DefaultDurationBuckets()...),
		),
	}
	solomon.Rated(rv.calls1xx)
	solomon.Rated(rv.calls2xx)
	solomon.Rated(rv.calls3xx)
	solomon.Rated(rv.calls4xx)
	solomon.Rated(rv.calls5xx)
	solomon.Rated(rv.fails)
	solomon.Rated(rv.canceled)
	solomon.Rated(rv.panics)
	solomon.Rated(rv.durationHistogram)

	return rv
}
func (a *httpMetrics) recordCode(code int) {
	switch code / 100 {
	case 1:
		a.calls1xx.Inc()
	case 2:
		a.calls2xx.Inc()
	case 3:
		a.calls3xx.Inc()
	case 4:
		a.calls4xx.Inc()
	default:
		a.calls5xx.Inc()
	}
}

type augmentedTransport struct {
	rt      http.RoundTripper
	conf    *Config
	metrics *httpMetrics
}

func newAugmentedTransport(conf *Config, mr metrics.Registry) augmentedTransport {
	rv := augmentedTransport{
		rt:      http.DefaultTransport,
		conf:    conf,
		metrics: newHTTPMetrics(mr),
	}

	return rv
}

func (a augmentedTransport) RoundTrip(request *http.Request) (*http.Response, error) {
	startTime := time.Now()
	defer func() {
		if r := recover(); r != nil {
			a.metrics.panics.Inc()
			panic(r)
		}
	}()
	defer func() {
		a.metrics.durationHistogram.RecordDuration(time.Since(startTime))
	}()

	rsp, err := a.rt.RoundTrip(request)
	if err != nil {
		a.metrics.recordCode(500)
		if xerrors.Is(err, context.Canceled) {
			a.metrics.canceled.Inc()
		} else {
			a.metrics.fails.Inc()
		}
	} else {
		a.metrics.recordCode(rsp.StatusCode)
	}
	return rsp, err
}
