package staff

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	idmmodel "a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/model"
	staffmodel "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/model"
	"github.com/karlseguin/ccache/v2"
)

const UserGroupCacheTTL = time.Minute * 30

type StaffGroupsCache struct {
	StaffClient     IClient
	GroupIDCache    sync.Map
	GroupNameCache  sync.Map
	UserGroupsCache *ccache.Cache
	Enabled         bool
}

func NewStaffGroupsCache(staffClient IClient) *StaffGroupsCache {
	conf := ccache.Configure()
	conf.MaxSize(15000)
	return &StaffGroupsCache{
		StaffClient:     staffClient,
		GroupIDCache:    sync.Map{},
		GroupNameCache:  sync.Map{},
		UserGroupsCache: ccache.New(conf),
		Enabled:         true,
	}
}

func (s *StaffGroupsCache) Stop() {
	s.UserGroupsCache.Stop()
}

func (s *StaffGroupsCache) UpdateGroups(ctx context.Context, dbObjects *idmmodel.DBObjects) error {
	if !s.Enabled {
		return nil
	}

	groupMap := map[string]bool{}
	var groups []string

	for _, namespace := range dbObjects.Namespaces {
		for _, p := range namespace.Meta.Permissions.GetSubjects() {
			if p.Source == taskletv2.PermissionsSubject_E_SOURCE_ABC && !groupMap[p.Name] {
				groupMap[p.Name] = true
				if _, ok := s.GroupNameCache.Load(p.Name); !ok {
					groups = append(groups, p.Name)
				}
			}
		}
	}

	for _, tasklet := range dbObjects.Tasklets {
		for _, p := range tasklet.Meta.Permissions.GetSubjects() {
			if p.Source == taskletv2.PermissionsSubject_E_SOURCE_ABC && !groupMap[p.Name] {
				groupMap[p.Name] = true
				if _, ok := s.GroupNameCache.Load(p.Name); !ok {
					groups = append(groups, p.Name)
				}
			}
		}
	}

	groupsInfo, err := s.StaffClient.ListGroupsByNames(ctx, groups)
	if err != nil {
		return xerrors.Errorf("Error in Staff API call for groups %s. Error: %w", groups, err)
	}
	for _, g := range groupsInfo {
		s.GroupIDCache.Store(g.ID, g)
		s.GroupNameCache.Store(g.URL, g)
	}
	return nil
}

func (s *StaffGroupsCache) AddGroup(g *staffmodel.GroupInfo) {
	s.GroupIDCache.Store(g.ID, g)
	s.GroupNameCache.Store(g.URL, g)
}

func (s *StaffGroupsCache) ListUserGroups(ctx context.Context, login string) (map[string]bool, error) {
	item := s.UserGroupsCache.Get(login)
	if item != nil && !item.Expired() {
		return item.Value().(map[string]bool), nil
	}
	if !s.Enabled {
		return map[string]bool{}, nil
	}
	groups, err := s.StaffClient.ListUserGroups(ctx, login)
	if err != nil {
		return nil, err
	}
	s.UserGroupsCache.Set(login, groups, UserGroupCacheTTL)
	return groups, nil
}
