package staff

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/lib"
	staffmodel "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/model"

	"golang.org/x/net/context/ctxhttp"
)

const requestChunkSize = 15

type IClient interface {
	ListGroupsByIDs(ctx context.Context, ids []int) ([]*staffmodel.GroupInfo, error)
	ListGroupsByNames(ctx context.Context, names []string) ([]*staffmodel.GroupInfo, error)
	GroupByID(ctx context.Context, id int) (*staffmodel.GroupInfo, error)
	GroupByName(ctx context.Context, name string) (*staffmodel.GroupInfo, error)
	ListUserGroups(ctx context.Context, login string) (map[string]bool, error)
}

type Client struct {
	Logger      log.Logger
	TVMClient   tvm.Client
	TVMClientID tvm.ClientID
	URL         string
	HTTPClient  *http.Client
}

func (c *Client) Do(ctx context.Context, req *http.Request) (*http.Response, error) {
	fields := ctxlog.ContextFields(ctx)

	fields = append(fields, log.String("url", req.URL.String()))
	fields = append(fields, log.String("method", req.Method))

	reqID := consts.NewRequestID().String()
	fields = append(fields, log.String("staff_api_request_id", reqID))
	req.Header.Add("X-System-Request-Id", reqID)

	ticket, err := c.TVMClient.GetServiceTicketForID(ctx, c.TVMClientID)
	if err != nil {
		return nil, xerrors.Errorf("Error on getting TVM service ticket. Error: %w", err)
	}
	req.Header.Add("X-Ya-Service-Ticket", ticket)

	ctxlog.Debug(ctx, c.Logger, "sending Staff request", fields...)
	rsp, err := ctxhttp.Do(ctx, c.HTTPClient, req)

	fields = ctxlog.ContextFields(ctx)

	if err == nil {
		fields = append(
			fields,
			log.Int("status_code", rsp.StatusCode),
		)
	}

	ctxlog.Debug(ctx, c.Logger, "received Staff response", fields...)

	return rsp, err
}

func (c *Client) fetchGroups(ctx context.Context, names []string, ids []string) ([]*staffmodel.GroupInfo, error) {
	if len(names) == 0 && len(ids) == 0 {
		return []*staffmodel.GroupInfo{}, nil
	}

	query := ""

	if len(names) > 0 {
		query = fmt.Sprintf("url in [\"%s\"]", strings.Join(names, "\", \""))
	} else {
		query = fmt.Sprintf("id in [%s]", strings.Join(ids, ", "))
	}

	requestURL := fmt.Sprintf("%s/groups?_fields=id,url&_query=%s", c.URL, url.QueryEscape(query))
	rsp, err := lib.Get(c, ctx, requestURL)

	if err != nil {
		return nil, err
	}

	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return nil, err
	}

	var reply staffmodel.GroupResponse
	if err = json.NewDecoder(rsp.Body).Decode(&reply); err != nil {
		return nil, err
	}
	return reply.Result, nil
}

func (c *Client) ListGroupsByIDs(ctx context.Context, ids []int) ([]*staffmodel.GroupInfo, error) {
	var chunks []string
	var groups []*staffmodel.GroupInfo
	for idx, id := range ids {
		chunks = append(chunks, strconv.Itoa(id))
		if len(chunks) > requestChunkSize || (idx+1 == len(ids) && len(chunks) > 0) {
			chunkGroups, err := c.fetchGroups(ctx, []string{}, chunks)
			if err != nil {
				return nil, err
			}
			groups = append(groups, chunkGroups...)
			chunks = []string{}
		}
	}
	return groups, nil
}

func (c *Client) ListGroupsByNames(ctx context.Context, names []string) ([]*staffmodel.GroupInfo, error) {
	var chunks []string
	var groups []*staffmodel.GroupInfo
	for idx, name := range names {
		chunks = append(chunks, name)
		if len(chunks) > requestChunkSize || (idx+1 == len(names) && len(chunks) > 0) {
			chunkGroups, err := c.fetchGroups(ctx, chunks, []string{})
			if err != nil {
				return nil, err
			}
			groups = append(groups, chunkGroups...)
			chunks = []string{}
		}
	}
	return groups, nil
}

func (c *Client) GroupByID(ctx context.Context, id int) (*staffmodel.GroupInfo, error) {
	groups, err := c.ListGroupsByIDs(ctx, []int{id})
	if err != nil {
		return nil, err
	}

	if len(groups) == 0 {
		return nil, &lib.HTTPError{ErrorCode: "404", Msg: fmt.Sprintf("Group '%v' not found", id)}
	}
	return groups[0], nil
}

func (c *Client) GroupByName(ctx context.Context, name string) (*staffmodel.GroupInfo, error) {
	groups, err := c.ListGroupsByNames(ctx, []string{name})
	if err != nil {
		return nil, err
	}

	if len(groups) == 0 {
		return nil, &lib.HTTPError{ErrorCode: "404", Msg: fmt.Sprintf("Group '%s' not found", name)}
	}
	return groups[0], nil
}

func (c *Client) ListUserGroups(ctx context.Context, login string) (map[string]bool, error) {
	query := fmt.Sprintf("login==\"%v\"", login)
	requestURL := fmt.Sprintf("%s/persons?_fields=groups.group.url&_query=%s", c.URL, url.QueryEscape(query))
	rsp, err := lib.Get(c, ctx, requestURL)

	if err != nil {
		return nil, err
	}

	defer func() { _ = rsp.Body.Close() }()

	if err = lib.CheckStatusCode(rsp); err != nil {
		return nil, err
	}

	var reply staffmodel.UserGroupsResponse
	if err = json.NewDecoder(rsp.Body).Decode(&reply); err != nil {
		return nil, err
	}
	result := map[string]bool{}
	for _, groupItem := range reply.Result {
		for _, groupInfo := range groupItem.Groups {
			result[groupInfo.Group.URL] = true
		}
	}
	return result, nil
}
