package xydb

import (
	"context"
	"fmt"
	"reflect"
	"time"

	ydbZap "github.com/ydb-platform/ydb-go-sdk-zap"
	"github.com/ydb-platform/ydb-go-sdk/v3"
	"github.com/ydb-platform/ydb-go-sdk/v3/sugar"
	"github.com/ydb-platform/ydb-go-sdk/v3/table"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/result"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/result/named"
	"github.com/ydb-platform/ydb-go-sdk/v3/trace"
	"go.uber.org/atomic"
	uberzap "go.uber.org/zap"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	ydbMetrics "a.yandex-team.ru/library/go/yandex/ydb/metrics"
)

type YDBTransactionType *table.TransactionControl

type Client struct {
	DB             ydb.Connection
	Database       string
	Folder         string
	ReadTxControl  YDBTransactionType
	WriteTxControl YDBTransactionType
	logger         log.Logger
	logQueries     atomic.Bool
}

const detailsFull = trace.DriverRepeaterEvents |
	trace.DriverConnEvents |
	trace.DriverBalancerEvents |
	trace.TablePoolEvents |
	trace.RetryEvents |
	trace.DiscoveryEvents |
	trace.SchemeEvents

const detailsTiny = trace.DriverConnEvents |
	trace.TablePoolEvents

const detailsNone = trace.Details(0)

const details = detailsNone
const _ = detailsTiny
const _ = detailsFull

func setupLogger(logger log.Logger) []ydb.Option {
	opts := make([]ydb.Option, 0)
	var zapLog *uberzap.Logger
	if logger == nil {
		zapLog = uberzap.NewNop()
	} else if z, ok := logger.(*zap.Logger); ok {
		zapLog = z.L
	} else {
		logger.Errorf("Using Nop logger for YDB driver. Unexpected logger type: %v", reflect.TypeOf(logger))
		zapLog = uberzap.NewNop()
	}

	opts = append(
		opts,
		ydb.WithTraceDriver(
			ydbZap.Driver(
				zapLog.WithOptions(),
				details,
			),
		),
		ydb.WithTraceTable(
			ydbZap.Table(
				zapLog,
				details,
			),
		),
		ydb.WithTraceScheme(
			ydbZap.Scheme(
				zapLog,
				details,
			),
		),
	)
	return opts
}

func setupMetrics(mr metrics.Registry) []ydb.Option {
	opts := make([]ydb.Option, 0)
	if mr == nil {
		return opts
	}

	opts = append(
		opts,
		ydb.WithTraceDriver(
			ydbMetrics.Driver(
				mr,
				ydbMetrics.WithNamespace("ydb"),
				ydbMetrics.WithSeparator("."),
				ydbMetrics.WithDetails(trace.DriverConnEvents),
			),
		),
		ydb.WithTraceTable(
			ydbMetrics.Table(
				mr,
				ydbMetrics.WithNamespace("ydb"),
				ydbMetrics.WithSeparator("."),
				ydbMetrics.WithDetails(trace.TablePoolEvents),
			),
		),
	)
	return opts
}

func NewClient(ctx context.Context, config *Config, token string, logger log.Logger, mr metrics.Registry) (
	*Client,
	error,
) {
	opts := make([]ydb.Option, 0)
	if token != "" {
		opts = append(opts, ydb.WithAccessTokenCredentials(token))
	}
	opts = append(opts, setupLogger(logger)...)
	opts = append(opts, setupMetrics(mr)...)
	db, err := ydb.Open(
		ctx,
		sugar.DSN(config.Endpoint, config.Database, false),
		opts...,
	)

	if err != nil {
		return nil, xerrors.Errorf("connect error: %w", err)
	}

	return &Client{
		DB:             db,
		Database:       config.Database,
		Folder:         config.Folder,
		ReadTxControl:  table.TxControl(table.BeginTx(table.WithOnlineReadOnly()), table.CommitTx()),
		WriteTxControl: table.TxControl(table.BeginTx(table.WithSerializableReadWrite()), table.CommitTx()),
		logger:         logger,
	}, nil
}

func (client *Client) SetLogQueries(value bool) {
	client.logQueries.Store(value)
}

func (client *Client) Close(ctx context.Context) error {
	closeCtx, cancel := context.WithTimeout(ctx, time.Second*30)
	defer cancel()
	ts := time.Now()
	defer func() {
		ctxlog.Infof(ctx, client.logger, "ydb close duration: %s", time.Since(ts))
	}()
	return client.DB.Close(closeCtx)
}

func (client *Client) GetPath(tableName string) string {
	return fmt.Sprintf("%s/%s/%s", client.Database, client.Folder, tableName)
}

func (client *Client) GetPrefix() string {
	return fmt.Sprintf("%s/%s", client.Database, client.Folder)
}

func (client *Client) QueryPrefix() string {
	return fmt.Sprintf(
		`
		--!syntax_v1
		PRAGMA TablePathPrefix("%s/%s");
		`,
		client.Database,
		client.Folder,
	)

}

func (client *Client) ExecuteWriteQuery(
	ctx context.Context,
	query string,
	opts ...table.ParameterOption,
) error {
	client.logQuery(ctx, query)
	return client.DB.Table().Do(
		ctx, func(c context.Context, s table.Session) error {
			if _, res, err := s.Execute(
				c, client.WriteTxControl, query, table.NewQueryParameters(opts...),
			); err != nil {
				return err
			} else {
				return res.Close()
			}
		},
	)
}

// Write performs write only request. No output is returned.
func (client *Client) Write(
	ctx context.Context,
	query string,
	params ...table.ParameterOption,
) (err error) {
	res, errDo := client.Do(ctx, query, client.WriteTxControl, params...)
	if errDo != nil {
		return errDo
	}
	defer func() {
		errClose := res.Close()
		if err == nil {
			err = errClose
		}
	}()
	return nil
}

func (client *Client) Do(
	ctx context.Context,
	query string,
	txType YDBTransactionType,
	params ...table.ParameterOption,
) (res result.Result, err error) {

	query = client.QueryPrefix() + query
	client.logQuery(ctx, query)

	var opts []table.Option

	err = client.DB.Table().Do(
		ctx,
		func(c context.Context, s table.Session) (err error) {
			_, res, err = s.Execute(
				c, txType, query, table.NewQueryParameters(params...),
				options.WithCollectStatsModeBasic(),
			)
			return
		}, opts...,
	)
	if err != nil {
		return nil, xerrors.Errorf("failed ydb request: %w", err)
	}

	return res, nil
}

func (client *Client) ExecuteReadQuery(
	ctx context.Context,
	query string,
	opts ...table.ParameterOption,
) (res result.Result, err error) {
	client.logQuery(ctx, query)
	err = client.DB.Table().Do(
		ctx, func(c context.Context, s table.Session) error {
			_, res, err = s.Execute(
				c,
				client.ReadTxControl,
				query,
				table.NewQueryParameters(opts...),
			)
			return err
		},
	)
	return
}

// logQuery test helper
func (client *Client) logQuery(ctx context.Context, query string) {
	if !client.logQueries.Load() {
		return
	}
	ctxlog.Infof(ctx, client.logger, "Query: %s", query)
}

type ReadValues []named.Value

// ReadOneRow executes query and unmarshals exactly one row (if result is no rows or more than one row - error will be returned)
func (client *Client) ReadOneRow(
	ctx context.Context,
	query string,
	values ReadValues,
	params ...table.ParameterOption,
) (err error) {
	res, errDo := client.Do(ctx, query, client.ReadTxControl, params...)
	if errDo != nil {
		return errDo
	}
	defer func() {
		errClose := res.Close()
		if err == nil {
			err = errClose
		}
	}()

	if err := EnsureOneRowCursor(ctx, res); err != nil {
		return err
	}

	if err := res.ScanNamed(values...); err != nil {
		return xerrors.Errorf("failed to scan result: %w", err)
	}

	return nil
}

type RowConsumer interface {
	GetColumnSet() []string
	NextReadValues() ReadValues
	ConsumeReadValues() error
	Done()
}

func (client *Client) StreamReadTable(ctx context.Context, tableName string, consumer RowConsumer) (err error) {
	defer consumer.Done()

	tablePath := client.GetPath(tableName)

	var opts []options.ReadTableOption
	for _, column := range consumer.GetColumnSet() {
		opts = append(opts, options.ReadColumn(column))
	}
	session, err := client.DB.Table().CreateSession(
		ctx,
		table.WithTxSettings(table.TxSettings(table.WithOnlineReadOnly())),
	)
	if err != nil {
		return xerrors.Errorf("can not create session: %w", err)
	}
	defer func() {
		errClose := session.Close(context.TODO())
		if err == nil && errClose != nil {
			err = xerrors.Errorf("close session error: %w", errClose)
		}
	}()
	stream, err := session.StreamReadTable(ctx, tablePath, opts...)
	if err != nil {
		return xerrors.Errorf("stream read error: %w", err)
	}
	for stream.NextResultSet(ctx, consumer.GetColumnSet()...) {
		for stream.NextRow() {
			err = stream.ScanNamed(consumer.NextReadValues()...)
			if err != nil {
				return xerrors.Errorf("scan error: %w", err)
			}
			if err = consumer.ConsumeReadValues(); err != nil {
				return xerrors.Errorf("consumer error: %w", err)
			}
		}
	}
	return stream.Err()
}

// clearTable is legacy from locks unit tests
func (client *Client) clearTable(ctx context.Context, name string) error {
	query := fmt.Sprintf("DELETE FROM %s", name)
	query = client.QueryPrefix() + query
	return client.ExecuteWriteQuery(ctx, query)
}

// ResetTable is legacy from locks unit tests
func (client *Client) ResetTable(ctx context.Context, name string, opts ...options.CreateTableOption) error {
	if client.clearTable(ctx, name) == nil {
		return nil
	}
	return client.DB.Table().Do(
		ctx, func(c context.Context, s table.Session) error {
			return s.CreateTable(c, client.GetPath(name), opts...)
		},
	)
}
