package xydb

import (
	"context"
	"strings"

	"github.com/ydb-platform/ydb-go-sdk/v3"
	"github.com/ydb-platform/ydb-go-sdk/v3/scheme"
	"github.com/ydb-platform/ydb-go-sdk/v3/table"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/options"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"golang.org/x/sync/errgroup"
)

func (client *Client) CreateTable(ctx context.Context, tableName string, opts ...options.CreateTableOption) error {
	tablePath := client.GetPath(tableName)
	ctxlog.Debugf(ctx, client.logger, "Creating table. Path: %q", tablePath)
	return client.DB.Table().Do(
		ctx, func(c context.Context, s table.Session) error {
			return s.CreateTable(c, tablePath, opts...)
		},
	)
}

func (client *Client) AlterTable(ctx context.Context, tableName string, opts ...options.AlterTableOption) error {
	tablePath := client.GetPath(tableName)
	ctxlog.Debugf(ctx, client.logger, "Altering table. Path: %q", tablePath)
	return client.DB.Table().Do(
		ctx, func(c context.Context, s table.Session) (err error) {
			return s.AlterTable(c, tablePath, opts...)
		},
	)
}

func (client *Client) SchemeQuery(ctx context.Context, query string, opts ...options.ExecuteSchemeQueryOption) error {
	query = client.QueryPrefix() + query
	client.logQuery(ctx, query)
	err := client.DB.Table().Do(
		ctx, func(ctx context.Context, session table.Session) (err error) {
			return session.ExecuteSchemeQuery(ctx, query, opts...)
		},
	)
	if err != nil {
		return xerrors.Errorf("failed to execute schema query: %w", err)
	}
	return nil
}

func (client *Client) DescribeTable(ctx context.Context, tableName string) (options.Description, error) {
	var res options.Description
	err := client.DB.Table().Do(
		ctx, func(c context.Context, session table.Session) (err error) {
			res, err = session.DescribeTable(c, client.GetPath(tableName))
			return err
		},
	)
	return res, err
}

func (client *Client) DropTable(ctx context.Context, tableName string) error {
	tablePath := client.GetPath(tableName)
	ctxlog.Debugf(ctx, client.logger, "Dropping table. Path: %q", tablePath)
	return client.DB.Table().Do(
		ctx, func(ctx context.Context, session table.Session) (err error) {
			err = session.DropTable(ctx, tablePath)
			if ydb.IsYdbError(err) && strings.Contains(err.Error(), "Path does not exist") {
				return nil
			}
			if err != nil {
				return xerrors.Errorf("failed to drop table %q: %w", tableName, err)
			}
			return nil
		},
	)
}

// ListTables list tables in specified directory with respect to client.Folder
func (client *Client) ListTables(ctx context.Context, path string) (*scheme.Directory, error) {
	if res, err := client.DB.Scheme().ListDirectory(ctx, client.GetPrefix()+"/"+path); err == nil {
		return &res, nil
	} else {
		return nil, xerrors.Errorf("failed to list tables in path %q: %w", path, err)
	}
}

func (client *Client) CreateTablesBySchema(
	ctx context.Context,
	tables map[string]TableSchema,
) error {
	grp, gCtx := errgroup.WithContext(ctx)
	for _, t := range tables {
		t := t
		grp.Go(
			func() error {
				tableOptions, err := t.toYdbOptions()
				if err != nil {
					return xerrors.Errorf("Failed to create table %q: %w", t.Name, err)
				}
				if err := client.CreateTable(gCtx, t.Name, tableOptions...); err != nil {
					return xerrors.Errorf("Failed to create table %q: %w", t.Name, err)
				}
				return nil
			},
		)
	}
	return grp.Wait()
}

func (client *Client) PurgeTablesBySchema(
	ctx context.Context,
	tables map[string]TableSchema,
) error {
	grp, gCtx := errgroup.WithContext(ctx)
	for _, t := range tables {
		t := t
		grp.Go(
			func() error {
				if err := client.DropTable(gCtx, t.Name); err != nil {
					return err
				}
				return nil
			},
		)
	}
	return grp.Wait()
}
