package xydb

import (
	"context"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"github.com/ydb-platform/ydb-go-genproto/protos/Ydb_Table"
	"github.com/ydb-platform/ydb-go-sdk/v3"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"

	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
)

type SchemeSuite struct {
	suite.Suite
	suiteClient *Client
}

func (es *SchemeSuite) TestTableFuncs() {
	ctx := context.Background()
	r := es.Require()
	tableName := "create_table_test"
	err := es.suiteClient.CreateTable(
		ctx, tableName,
		options.WithColumn("id", ydbTypes.Optional(ydbTypes.TypeUint64)),
		options.WithPrimaryKeyColumn("id"),
	)
	r.NoError(err)

	desc, err := es.suiteClient.DescribeTable(ctx, tableName)
	r.NoError(err)
	r.Equal(tableName, desc.Name)
	r.Equal([]string{"id"}, desc.PrimaryKey)

	dir, err := es.suiteClient.ListTables(ctx, "")
	r.NoError(err)
	r.True(dir.IsDirectory())
	r.Len(dir.Children, 1)
	r.Equal(tableName, dir.Children[0].Name)

	r.NoError(es.suiteClient.DropTable(ctx, tableName))
	r.NoError(es.suiteClient.DropTable(ctx, tableName))

	_, err = es.suiteClient.DescribeTable(ctx, tableName)
	r.True(ydb.IsOperationErrorSchemeError(err) && strings.Contains(err.Error(), "Path not found"))
}

func (es *SchemeSuite) TestAlterTable() {
	ctx := context.Background()
	r := es.Require()
	tableName := "create_table_test"
	err := es.suiteClient.CreateTable(
		ctx, tableName,
		options.WithColumn("id", ydbTypes.Optional(ydbTypes.TypeUint64)),
		options.WithColumn("foo", ydbTypes.Optional(ydbTypes.TypeUint64)),
		options.WithPrimaryKeyColumn("id"),
	)
	r.NoError(err)

	r.NoError(
		es.suiteClient.AlterTable(
			ctx, tableName,
			options.WithAddColumn("boo", ydbTypes.Optional(ydbTypes.TypeTimestamp)),
			options.WithDropColumn("foo"),
		),
	)

	desc, err := es.suiteClient.DescribeTable(ctx, tableName)
	r.NoError(err)
	r.Equal(tableName, desc.Name)
	r.Equal([]string{"id"}, desc.PrimaryKey)
	r.Len(desc.Columns, 2)
	for _, c := range desc.Columns {
		switch c.Name {
		case "id":
			r.Equal(ydbTypes.Optional(ydbTypes.TypeUint64), c.Type)
		case "boo":
			r.Equal(ydbTypes.Optional(ydbTypes.TypeTimestamp), c.Type)
		default:
			r.Fail("Unexpected column:  %v", c.Name)

		}
	}

}

func (es *SchemeSuite) TestCreateBySchema() {
	ctx := context.Background()
	r := es.Require()

	tables := map[string]TableSchema{
		"test_table": {
			Name: "test_table",
			Columns: []YdbColumn{
				{
					Name:       "one",
					ValueType:  ydbTypes.Optional(ydbTypes.TypeInt32),
					PrimaryKey: true,
				},
				{
					Name:       "two",
					ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
					PrimaryKey: true,
				},
				{
					Name:       "three",
					ValueType:  ydbTypes.Optional(ydbTypes.TypeBool),
					PrimaryKey: false,
				},
			},
			SecondaryIndexes: []SecondaryIndex{
				{
					Name:    "two_three_view",
					Columns: []string{"two", "three"},
				},
			},
			Queries:     nil,
			TTLSettings: nil,
		},
	}

	err := es.suiteClient.CreateTablesBySchema(ctx, tables)
	r.NoError(err)

	desc, err := es.suiteClient.DescribeTable(ctx, "test_table")
	r.NoError(err)
	r.Equal("test_table", desc.Name)
	r.Equal([]string{"one", "two"}, desc.PrimaryKey)
	r.Len(desc.Indexes, 1)
	view := desc.Indexes[0]
	r.Equal("two_three_view", view.Name)
	r.Equal([]string{"two", "three"}, view.IndexColumns)
	r.Equal(Ydb_Table.TableIndexDescription_STATUS_READY, view.Status)

	r.NoError(es.suiteClient.PurgeTablesBySchema(ctx, tables))
	_, err = es.suiteClient.DescribeTable(ctx, "test_table")
	r.True(ydb.IsOperationErrorSchemeError(err) && strings.Contains(err.Error(), "Path not found"))
}

func TestScheme(t *testing.T) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	tmpdir := testutils.TwistTmpDir(t)
	logger := testutils.TwistMakeLogger(tmpdir, "client.log")

	client := MustGetYdbClient(ctx, logger, t.Name())
	client.SetLogQueries(true)
	defer func() {
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
		defer cancel()
		require.NoError(t, client.Close(ctx))
	}()

	s := &SchemeSuite{
		suiteClient: client,
	}
	suite.Run(t, s)
}
