package xydb

import (
	"context"
	"os"
	"os/user"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/utils"
)

const unitTestsYdbRoot = "unit_tests"

// GetYdbConfigFromEnv constructs config in arcadia unit tests
func GetYdbConfigFromEnv() (*Config, string) {
	mustGetEnv := func(key string) string {
		value, ok := os.LookupEnv(key)
		if !ok {
			panic(xerrors.Errorf("No env key: %q", key))
		}
		return value
	}
	return &Config{
		Endpoint:  mustGetEnv("YDB_ENDPOINT"),
		Database:  mustGetEnv("YDB_DATABASE"),
		Folder:    unitTestsYdbRoot,
		TokenPath: "",
	}, ""
}

var wellKnownYdbConfigs = map[string]Config{
	"alximik": {
		Endpoint:  "ydb-ru-prestable.yandex.net:2135",
		Database:  "/ru-prestable/ydb_home/alximik/alximik",
		Folder:    unitTestsYdbRoot,
		TokenPath: "/home/alximik/terraform/ydb_token_int",
	},
}

func MustGetYdbClient(ctx context.Context, logger log.Logger, subFolder string) *Client {
	if logger == nil {
		logger = &nop.Logger{}
	}
	var conf Config
	var token string
	if testutils.IsYaTest() {
		confPtr, envToken := GetYdbConfigFromEnv()
		conf = *confPtr
		token = envToken
	} else {
		curUser, err := user.Current()
		if err != nil {
			panic(err)
		}
		value, ok := wellKnownYdbConfigs[curUser.Username]
		if !ok {
			panic(curUser.Username)
		}
		conf = value
		token, err = utils.LoadToken(conf.TokenPath)
		if err != nil {
			panic(err)
		}
	}
	if !strings.HasPrefix(conf.Folder, unitTestsYdbRoot) {
		panic(conf.Folder)
	}
	conf.Folder = conf.Folder + "/" + subFolder
	client, err := NewClient(ctx, &conf, token, logger, nil)
	if err != nil {
		panic(err)
	}
	return client
}
