package xydb

import (
	"context"

	"github.com/ydb-platform/ydb-go-sdk/v3/table"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/result"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type TransactionWrapper struct {
	tx     table.TransactionActor
	client *Client
}

func (t *TransactionWrapper) ID() string {
	return t.tx.ID()
}

func (t *TransactionWrapper) Do(
	ctx context.Context,
	query string,
	params ...table.ParameterOption,
) (result.Result, error) {
	query = t.client.QueryPrefix() + query
	t.client.logQuery(ctx, query)
	return t.tx.Execute(ctx, query, table.NewQueryParameters(params...), options.WithCollectStatsModeBasic())
}

func (t *TransactionWrapper) ReadOneRow(
	ctx context.Context,
	query string,
	values ReadValues,
	params ...table.ParameterOption,
) (err error) {
	res, errDo := t.Do(ctx, query, params...)
	if errDo != nil {
		return errDo
	}
	defer func() {
		errClose := res.Close()
		if err == nil {
			err = errClose
		}
	}()

	if err := EnsureOneRowCursor(ctx, res); err != nil {
		return err
	}

	if err := res.ScanNamed(values...); err != nil {
		return xerrors.Errorf("failed to scan result: %w", err)
	}

	return nil

}

// Write performs write only request. No output is returned.
func (t *TransactionWrapper) Write(
	ctx context.Context,
	query string,
	params ...table.ParameterOption,
) (err error) {
	res, errDo := t.Do(ctx, query, params...)
	if errDo != nil {
		return errDo
	}
	defer func() {
		errClose := res.Close()
		if err == nil {
			err = errClose
		}
	}()
	return nil
}

func (client *Client) DoTx(
	ctx context.Context,
	operation func(ctx2 context.Context, actor *TransactionWrapper) error,
	opts ...table.Option,
) (err error) {
	err = client.DB.Table().DoTx(
		ctx,
		func(c context.Context, tx table.TransactionActor) (err error) {
			wr := &TransactionWrapper{
				tx:     tx,
				client: client,
			}
			return operation(c, wr)
		}, opts...,
	)
	if err != nil {
		return xerrors.Errorf("failed ydb request: %w", err)
	}

	return nil
}
