package xydb

import (
	"bytes"
	"fmt"
	"text/template"

	"github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	"github.com/ydb-platform/ydb-go-sdk/v3/table/types"
)

type NamedQuery struct {
	Name     string
	template *template.Template
	query    string
}

func NewQuery(name, textTemplate string) *NamedQuery {
	return &NamedQuery{
		Name:     name,
		template: template.Must(template.New(name).Parse(textTemplate)),
		query:    "",
	}
}

func (q *NamedQuery) MustRender(data interface{}) {
	var buf bytes.Buffer
	err := q.template.Execute(&buf, data)
	if err != nil {
		panic(err)
	}
	q.query = buf.String()
}

func (q *NamedQuery) Query() string {
	if q.query == "" {
		panic("uninitialized " + q.Name)
	}
	return q.query
}

type Query string

type SecondaryIndex struct {
	Name    string
	Columns []string
}

type YdbColumn struct {
	Name       string
	ValueType  types.Type
	PrimaryKey bool
}

type TableSchema struct {
	Name             string
	Columns          []YdbColumn
	SecondaryIndexes []SecondaryIndex
	Queries          map[Query]string
	TTLSettings      *options.TimeToLiveSettings
}

func (t TableSchema) toYdbOptions() ([]options.CreateTableOption, error) {
	res := make([]options.CreateTableOption, 0)
	primaryKeys := make([]string, 0)
	knownColumns := make(map[string]bool)
	for _, c := range t.Columns {
		res = append(res, options.WithColumn(c.Name, c.ValueType))
		if _, ok := knownColumns[c.Name]; ok {
			return nil, fmt.Errorf("duplicate column: %q", c.Name)
		}
		knownColumns[c.Name] = true
		if c.PrimaryKey {
			primaryKeys = append(primaryKeys, c.Name)
		}
	}
	if len(primaryKeys) == 0 {
		return nil, fmt.Errorf("no primary key. Table: %q", t.Name)
	}
	res = append(res, options.WithPrimaryKeyColumn(primaryKeys...))
	for _, i := range t.SecondaryIndexes {
		if _, ok := knownColumns[i.Name]; ok {
			return nil, fmt.Errorf("secondary index Name probably invalid: %q", i.Name)
		}
		for _, c := range i.Columns {
			if _, ok := knownColumns[c]; !ok {
				return nil, fmt.Errorf("secondary index uses missing column: %q", c)
			}
		}
		res = append(
			res,
			options.WithIndex(
				i.Name,
				options.WithIndexType(options.GlobalIndex()),
				options.WithIndexColumns(i.Columns...),
			),
		)
	}
	if t.TTLSettings != nil {
		res = append(
			res,
			options.WithTimeToLiveSettings(*t.TTLSettings),
		)
	}
	return res, nil
}
