package ytdriver

import (
	"context"
	"fmt"

	"google.golang.org/protobuf/types/known/structpb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/state"
	"a.yandex-team.ru/tasklet/experimental/internal/utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
	"a.yandex-team.ru/yt/go/guid"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/yt/ytrpc"
)

type YTResource struct {
	SandboxResourceID int64
	MD5               string
	YTPath            ypath.Rich // FIXME: PATH
}

var ResourceNotRegistered = xerrors.NewSentinel("Resource not registered")

// TODO: check yt/go/ytrecipe/internal/jobfs
type ResourceProvider interface {
	RequireDynamicSandboxResource(ctx context.Context, resID int64) (YTResource, error)
	RequireStaticSandboxResource(resourceType consts.SandboxResourceType) (YTResource, error)
	PrepareSandboxResource(ctx context.Context, resource YTResource) error
	Reconcile(ctx context.Context) error
}

type DynamicResourceProvider struct {
	sbx       *sandbox.Client
	ytc       yt.Client
	cachePath ypath.Path
	cache     map[int64]*YTResource
}

func NewDynamicResourceProvider(sbx *sandbox.Client, ytc yt.Client, path ypath.Path) *DynamicResourceProvider {
	return &DynamicResourceProvider{
		sbx:       sbx,
		ytc:       ytc,
		cachePath: path,
		cache:     make(map[int64]*YTResource, 0),
	}
}

func (d *DynamicResourceProvider) Reconcile(ctx context.Context) error {
	_ = ctx
	// FIXME: do reconcile
	return nil
}

func (d *DynamicResourceProvider) RequireStaticSandboxResource(resType consts.SandboxResourceType) (
	YTResource,
	error,
) {
	resourceInfo, err := state.SandboxState.GetResource(resType)
	if err != nil {
		return YTResource{}, err
	}
	return d.makeResourceFromSandboxInfo(resourceInfo), nil
}

func (d *DynamicResourceProvider) makeResourceFromSandboxInfo(resInfo *sandbox.ResourceInfo) YTResource {
	rv := YTResource{}
	rv.SandboxResourceID = resInfo.ID
	rv.MD5 = resInfo.MD5
	rv.YTPath = *d.cachePath.JoinChild(resInfo.MD5).Rich()
	return rv
}

func (d *DynamicResourceProvider) RequireDynamicSandboxResource(ctx context.Context, resID int64) (
	YTResource,
	error,
) {
	resInfo, err := d.sbx.GetResourceInfo(ctx, resID)
	if err != nil {
		return YTResource{}, xerrors.Errorf("Failed to get resource info. ID: %v, Err: %w", resID, err)
	}
	return d.makeResourceFromSandboxInfo(resInfo), nil
}

func (d *DynamicResourceProvider) PrepareSandboxResource(ctx context.Context, resource YTResource) error {
	ok, err := d.ytc.NodeExists(ctx, resource.YTPath.Path, nil)
	if err != nil {
		return err
	}

	if ok {
		return nil
	}
	sbInfo, err := d.sbx.GetResourceInfo(ctx, resource.SandboxResourceID)
	if err != nil {
		return err
	}

	return yt.ExecTx(
		ctx, d.ytc, func(ctx context.Context, tx yt.Tx) error {
			_, err := tx.CreateNode(ctx, resource.YTPath.Path, yt.NodeFile, nil)
			if err != nil {
				return err
			}

			w, err := tx.WriteFile(ctx, resource.YTPath.Path, &yt.WriteFileOptions{ComputeMD5: true})
			if err != nil {
				return err
			}
			defer func() { _ = w.Close() }()

			if err := d.sbx.CopyResource(ctx, *sbInfo, w); err != nil {
				return err
			}
			return w.Close()
		}, nil,
	)
}

type YTDriver struct {
	Conf           *Config
	executorConfig *apiclient.Config
	l              log.Logger
	ytc            yt.Client
	operations     map[consts.ExecutionID]*YTOperation

	resources ResourceProvider
}

func New(conf *Config, l log.Logger, sbx *sandbox.Client, executorConfig *apiclient.Config) (*YTDriver, error) {
	ytConfig := &yt.Config{
		Proxy:  conf.Cluster,
		Token:  utils.MustToken(utils.LoadToken(conf.TokenPath)),
		Logger: l.WithName("yt").Structured(),
	}
	// NB: rpcCli does not implement file copy ATM
	rpcCli, err := ytrpc.NewClient(ytConfig)
	if err != nil {
		return nil, err
	}

	// NB: For file copy in resource provider
	httpCli, err := ythttp.NewClient(ytConfig)
	if err != nil {
		return nil, err
	}

	return &YTDriver{
		Conf:           conf,
		executorConfig: executorConfig,
		l:              l,
		ytc:            rpcCli,
		operations:     make(map[consts.ExecutionID]*YTOperation),
		resources: NewDynamicResourceProvider(
			sbx,
			httpCli,
			ypath.Path(conf.ResourceCache),
		),
	}, nil
}

func (d *YTDriver) Stop() {
	d.ytc.Stop()
}

func (d *YTDriver) RegisterExecution(
	ctx context.Context,
	b *taskletv2.Build,
	e *taskletv2.Execution,
	secureSession sandbox.SandboxExternalSession,
) error {

	executionID := consts.ExecutionID(e.Meta.Id)
	if _, ok := d.operations[executionID]; ok {
		return nil
	}

	op := newOperation(b, e, d.executorConfig)

	if err := op.Build(ctx, d.resources); err != nil {
		return err
	}
	op.ytOperation.SecureVault["TASKLET_SESSION"] = secureSession.String()

	d.operations[op.GetID()] = op
	return nil
}

func (d *YTDriver) getOperation(id consts.ExecutionID) (*YTOperation, bool) {
	op, ok := d.operations[id]
	return op, ok
}

func (d *YTDriver) prepareResources(ctx context.Context, op *YTOperation) error {

	// FIXME: locks + parallel prepare
	for _, resource := range op.resources {
		ctxlog.Infof(
			ctx,
			d.l,
			"Preparing resource. ExecutionID: %v, ResourceID: %v, Path: %q",
			op.GetID(),
			resource.SandboxResourceID,
			resource.YTPath.Path.String(),
		)
		err := d.resources.PrepareSandboxResource(ctx, resource)
		if err != nil {
			return err
		}
	}
	return nil
}

func (d *YTDriver) Spawn(ctx context.Context, id consts.ExecutionID) (string, error) {

	op, ok := d.getOperation(id)
	if !ok {
		return "", xerrors.Errorf("Execution not registered. ID: %q", id)
	}

	if err := d.prepareResources(ctx, op); err != nil {
		return "", err
	}

	spec, err := op.GetSpec()
	if err != nil {
		return "", err
	}

	// FIXME persist mutation ?
	opID, err := d.ytc.StartOperation(
		ctx,
		yt.OperationVanilla,
		spec,
		&yt.StartOperationOptions{
			MutatingOptions: &yt.MutatingOptions{MutationID: yt.MutationID(guid.New())},
		},
	)
	if err != nil {
		return "", err
	}
	return opID.String(), nil
}

type YTOperationResult struct {
	Finished     bool
	IsError      bool
	State        string
	Error        *structpb.Struct
	ErrorSummary string
}

// var (
// 	failedJobLimitExceededRE = regexp.MustCompile("Failed jobs limit exceeded")
// )

func (d *YTDriver) fillOperationError(
	ctx context.Context,
	opID yt.OperationID,
	status *yt.OperationStatus,
	rv *YTOperationResult,
) error {
	_ = ctx
	innerErr := status.Result.Error

	// if yterrors.ContainsMessageRE(status.Result.Error, failedJobLimitExceededRE) {
	// 	result, err := d.ytc.ListJobs(ctx, opID, &yt.ListJobsOptions{JobState: &yt.JobFailed})
	// 	if err != nil {
	// 		return yterrors.Err("unable to get list of failed jobs", innerErr)
	// 	}
	//
	// 	if len(result.Jobs) == 0 {
	// 		return yterrors.Err("no failed jobs found", innerErr)
	// 	}
	//
	// 	job := result.Jobs[0]
	// 	stderr, err := d.ytc.GetJobStderr(ctx, opID, job.ID, nil)
	// 	if err != nil {
	// 		return yterrors.Err("unable to get job stderr", innerErr)
	// 	}
	//
	// 	jobError := yterrors.Err("job failed",
	// 		innerErr,
	// 		yterrors.Attr("stderr", string(stderr)),
	// 	)
	//  rv.Error = CAST(jobError)
	//
	// }
	bytes, err := innerErr.MarshalJSON()
	if err != nil {
		return err
	}
	parsed := &structpb.Struct{}
	if err := parsed.UnmarshalJSON(bytes); err != nil {
		return err
	}
	rv.Error = parsed
	rv.ErrorSummary = fmt.Sprintf("Operation %q failed with error: %s", opID, innerErr.Message)
	return nil
}

func (d *YTDriver) CheckOperationStatus(ctx context.Context, operationID string) (*YTOperationResult, error) {
	operationGUID, err := guid.ParseString(operationID)
	if err != nil {
		return nil, err
	}
	opID := yt.OperationID(operationGUID)
	rv := &YTOperationResult{
		Finished: false,
		IsError:  false,
		Error:    nil,
	}

	// TODO: rate limit or batch. Need a way to control RPS limit
	status, err := d.ytc.GetOperation(
		ctx,
		opID,
		nil,
	)

	if err != nil {
		return nil, err
	}

	rv.State = string(status.State)
	rv.Finished = status.State.IsFinished()
	if !rv.Finished {
		return rv, nil
	}

	if status.Result.Error != nil && status.Result.Error.Code != 0 {
		rv.IsError = true
		return rv, d.fillOperationError(ctx, opID, status, rv)
	}
	return rv, nil
}
