package ytdriver

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"path"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"a.yandex-team.ru/library/go/test/yatest"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/yt/go/guid"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ytrpc"
	"a.yandex-team.ru/yt/go/yttest"
)

const (
	mockExecutorBinaryPath = "tasklet/experimental/internal/yandex/ytdriver/mocks/executor/executor"
)

var fakeResourceIDs = map[consts.SandboxResourceType]int64{
	consts.TaskletExecutorResourceType: 57,
}

func setupStaticResourceCache(
	t *testing.T,
	ctx context.Context,
	ytc yt.Client,
	root ypath.Path,
) *StaticResourceProvider {
	provider, err := NewStaticResourceProviderBuilder(ytc, root).
		AddStaticResource(ctx, mockExecutorBinaryPath, consts.TaskletExecutorResourceType).
		AddDynamicResource(ctx, mockExecutorBinaryPath, testutils.DefaultUserPayloadResourceID).
		Build()
	require.NoError(t, err)
	return provider
}

func TestTrackOperation(t *testing.T) {
	testutils.EnsureArcadiaTest(t)

	tmpdir := testutils.TwistTmpDir(t)
	logger := testutils.TwistMakeLogger(tmpdir, "driver")
	env := yttest.New(t, yttest.WithLogger(logger.WithName("yt_env").Structured()))

	ctx, cancel := context.WithTimeout(context.Background(), time.Second*30)
	defer cancel()

	testGUID := guid.New().String()
	testRoot := ypath.Path("//tmp").Child(testGUID)
	resourceCache := testRoot.JoinChild("resource_cache")

	_, err := env.YT.CreateNode(ctx, resourceCache, yt.NodeMap, &yt.CreateNodeOptions{Recursive: true})
	require.NoError(t, err)

	// Setup test users

	for _, user := range []string{"robot-tasklets", "alximik", "abash"} {
		_, err := env.YT.CreateObject(
			ctx,
			yt.NodeUser,
			&yt.CreateObjectOptions{Attributes: map[string]interface{}{"name": user}},
		)
		require.NoError(t, err)
	}
	{
		var rv interface{}
		assert.NoError(t, env.YT.GetNode(ctx, ypath.Path("//sys/users"), &rv, nil))

		bytes, err := json.MarshalIndent(rv, "", "  ")
		assert.NoError(t, err)
		logger.Warn(string(bytes))
	}

	driver := &YTDriver{
		Conf: &Config{
			Cluster:       "localhost",
			TokenPath:     "none",
			ResourceCache: resourceCache.String(),
		},
		executorConfig: &apiclient.Config{
			EndpointAddress: "localhost:6666",
		},
		l: logger,
		ytc: func() yt.Client {
			ytc, err := ytrpc.NewClient(
				&yt.Config{
					Proxy:  os.Getenv("YT_PROXY"),
					Logger: logger.WithName("driver.yt").Structured(),
				},
			)
			require.NoError(t, err)
			return ytc
		}(),
		operations: make(map[consts.ExecutionID]*YTOperation, 0),
		resources:  setupStaticResourceCache(t, ctx, env.YT, resourceCache),
	}
	t.Cleanup(driver.Stop)

	sharedPath := yatest.OutputPath(testGUID)

	// Prepare shared folder
	require.NoError(t, os.Mkdir(sharedPath, 0777))
	require.NoError(t, os.WriteFile(path.Join(sharedPath, "foo.txt"), []byte("keep folder on error"), 0666))

	// Build objects
	objectGenerator := testutils.NewObjectGenerator()
	tasklet := objectGenerator.NewTasklet("the_tasklet", objectGenerator.NewNamespace("the_namespace"))
	build := objectGenerator.NewBuild(tasklet)
	execution := objectGenerator.NewExecution(build)

	require.NoError(t, driver.RegisterExecution(ctx, build, execution, "sandbox_secure_session_uid"))

	op, ok := driver.getOperation(consts.ExecutionID(execution.Meta.Id))
	require.True(t, ok)

	// Turn off porto-specific options
	op.ytJob.LayerPaths = make([]ypath.Path, 0)
	// NB: https://a.yandex-team.ru/arc/trunk/arcadia/yt/yt/server/lib/containers/porto_executor.cpp?rev=r9087110#L90-92
	op.ytJob.EnablePorto = "none"
	if op.ytJob.Environment == nil {
		op.ytJob.Environment = make(map[string]string, 0)
	}
	op.ytJob.Environment["TASKLET_TESTS_PATH"] = sharedPath

	operationID, err := driver.Spawn(ctx, consts.ExecutionID(execution.Meta.Id))
	require.NoError(t, err)
	castedOperationID, err := guid.ParseString(operationID)
	require.NoError(t, err)

	// NB: wait 100ms to let operation initialize
	// TODO: wait for job start, not plain timeout
	<-time.After(100 * time.Millisecond)

	dumpOperationStatus := func() *yt.OperationStatus {

		opStatus, err := driver.ytc.GetOperation(
			ctx,
			yt.OperationID(castedOperationID),
			nil,
		)
		require.NoError(t, err)
		{
			// NB: debug dump operation info
			// Does not marshall nil RawValue
			opStatus.FullSpec = yson.RawValue("\"\"")
			{
				ss, err := yson.MarshalFormat(opStatus, yson.FormatPretty)
				require.NoError(t, err)
				fmt.Println(string(ss))
			}
		}
		return opStatus
	}
	{
		st := dumpOperationStatus()
		require.False(t, st.State.IsFinished(), "operation crashed unexpectedly")
	}

	status, err := driver.CheckOperationStatus(ctx, operationID)
	require.NoError(t, err)
	require.False(t, status.Finished)
	require.False(t, status.IsError)
	require.Nil(t, status.Error)

	exitFlag := path.Join(sharedPath, "done")
	touchExitFile := func(msg string) {
		if _, err := os.Stat(exitFlag); !os.IsNotExist(err) {
			return
		}
		err := os.WriteFile(exitFlag, []byte(msg), 0666)
		if err != nil {
			panic(err)
		}
	}
	defer touchExitFile("test should not get here")
	touchExitFile("done")

	timer := time.NewTicker(100 * time.Millisecond)
	defer timer.Stop()
AwaitLoop:
	for {
		select {
		case <-timer.C:
			_ = dumpOperationStatus()
			status, err := driver.CheckOperationStatus(ctx, operationID)
			require.NoError(t, err)
			if !status.Finished {
				logger.Infof("Waiting for operation")
				continue AwaitLoop
			}
			require.True(t, status.Finished)
			require.False(t, status.IsError)
			require.Nil(t, status.Error)
			break AwaitLoop
		case <-ctx.Done():
			require.Fail(t, "timed out waiting for operation")
		}
	}

}
