package ytdriver

import (
	"context"
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	operationSpec "a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
)

// yt --proxy hume upload --executable //home/alximik/tasklets/dummy_tasklet < dummy_tasklet

type ysonMap map[string]interface{}

type YTOperation struct {
	e                  *taskletv2.Execution
	b                  *taskletv2.Build
	executorResourceID int64
	executorConf       *apiclient.Config
	ytOperation        *operationSpec.Spec
	ytJob              *operationSpec.UserScript
	resources          []YTResource
}

const (
	taskletTaskName    = "tasklet"
	ytDefaultBaseLayer = "//porto_layers/base/bionic/porto_layer_search_ubuntu_bionic_app_lastest.tar.gz"
	ytDefaultJdkLayer  = "//porto_layers/delta/jdk/layer_with_jdk_lastest.tar.gz"
	executorJobFile    = "executor"
	taskletJobFile     = "tasklet"
)

func newOperation(
	b *taskletv2.Build,
	e *taskletv2.Execution,
	conf *apiclient.Config,
) *YTOperation {
	job := &operationSpec.UserScript{}
	job.FilePaths = make([]operationSpec.File, 0)
	job.LayerPaths = make([]ypath.Path, 0)
	rv := &YTOperation{
		e:            e,
		b:            b,
		executorConf: conf,
		ytOperation:  operationSpec.Vanilla(),
		ytJob:        job,
		resources:    make([]YTResource, 0, 2),
	}
	rv.ytOperation.Tasks = map[string]*operationSpec.UserScript{}
	rv.ytOperation.Tasks[taskletTaskName] = job
	return rv
}

func (op *YTOperation) GetID() consts.ExecutionID {
	return consts.ExecutionID(op.e.Meta.Id)
}

func (op *YTOperation) GetSpec() (*operationSpec.Spec, error) {

	return op.ytOperation.Clone(), nil
}

func (op *YTOperation) buildCmdLine() string {
	/*
	  --disable-auth              disable auth (for testing)
	  --endpoint-address string   address of tasklet grpc service
	  --endpoint-set string       endpoint set name of tasklet grpc service
	  --execution-id string       tasklet launch id
	  --tasklet-path string       path to binary that implements tasklet
	*/
	var args []string
	args = append(args, "./"+executorJobFile)
	args = append(args, "--execution-id", op.e.Meta.Id)
	args = append(args, "--tasklet-path", "./"+taskletJobFile)

	if op.executorConf.EndpointSetName != "" {
		args = append(args, "--endpoint-set", op.executorConf.EndpointSetName)
	} else {
		args = append(args, "--endpoint-address", op.executorConf.EndpointAddress)
	}

	if !op.executorConf.EnableAuth {
		args = append(args, "--disable-auth")
	}
	return strings.Join(args, " ")
}

func (op *YTOperation) buildJobSpec(ctx context.Context, res ResourceProvider) error {
	j := op.ytJob

	{
		// Setup executor
		executorResource, err := res.RequireStaticSandboxResource(consts.TaskletExecutorResourceType)
		if err != nil {
			return err
		}
		j.FilePaths = append(
			j.FilePaths, operationSpec.File{
				FileName:            executorJobFile,
				Executable:          true,
				BypassArtifactCache: false,
				CypressPath:         executorResource.YTPath.Path,
			},
		)
		op.resources = append(op.resources, executorResource)
	}

	{
		// Add payload
		payloadResourceID := op.b.GetSpec().GetPayload().GetSandboxResourceId()
		if payloadResourceID == 0 {
			return xerrors.New("Empty payload resource ID")
		}
		payloadResource, err := res.RequireDynamicSandboxResource(ctx, payloadResourceID)
		if err != nil {
			return err
		}
		j.FilePaths = append(
			j.FilePaths, operationSpec.File{
				FileName:            taskletJobFile,
				Executable:          true,
				BypassArtifactCache: false,
				CypressPath:         payloadResource.YTPath.Path,
			},
		)
		op.resources = append(op.resources, payloadResource)

	}
	j.Command = op.buildCmdLine()

	{
		computeResources := op.b.GetSpec().GetComputeResources()
		j.CPULimit = float32(computeResources.VcpuLimit) / 1000.0
		j.MemoryLimit = int64(computeResources.MemoryLimit)
		j.MemoryReserveFactor = 1
	}

	j.JobCount = 1

	// NB: always enable porto for YT
	j.EnablePorto = "isolate"

	// Always use base layer
	j.LayerPaths = append(
		op.ytJob.LayerPaths,
		ytDefaultBaseLayer,
	)

	// FIXME: not implemented
	// spec := YtSpec{
	// 	"interruption_signal":                "SIGKILL",
	// 	"max_stderr_size":                    1024 * 1024,
	// 	"restart_completed_jobs":             false,
	// 	"user_job_memory_digest_lower_bound": 1.0,
	// }

	return nil
}

func (op *YTOperation) buildDescription() ysonMap {
	// description — словарь, аналогичный annotations, также отображающийся в веб-интерфейсе на
	// странице операции. В данную секцию стоит добавлять исключительно человекочитаемую информацию
	// небольшого размера, чтобы не перегружать информацией страницу операции.

	return ysonMap{
		"alximik": yson.ValueWithAttrs{
			Attrs: map[string]interface{}{"_type_tag": "url"},
			Value: fmt.Sprintf(
				"https://sandbox.yandex-team.ru/tasklet/%v/%v/run/%v",
				op.b.Meta.Namespace,
				op.b.Meta.Tasklet,
				op.GetID().String(),
			),
		},
	}
}

func (op *YTOperation) buildAnnotations() ysonMap {
	// annotations — словарь, в который пользователь может записать произвольную
	// структурированную информацию, связанную с операцией. Содержимое annotations в
	// YSON text формате попадает в архив операций, так что по нему можно в дальнейшем искать
	// и идентифировать операции в архиве.

	return ysonMap{
		"tasklet_id":   op.b.Meta.TaskletId,
		"build_id":     op.b.Meta.Id,
		"execution_id": op.e.Meta.Id,
	}

}

func (op *YTOperation) setupLaunchRequirements() error {
	launchSpec := op.b.Spec.LaunchSpec
	// NB: executor стоит commandline для запуска пользовательского кода.
	// Всё, что делается здесь -- добавляем зависимости в спецификацию YT
	switch launchSpec.Type {
	case "binary":
		// nothing to do
	case consts.LaunchTypeJava11, consts.LaunchTypeJava17:
		op.ytJob.LayerPaths = append(
			op.ytJob.LayerPaths,
			ytDefaultJdkLayer,
		)
	default:
		return xerrors.Errorf("Launch type %q is not supported", launchSpec.Type)
	}
	return nil
}

func (op *YTOperation) setupEnvironmentRequirements() error {
	env := op.b.Spec.GetEnvironment()
	// NB: docker always enabled for YT
	if env.GetDocker().GetEnabled() {
		panic("no docker for YT!")
	}

	switch {
	case env.GetJava().GetJdk11().GetEnabled():
		fallthrough
	case env.GetJava().GetJdk17().GetEnabled():
		has := false
		for _, e := range op.ytJob.LayerPaths {
			if e == ytDefaultJdkLayer {
				has = true
				break
			}
		}
		if !has {
			op.ytJob.LayerPaths = append(
				op.ytJob.LayerPaths,
				ytDefaultJdkLayer,
			)
		}

	}
	return nil
}

func (op *YTOperation) setupACL() error {
	op.ytOperation.ACL = append(
		op.ytOperation.ACL, yt.ACE{
			Action: "allow",
			// FIXME: delete alximik, remove robot-tasklets magic constant
			Subjects:        []string{"robot-tasklets", "alximik", op.e.Spec.Author},
			Permissions:     []yt.Permission{yt.PermissionRead, yt.PermissionManage},
			InheritanceMode: "object_and_descendants",
		},
	)
	return nil
}

func (op *YTOperation) setupWorkspace() error {
	ws := op.b.Spec.GetWorkspace()
	// Надо понять, как поддерживать ssd. Пока fallthrough на ram
	// $ yt --proxy hahn get //sys/media
	// <> {
	//     "cloud" = #;
	//     "default" = #;
	//     "cache" = #;
	//     "ssd_blobs" = #;
	//     "ssd_journals" = #;
	//     "pigeons" = #;
	//     "ssd_slots_physical" = #;
	// }

	switch cls := ws.StorageClass; cls {
	case taskletv2.EStorageClass_E_STORAGE_CLASS_INVALID:
		panic(cls)
	case taskletv2.EStorageClass_E_STORAGE_CLASS_HDD:

		// diskSpec := YtSpec{
		// 	"medium":    "default",
		// 	"disk_size": ws.StorageSize,
		// }
		// return op.addJobSpecPatch("disk_request", diskSpec)
		op.ytJob.TmpfsSize = ws.StorageSize
		op.ytJob.MemoryLimit += ws.StorageSize
		op.ytJob.TmpfsPath = "."
	case taskletv2.EStorageClass_E_STORAGE_CLASS_SSD:
		// https://a.yandex-team.ru/arc/trunk/arcadia/sdg/sdc/infra/ci/yt_test_runner/yt_test_runner.py?rev=r8451789#L379-386
		fallthrough
	case taskletv2.EStorageClass_E_STORAGE_CLASS_RAM:
		op.ytJob.TmpfsSize = ws.StorageSize
		op.ytJob.MemoryLimit += ws.StorageSize
		op.ytJob.TmpfsPath = "."
	default:
		panic(cls)
	}
	return nil
}

func (op *YTOperation) buildJob() error {

	return nil
}

func (op *YTOperation) Build(ctx context.Context, res ResourceProvider) error {
	o := op.ytOperation
	o.Title = fmt.Sprintf("Tasklet: %s, Execution %s", op.b.Meta.Tasklet, op.e.Meta.Id)
	// alias – алиас данной операции, делает операцию доступной в поиске и с использованием команды
	// get-operation по указанной строке.
	// NB: unsupported
	// o.Alias = fmt.Sprintf("tasklet-%s", op.e.Meta.Id)

	o.Description = op.buildDescription()
	o.Annotations = op.buildAnnotations()

	// TODO: add secrets
	o.SecureVault = make(map[string]string, 0)

	// TODO: add stderr and core tables

	o.MaxFailedJobCount = 1

	if err := op.buildJobSpec(ctx, res); err != nil {
		return err
	}

	if err := op.setupLaunchRequirements(); err != nil {
		return err
	}

	if err := op.setupEnvironmentRequirements(); err != nil {
		return err
	}

	if err := op.setupWorkspace(); err != nil {
		return err
	}

	if err := op.setupACL(); err != nil {
		return err
	}
	// TODO:
	// secure_vault
	// pool
	// acl
	// artifacts
	// stderr+cores tables
	// config

	return nil
}
