package ytdriver

import (
	"context"
	"fmt"
	"io"
	"os"
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/test/yatest"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/ytdriver/corpus"
	operationSpec "a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
)

func remarshallOperation(t *testing.T, spec *operationSpec.Spec) *operationSpec.Spec {
	rv := &operationSpec.Spec{}
	bytebuf, err := yson.MarshalFormat(spec, yson.FormatPretty)
	require.NoError(t, err)
	if false { // debug
		fmt.Println(string(bytebuf))
	}
	require.NoError(t, yson.Unmarshal(bytebuf, rv))
	return rv
}

type DriverFixture struct {
	ctx       context.Context
	driver    *YTDriver
	build     *taskletv2.Build
	execution *taskletv2.Execution
	resources *StaticResourceProvider
}

func NewFixture(t *testing.T) *DriverFixture {
	tmpdir := testutils.TwistTmpDir(t)
	logger := testutils.TwistMakeLogger(tmpdir, "operation")
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*1)
	t.Cleanup(cancel)

	testRoot := ypath.Path("//tmp").Child("random_uuid_string")
	resourceCache := testRoot.JoinChild("resource_cache")

	dynamicResources := make([]YTResource, 0)
	staticResources := make(map[consts.SandboxResourceType]YTResource)

	{
		ytpath := resourceCache.Child("executor").Rich()
		ytpath.FileName = "executor"
		staticResources[consts.TaskletExecutorResourceType] = YTResource{
			SandboxResourceID: fakeResourceIDs[consts.TaskletExecutorResourceType],
			YTPath:            *ytpath,
		}
	}
	{
		ytpath := resourceCache.Child("tasklet").Rich()
		ytpath.FileName = "tasklet"

		dynamicResources = append(
			dynamicResources,
			YTResource{
				SandboxResourceID: testutils.DefaultUserPayloadResourceID,
				YTPath:            *ytpath,
			},
		)
	}
	resources := &StaticResourceProvider{
		dynamicResources: dynamicResources,
		staticResources:  staticResources,
	}

	driver := &YTDriver{
		Conf: &Config{
			Cluster:       "localhost",
			TokenPath:     "none",
			ResourceCache: resourceCache.String(),
		},
		executorConfig: &apiclient.Config{
			EndpointSetName: "some-endpoint-address.grpc",
			EnableAuth:      true,
		},
		l:          logger,
		ytc:        nil,
		operations: make(map[consts.ExecutionID]*YTOperation, 0),
		resources:  resources,
	}
	// t.Cleanup(driver.Stop)

	objectGenerator := testutils.NewObjectGenerator()
	tasklet := objectGenerator.NewTasklet("the_tasklet", objectGenerator.NewNamespace("the_namespace"))
	build := objectGenerator.NewBuild(tasklet)
	execution := objectGenerator.NewExecution(build)

	return &DriverFixture{
		ctx:       ctx,
		driver:    driver,
		build:     build,
		execution: execution,
		resources: resources,
	}
}

func TestBuildOperation(t *testing.T) {
	type test struct {
		setup func(*testing.T, *DriverFixture)
		patch func(*testing.T, *DriverFixture, *operationSpec.Spec)
	}

	tests := map[string]test{
		"default": {
			setup: func(*testing.T, *DriverFixture) {},
			patch: func(*testing.T, *DriverFixture, *operationSpec.Spec) {},
		},
		"java": {
			setup: func(tt *testing.T, f *DriverFixture) {
				f.build.Spec.LaunchSpec.Type = consts.LaunchTypeJava11
				f.build.Spec.LaunchSpec.Jdk = &taskletv2.LaunchSpec_JDKOptions{MainClass: "ru.yandex.test.JavaClass"}
			},
			patch: func(*testing.T, *DriverFixture, *operationSpec.Spec) {},
		},
		"java_env": {
			setup: func(tt *testing.T, f *DriverFixture) {
				java := f.build.Spec.Environment.Java
				if java == nil {
					java = &taskletv2.JavaEnvironment{}
				}
				if java.Jdk17 == nil {
					java.Jdk17 = &taskletv2.JdkReq{}
				}
				java.Jdk17.Enabled = true
				f.build.Spec.Environment.Java = java
			},
			patch: func(*testing.T, *DriverFixture, *operationSpec.Spec) {},
		},
	}

	// NB: check corpus equals tests
	corpusFiles := make(map[string]int, 0)
	for _, file := range corpus.List() {
		corpusFiles[file] = 1
	}
	for test := range tests {
		corpusFiles[test] -= 1
	}

	for test, value := range corpusFiles {
		require.Equal(t, value, 0, "inconsistent test: id: %s, value: %v", test, value)
	}

	for name, value := range tests {
		value := value
		name := name
		t.Run(
			name, func(tt *testing.T) {
				fixture := NewFixture(tt)
				value.setup(tt, fixture)

				require.NoError(
					tt,
					fixture.driver.RegisterExecution(
						fixture.ctx,
						fixture.build,
						fixture.execution,
						"sandbox_secure_session_uid",
					),
				)
				op, ok := fixture.driver.getOperation(consts.ExecutionID(fixture.execution.Meta.Id))
				require.True(tt, ok)
				spec, err := op.GetSpec()
				require.NoError(tt, err)
				// NB: remarshall spec to drop inconsistencies nil vs empty slice
				actual := remarshallOperation(tt, spec)

				value.patch(tt, fixture, actual)

				expected, err := corpus.Get(name)
				require.NoError(tt, err)
				require.Equal(tt, expected, actual)
			},
		)
	}

}

// StaticResourceProvider static data for tests
type StaticResourceProvider struct {
	dynamicResources []YTResource
	staticResources  map[consts.SandboxResourceType]YTResource
}

func (s *StaticResourceProvider) RequireDynamicSandboxResource(ctx context.Context, resID int64) (YTResource, error) {
	_ = ctx
	for _, i := range s.dynamicResources {
		if i.SandboxResourceID != resID {
			continue
		}
		return i, nil
	}

	return YTResource{}, ResourceNotRegistered
}

func (s *StaticResourceProvider) RequireStaticSandboxResource(resourceType consts.SandboxResourceType) (
	YTResource,
	error,
) {
	res, ok := s.staticResources[resourceType]
	if !ok {
		return YTResource{}, xerrors.Errorf("static resource not found. Type: %s", resourceType)
	}
	return res, nil
}

func (s *StaticResourceProvider) Reconcile(ctx context.Context) error {
	_ = ctx
	// noop
	return nil
}

func (s *StaticResourceProvider) PrepareSandboxResource(ctx context.Context, resource YTResource) error {
	// NB: all resources are pre-cached
	_ = ctx
	_ = resource
	return nil
}

type StaticResourceProviderBuilder struct {
	ytc      yt.Client
	root     ypath.Path
	provider *StaticResourceProvider
	err      error
}

func NewStaticResourceProviderBuilder(ytc yt.Client, root ypath.Path) *StaticResourceProviderBuilder {
	return &StaticResourceProviderBuilder{
		ytc:  ytc,
		root: root,
		provider: &StaticResourceProvider{
			dynamicResources: make([]YTResource, 0),
			staticResources:  make(map[consts.SandboxResourceType]YTResource),
		},
	}
}

func (s *StaticResourceProviderBuilder) uploadFile(ctx context.Context, dest ypath.Path, inputFile string) error {

	if _, err := s.ytc.CreateNode(ctx, dest, yt.NodeFile, nil); err != nil {
		return err
	}
	w, err := s.ytc.WriteFile(ctx, dest, nil)
	if err != nil {
		return err
	}
	defer func() { _ = w.Close() }()

	if r, err := os.Open(inputFile); err != nil {
		return err
	} else {
		defer func() { _ = r.Close() }()
		if _, err := io.Copy(w, r); err != nil {
			return err
		}
	}

	return nil
}

func (s *StaticResourceProviderBuilder) AddStaticResource(
	ctx context.Context,
	arcPath string,
	resType consts.SandboxResourceType,
) (rv *StaticResourceProviderBuilder) {
	rv = s
	if s.err != nil {
		return
	}
	binaryPath, err := yatest.BinaryPath(arcPath)
	if err != nil {
		s.err = err
		return
	}
	destination := s.root.Child(resType.String())
	if err := s.uploadFile(ctx, destination, binaryPath); err != nil {
		s.err = err
		return
	}

	s.provider.staticResources[resType] = YTResource{
		SandboxResourceID: fakeResourceIDs[resType],
		YTPath:            *destination.Rich(),
	}
	return s
}

func (s *StaticResourceProviderBuilder) AddDynamicResource(
	ctx context.Context,
	arcPath string,
	resID int64,
) (rv *StaticResourceProviderBuilder) {
	rv = s
	if s.err != nil {
		return
	}
	binaryPath, err := yatest.BinaryPath(arcPath)
	if err != nil {
		s.err = err
		return
	}
	destination := s.root.Child(fmt.Sprintf("resource-%v", resID))
	if err := s.uploadFile(ctx, destination, binaryPath); err != nil {
		s.err = err
		return
	}

	s.provider.dynamicResources = append(
		s.provider.dynamicResources, YTResource{
			SandboxResourceID: resID,
			YTPath:            *destination.Rich(),
		},
	)
	return
}

func (s *StaticResourceProviderBuilder) Build() (*StaticResourceProvider, error) {
	if s.err != nil {
		return nil, s.err
	}
	return s.provider, nil
}
