package ydbmigrate

import (
	"context"
	"time"

	"github.com/ydb-platform/ydb-go-genproto/protos/Ydb"
	"github.com/ydb-platform/ydb-go-sdk/v3"
	ydbTable "github.com/ydb-platform/ydb-go-sdk/v3/table"
	ydbNamed "github.com/ydb-platform/ydb-go-sdk/v3/table/result/named"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"

	"a.yandex-team.ru/library/go/core/xerrors"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

type MigrationClient struct {
	logger log.Logger
	db     *xydb.Client
}

func NewMigrationClient(db *xydb.Client, logger log.Logger) *MigrationClient {
	return &MigrationClient{
		logger: logger,
		db:     db,
	}
}

var GetSchemaQuery = xydb.NewQuery(
	"get_schema", `
	SELECT
		{{ .Version }}, {{ .Description }}, {{ .MigrationTime }}
    FROM {{ .MigrationsTable }}
	ORDER BY {{ .Version }} DESC
    LIMIT 1;
`,
)

func (mc *MigrationClient) GetSchemaVersion(ctx context.Context) (
	DBVersionInfo,
	error,
) {
	// Defaults for empty database
	version := InitialDBVersionInfo

	if err := mc.db.ReadOneRow(
		ctx,
		GetSchemaQuery.Query(),
		xydb.ReadValues{
			ydbNamed.OptionalWithDefault(MetaQueryConsts.Version, (*int32)(&version.Version)),
			ydbNamed.OptionalWithDefault(MetaQueryConsts.Description, &version.Description),
			ydbNamed.OptionalWithDefault(MetaQueryConsts.MigrationTime, &version.MigrationTime),
		},
	); err != nil {
		return InitialDBVersionInfo, err
	}
	return version, nil
}

var GetLeaseQuery = xydb.NewQuery(
	"get_lease_query", `
	DECLARE ${{ .Version }} AS Int32; -- Target version
	DECLARE ${{ .Mutation }} AS String;
	DECLARE ${{ .Lease }} AS String;
	DECLARE ${{ .LeaseTTL }} AS Timestamp;


	SELECT
		IF(
			{{ .Lease }} IS NULL OR {{ .LeaseTTL }} < CurrentUtcTimestamp(),
			${{ .Lease }},
			{{ .Lease }}
		) AS {{ .Lease }},
		{{ .Applied }}
	FROM {{ .MigrationsLeasesTable }}
	WHERE {{ .Version }} == ${{ .Version }} AND ${{ .Mutation }} == {{ .Mutation }}
	;

	DISCARD SELECT
		{{ .Version }},
		Ensure(
			0,
			${{ .Version }} == {{ .Version }} + 1,
			"Bad next version: Current: '" || CAST(UNWRAP({{ .Version }}) AS String) || ", Expected: " || CAST(${{ .Version }} AS String) || "'"
		)
	FROM {{ .MigrationsTable }}
	ORDER BY {{ .Version }} DESC
	LIMIT 1
	;

	DISCARD SELECT
		Ensure(
			0,
			{{ .Applied }} == true,
			"Mutation not applied: ID: '" || UNWRAP({{ .Mutation }}) || "'"
		)
	FROM {{ .MigrationsLeasesTable }}
	WHERE {{ .Version }} == ${{ .Version }} AND ${{ .Mutation }} > {{ .Mutation }} ;

	UPDATE {{ .MigrationsLeasesTable }}
	SET
		{{ .Lease }} = ${{ .Lease }},
		{{ .LeaseTTL }} = ${{ .LeaseTTL }}
	WHERE
		{{ .Version }} == ${{ .Version }}
		AND ${{ .Mutation }} == {{ .Mutation }}
		AND (
			{{ .Lease }} IS NULL
			OR {{ .LeaseTTL }} < CurrentUtcTimestamp()
		)
	;
`,
)

var PrepareMutationQuery = xydb.NewQuery(
	"prepare_mutation_query", `
	DECLARE ${{ .Version }} AS Int32; -- Base version
	DECLARE ${{ .Mutation }} AS String;

	INSERT INTO {{ .MigrationsLeasesTable }}
		({{ .Version }}, {{ .Mutation }}, {{ .MigrationTime }}, {{ .Applied }} )
	VALUES
		(${{ .Version }}, ${{ .Mutation }}, CurrentUtcTimestamp(), false);
`,
)

func (mc MigrationClient) GetMutationLease(
	ctx context.Context,
	lease MigrationLease,
	migration *Migration,
	idx int,
) (
	MigrationLease,
	bool,
	error,
) {
	prepareQuery := mc.db.QueryPrefix() + PrepareMutationQuery.Query()
	err := mc.db.ExecuteWriteQuery(
		ctx,
		prepareQuery,
		ydbTable.ValueParam("$"+MetaQueryConsts.Version, ydbTypes.Int32Value(migration.GetTargetVersion().ToInt32())),
		ydbTable.ValueParam(
			"$"+MetaQueryConsts.Mutation,
			ydbTypes.StringValueFromString(migration.Mutations[idx].MutationName),
		),
	)
	// NB: insert fails on existing mutation row
	if err != nil && !ydb.IsOperationError(err, Ydb.StatusIds_PRECONDITION_FAILED) {
		return NilMigrationLease, false, err
	}

	res, err := mc.db.Do(
		ctx,
		GetLeaseQuery.Query(),
		mc.db.WriteTxControl,
		ydbTable.ValueParam("$"+MetaQueryConsts.Version, ydbTypes.Int32Value(migration.GetTargetVersion().ToInt32())),
		ydbTable.ValueParam(
			"$"+MetaQueryConsts.Mutation,
			ydbTypes.StringValueFromString(migration.Mutations[idx].MutationName),
		),
		ydbTable.ValueParam("$"+MetaQueryConsts.Lease, ydbTypes.StringValueFromString(lease.String())),
		ydbTable.ValueParam(
			"$"+MetaQueryConsts.LeaseTTL,
			ydbTypes.TimestampValueFromTime(time.Now().UTC().Add(time.Hour)),
		),
	)
	if err != nil {
		return NilMigrationLease, false, err
	}
	defer func() {
		_ = res.Close()
	}()

	if err := xydb.EnsureOneRowCursor(ctx, res); err != nil {
		return NilMigrationLease, false, err
	}

	outLease := NilMigrationLease
	applied := false
	scanErr := res.ScanNamed(
		ydbNamed.OptionalWithDefault(MetaQueryConsts.Lease, &outLease),
		ydbNamed.OptionalWithDefault(MetaQueryConsts.Applied, &applied),
	)
	if scanErr != nil {
		return NilMigrationLease, false, err
	}
	return outLease, applied, nil
}

var CompleteMutationQuery = xydb.NewQuery(
	"complete_mutation_query", `
	DECLARE ${{ .Version }} AS Int32; -- Target version
	DECLARE ${{ .Mutation }} AS String;
	DECLARE ${{ .Lease }} AS String;

	DISCARD SELECT
		Ensure(
			0,
			{{ .Applied }} == false,
			"Mutation already applied"
		),
		Ensure(
			0,
			{{ .Lease }} == ${{ .Lease }},
			"Bad lease. Current: " || UNWRAP({{ .Lease }}) || ", Expected: " || ${{ .Lease }}
		)
	FROM {{ .MigrationsLeasesTable }}
	WHERE {{ .Version }} == ${{ .Version }} AND ${{ .Mutation }} == {{ .Mutation }} ;

	UPDATE {{ .MigrationsLeasesTable }}
	SET
		{{ .Lease }} = NULL,
		{{ .LeaseTTL }} = NULL,
		{{ .Applied }} = true
	WHERE
		{{ .Version }} == ${{ .Version }}
		AND ${{ .Mutation }} == {{ .Mutation }}
		AND {{ .Lease }}  == ${{ .Lease }}
	;
`,
)

func (mc *MigrationClient) CompleteMutation(
	ctx context.Context,
	lease MigrationLease,
	migration *Migration,
	idx int,
) error {
	query := mc.db.QueryPrefix() + CompleteMutationQuery.Query()
	err := mc.db.ExecuteWriteQuery(
		ctx,
		query,
		ydbTable.ValueParam("$"+MetaQueryConsts.Version, ydbTypes.Int32Value(migration.GetTargetVersion().ToInt32())),
		ydbTable.ValueParam(
			"$"+MetaQueryConsts.Mutation,
			ydbTypes.StringValueFromString(migration.Mutations[idx].MutationName),
		),
		ydbTable.ValueParam("$"+MetaQueryConsts.Lease, ydbTypes.StringValueFromString(lease.String())),
	)
	return err
}

var CompleteMigrationQuery = xydb.NewQuery(
	"complete_migration", `
	DECLARE ${{ .Version }} AS Int32; -- target version
	DECLARE ${{ .Description }} AS String;

	DISCARD SELECT
		{{ .Version }},
		Ensure(
			0,
			${{ .Version }} == {{ .Version }} + 1,
			"Bad next version: Current: '" || CAST(UNWRAP({{ .Version }}) AS String) || ", Expected: " || CAST(${{ .Version }} AS String) || "'"
		)
	FROM {{ .MigrationsTable }}
	ORDER BY {{ .Version }} DESC
	LIMIT 1
	;

	DISCARD SELECT
		Ensure(
			0,
			{{ .Applied }} == true,
			"Mutation not applied: ID: '" || UNWRAP({{ .Mutation }}) || "'"
		)
	FROM {{ .MigrationsLeasesTable }}
	WHERE {{ .Version }} == ${{ .Version }};

	INSERT INTO {{ .MigrationsTable }}
		({{ .Version }}, {{ .Description }}, {{ .MigrationTime }})
	VALUES
		(${{ .Version }}, ${{ .Description }}, CurrentUtcTimestamp());
`,
)

func (mc *MigrationClient) CompleteMigration(ctx context.Context, migration *Migration) error {
	query := mc.db.QueryPrefix() + CompleteMigrationQuery.Query()
	err := mc.db.ExecuteWriteQuery(
		ctx,
		query,
		ydbTable.ValueParam("$"+MetaQueryConsts.Version, ydbTypes.Int32Value(migration.GetTargetVersion().ToInt32())),
		ydbTable.ValueParam("$"+MetaQueryConsts.Description, ydbTypes.StringValueFromString(migration.Description)),
	)
	return err
}

func (mc *MigrationClient) MigrationCheckLoop(ctx context.Context, schemaVersion SchemaVersion) error {
	versionInfo, err := mc.GetSchemaVersion(ctx)
	if err != nil {
		return xerrors.Errorf("failed to get db version: %w", err)
	}

	for versionInfo.Version != schemaVersion {
		mc.logger.Warnf("DB version mismatch. Current: %d, Expected: %d", versionInfo.Version, schemaVersion)
		select {
		case <-time.After(time.Second * 10):
			versionInfo, err = mc.GetSchemaVersion(ctx)
			if err != nil {
				return xerrors.Errorf("failed to get db version: %w", err)
			}
		case <-ctx.Done():
			return nil
		}
	}
	return nil
}

func init() {
	InitSchemaQuery.MustRender(MetaQueryConsts)
	GetSchemaQuery.MustRender(MetaQueryConsts)
	GetLeaseQuery.MustRender(MetaQueryConsts)
	PrepareMutationQuery.MustRender(MetaQueryConsts)
	CompleteMigrationQuery.MustRender(MetaQueryConsts)
	CompleteMutationQuery.MustRender(MetaQueryConsts)
}
