package ydbmigrate

import (
	"context"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"
	"github.com/ydb-platform/ydb-go-genproto/protos/Ydb"
	"github.com/ydb-platform/ydb-go-sdk/v3"

	"a.yandex-team.ru/library/go/core/log"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

type MetaSuite struct {
	suite.Suite
	suiteClient *xydb.Client
	ydbRoot     string
	tmpdir      string
	testLogger  log.Logger
}

func (es *MetaSuite) SetupSuite() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.suiteClient = xydb.MustGetYdbClient(ctx, nil, es.T().Name())
	es.ydbRoot = es.suiteClient.Folder
	es.tmpdir = testutils.TwistTmpDir(es.T())
}

func (es *MetaSuite) SetupTest() {
	logger := testutils.TwistMakeLogger(es.tmpdir, strings.ReplaceAll(es.T().Name(), "/", "_")+".log")
	es.suiteClient.Folder = es.ydbRoot + "/" + es.T().Name()
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*10)
	defer cancel()
	es.Require().NoError(PurgeMetaTables(ctx, es.suiteClient))
	es.Require().NoError(CreateMetaTables(ctx, es.suiteClient))
	es.testLogger = logger
}

func (es *MetaSuite) TestGetInitialsVersion() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()
	mc := NewMigrationClient(es.suiteClient, es.testLogger)
	v, err := mc.GetSchemaVersion(ctx)
	es.Require().NoError(err)
	es.Require().Equal(v.Version, NilSchemaVersion)
}

func (es *MetaSuite) TestCompleteMigration() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()

	mc := NewMigrationClient(es.suiteClient, es.testLogger)

	migration0to1 := &Migration{
		BaseVersion: NilSchemaVersion,
		Description: "v0->1",
	}
	es.Require().NoError(mc.CompleteMigration(ctx, migration0to1))

	migration1to2 := &Migration{
		BaseVersion: SchemaVersion(1),
		Description: "v1->2",
	}
	es.Require().NoError(mc.CompleteMigration(ctx, migration1to2))

	v, err := mc.GetSchemaVersion(ctx)
	es.Require().NoError(err)
	es.Require().Equal(v.Version, SchemaVersion(2))

	// test bad base version
	migration100to101 := &Migration{
		BaseVersion: SchemaVersion(100),
		Description: "v100->101",
	}
	es.Require().Error(mc.CompleteMigration(ctx, migration100to101))
}

func (es *MetaSuite) TestMigrate() {

	mc := NewMigrationClient(es.suiteClient, es.testLogger)

	mig := &Migration{
		BaseVersion: NilSchemaVersion,
		Description: "v0->1",
		Mutations: []Mutation{
			{
				"0000_foo",
				func(ctx context.Context, client *xydb.Client) error { return nil },
			},
			{
				"0001_bar",
				func(ctx context.Context, client *xydb.Client) error { return nil },
			},
		},
	}
	require := es.Require()

	ctx := context.Background()
	myLease := NewMigrationLease()
	lease, applied, err := mc.GetMutationLease(ctx, myLease, mig, 0)
	require.NoError(err)
	require.Equal(myLease, lease)
	require.False(applied)
	{
		lease2, applied2, err2 := mc.GetMutationLease(ctx, myLease, mig, 0)
		require.NoError(err2)
		require.Equal(myLease, lease2)
		require.False(applied2)
	}

	{
		// Concurrent mutation
		concurrentLease := NewMigrationLease()
		lease2, applied2, err2 := mc.GetMutationLease(ctx, concurrentLease, mig, 0)
		require.NoError(err2)
		require.Equal(myLease, lease2)
		require.False(applied2)
	}

	{
		// try mutation #1 on incomplete mutation #0
		lease2, applied2, err2 := mc.GetMutationLease(ctx, myLease, mig, 1)
		require.True(ydb.IsOperationError(err2, Ydb.StatusIds_GENERIC_ERROR))
		require.ErrorContains(err2, "Condition violated")
		require.ErrorContains(err2, "0000_foo")
		require.Equal(NilMigrationLease, lease2)
		require.False(applied2)
	}

	{
		// concurrent complete
		concurrentLease := NewMigrationLease()
		err := mc.CompleteMutation(ctx, concurrentLease, mig, 0)
		require.ErrorContains(err, "Bad lease")
	}

	// complete mutation #0
	require.NoError(mc.CompleteMutation(ctx, lease, mig, 0))

	// start mutation #1
	lease2, applied2, err2 := mc.GetMutationLease(ctx, myLease, mig, 1)
	require.NoError(err2)
	require.Equal(myLease, lease2)
	require.False(applied2)

	{
		// complete on unapplied mutation
		err := mc.CompleteMigration(ctx, mig)
		require.ErrorContains(err, "Mutation not applied")
		require.ErrorContains(err, "0001_bar")
	}
	// complete sequence
	require.NoError(mc.CompleteMutation(ctx, lease, mig, 1))
	require.NoError(mc.CompleteMigration(ctx, mig))

}

func (es *MetaSuite) TearDownSuite() {
	es.NoError(es.suiteClient.Close(context.TODO()))
}

func TestMeta(t *testing.T) {
	suite.Run(t, &MetaSuite{})
}
